# Copyright 2023 PKU-Alignment Team. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================

from __future__ import annotations

from datasets import load_dataset
from safe_rlhf.datasets.base import RawDataset, RawSample
from typing import ClassVar

import pandas as pd
import random

__all__ = ['NectarRMTrainDataset', 'NectarRMTestDataset', 'NectarSFTTrainDataset', 'NectarRLHFTrainDataset','NectarRLHFTestDataset']




class NectarRMDataset(RawDataset):
    SPLIT: ClassVar[str]
    PATH: ClassVar[str]

    def __init__(self, path: str | None = None) -> None:
         self.data= pd.read_parquet(self.PATH)

    def __getitem__(self, index: int) -> RawSample:
        data = self.data.iloc[index]

        input_text = data['prompt']
        input_text = input_text.replace('Human: ','')
        input_text = input_text.replace('Assistant: ','')

        answers = data['answers']
        rank1_answer = next((ans['answer'] for ans in answers if ans['rank'] == 1), None)
        other_answers = [ans['answer'] for ans in answers if ans['rank'] != 1]
        # other_answers = [ans['answer'] for ans in answers if ans['rank'] != 1 and ans['answer']!=rank1_answer]


        if rank1_answer and other_answers:
            random_other_answer = random.choice(other_answers)
        if rank1_answer==random_other_answer:
            random_other_answer = random_other_answer+'.'

        return RawSample(
                input=input_text,
                answer=rank1_answer,
                other_answer=random_other_answer,
                better=True,
        )

    def __len__(self) -> int:
        return len(self.data)



class NectarRMTrainDataset(NectarRMDataset):
    NAME: str = 'NectarRM/train'
    ALIASES: tuple[str, ...] = ('NectarRMDataset/train',)

    PATH: str = './datasets/nectar/rm.parquet'
    SPLIT: str = 'train'


class NectarRMTestDataset(NectarRMDataset):
    NAME: str = 'NectarRM/test'
    ALIASES: tuple[str, ...] = ('NectarRMDataset/test',)

    PATH: str = './datasets/nectar/test.parquet'
    SPLIT: str = 'test'




class NectarDataset(RawDataset):
    SPLIT: ClassVar[str]
    PATH: ClassVar[str]

    def __init__(self, path: str | None = None) -> None:
         self.data= pd.read_parquet(self.PATH)


    def __getitem__(self, index: int) -> RawSample:
        data = self.data.iloc[index]


        input_text = data['prompt']
        input_text = input_text.replace('Human: ','')
        input_text = input_text.replace('Assistant: ','')
        answer = ''
        answers = data['answers']
        for ans in answers:
            if ans['rank'] == 1:
                answer = ans['answer']
                break


        return RawSample(
                input=input_text,
                answer=answer,
        )

    def __len__(self) -> int:
        return len(self.data)


class NectarSFTTrainDataset(NectarDataset):
    NAME: str = 'NectarSFT/train'
    ALIASES: tuple[str, ...] = ('NectarSFTDataset/train',)
    PATH: str = './datasets/nectar/sft.parquet'
    SPLIT: str = 'train'

class NectarRLHFTrainDataset(NectarDataset):
    NAME: str = 'NectarRLHF/train'
    ALIASES: tuple[str, ...] = ('NectarRLHFDataset/train',)
    PATH: str = './datasets/nectar/rlhf.parquet'
    SPLIT: str = 'train'

class NectarRLHFTestDataset(NectarDataset):
    NAME: str = 'NectarRLHF/test'
    ALIASES: tuple[str, ...] = ('NectarRLHFDataset/test',)
    PATH: str = './datasets/nectar/test.parquet'
    SPLIT: str = 'test'

