# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.

# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.
# --------------------------------------------------------
# References:
# DeiT: https://github.com/facebookresearch/deit
# BEiT: https://github.com/microsoft/unilm/tree/master/beit
# --------------------------------------------------------
import argparse
import datetime
import json
import numpy as np
import os
import time
import torch
import copy
import torch.backends.cudnn as cudnn
import torchvision.transforms as transforms
import torchvision.datasets as datasets
import timm
#assert timm.__version__ == "0.3.2"  # version check
import timm.optim.optim_factory as optim_factory
import util.misc as misc
import models_mae
import util.lr_decay as lrd
import util.misc as misc
import random

from pathlib import Path
from torch.utils.tensorboard import SummaryWriter
from util.datasets import build_dataset
from util.pos_embed import interpolate_pos_embed
from util.misc import NativeScalerWithGradNormCount as NativeScaler
from new_constrain_opt import Constraint
from engine_pretrain import train_one_epoch
from torch.utils.data import Dataset


def get_args_parser():
    parser = argparse.ArgumentParser('MAE pre-training', add_help=False)
    parser.add_argument('--batch_size', default=64, type=int,
                        help='Batch size per GPU (effective batch size is batch_size * accum_iter * # gpus')
    parser.add_argument('--epochs', default=4, type=int)
    parser.add_argument('--accum_iter', default=1, type=int,
                        help='Accumulate gradient iterations (for increasing the effective batch size under memory constraints)')

    # Model parameters
    parser.add_argument('--model', default='mae_vit_base_patch16', type=str, metavar='MODEL',
                        help='Name of model to train')
    
    parser.add_argument('--chkpt', default='/root/autodl-tmp/img2img_unlearning/mae-main/mae_visualize_vit_base.pth',
                        help='finetune from checkpoint')

    parser.add_argument('--input_size', default=224, type=int,
                        help='images input size')

    parser.add_argument('--mask_ratio', default=0.25, type=float,
                        help='Masking ratio (percentage of removed patches).')

    parser.add_argument('--norm_pix_loss', action='store_true',
                        help='Use (per-patch) normalized pixels as targets for computing loss')
    parser.set_defaults(norm_pix_loss=False)

    # Optimizer parameters
    parser.add_argument('--weight_decay', type=float, default=0.00,
                        help='weight decay (default: 0.00)')

    parser.add_argument('--lr', type=float, default=None, metavar='LR',
                        help='learning rate (absolute lr)')
    parser.add_argument('--blr', type=float, default=1e-3, metavar='LR',
                        help='base learning rate: absolute_lr = base_lr * total_batch_size / 256')
    parser.add_argument('--min_lr', type=float, default=0., metavar='LR',
                        help='lower lr bound for cyclic schedulers that hit 0')

    parser.add_argument('--warmup_epochs', type=int, default=40, metavar='N',
                        help='epochs to warmup LR')

    # Dataset parameters
    parser.add_argument('--data_path_forget', default='/root/autodl-tmp/img2img_unlearning/mae-main/imagenet_forget', type=str,
                        help='forget dataset path')
    parser.add_argument('--nb_classes_forget', default=100, type=int,
                        help='number of the classification types')
    parser.add_argument('--data_path_retain', default='/root/autodl-tmp/img2img_unlearning/mae-main/imagenet_retain', type=str,
                        help='retain dataset path')
    parser.add_argument('--nb_classes_retain', default=100, type=int,
                        help='number of the classification types')

    parser.add_argument('--output_dir', default='./output_dir',
                        help='path where to save, empty for no saving')
    parser.add_argument('--log_dir', default='./output_dir',
                        help='path where to tensorboard log')
    parser.add_argument('--device', default='cuda',
                        help='device to use for training / testing')
    parser.add_argument('--seed', default=0, type=int)
    parser.add_argument('--resume', default='',
                        help='resume from checkpoint')

    parser.add_argument('--start_epoch', default=0, type=int, metavar='N',
                        help='start epoch')
    parser.add_argument('--num_workers', default=10, type=int)
    parser.add_argument('--pin_mem', action='store_true',
                        help='Pin CPU memory in DataLoader for more efficient (sometimes) transfer to GPU.')
    parser.add_argument('--no_pin_mem', action='store_false', dest='pin_mem')
    parser.set_defaults(pin_mem=True)

    # distributed training parameters
    parser.add_argument('--world_size', default=1, type=int,
                        help='number of distributed processes')
    parser.add_argument('--local_rank', default=-1, type=int)
    parser.add_argument('--dist_on_itp', action='store_true')
    parser.add_argument('--dist_url', default='env://',
                        help='url used to set up distributed training')
    
    # choose method
    parser.add_argument('--method', default='ours', help='please choose unlearning method')

    return parser

def reduce_dataset(da, ratio):
    """
    Reduces the size of da in-place, keeping only the specified ratio of images.

    da: dataset to be reduced.
    ratio: fraction of images to keep (between 0 and 1).
    """
    assert 0 <= ratio <= 1, "ratio should be between 0 and 1"

    num_samples = int(len(da) * ratio)
    selected_samples = random.sample(da.imgs, num_samples)

    da.imgs = selected_samples
    da.samples = selected_samples
    
    print(f'retain set has been reduce!')
    print(f'current retain samples length is: {len(da)}.')
    if hasattr(da, 'targets'):
        da.targets = [item[1] for item in selected_samples]

def expand_dataset(da1, da2):
    """
    Expands the da2 dataset in-place to match the number of images in da1.

    da1: Larger dataset.
    da2: Smaller dataset which will be expanded.
    """
    len_da1, len_da2 = len(da1), len(da2)
    print(len_da1, len_da2)
    if len_da2 >= len_da1:
        return  # da2 is already larger or equal to da1 in size
    
    diff = len_da1 - len_da2
    print(f'diff is : {diff}')

    # Create a deep copy of da2 to avoid modifying the original da2.
    da2_copy = copy.deepcopy(da2)

    # Randomly select samples from da2_copy to create new samples until the size matches da1.
    additional_samples = [da2_copy.imgs[random.randint(0, len_da2 - 1)] for _ in range(diff)]
    print(f'additional samples is : {len(additional_samples)}')

    # Now extend the imgs and targets (if exists) with the additional samples
    da2_copy.imgs.extend(additional_samples)
    
    # If 'samples' is a separate attribute, update it as well - otherwise, this is not needed.
    if 'samples' in dir(da2_copy) and da2_copy.samples is not da2_copy.imgs:
        da2_copy.samples.extend(additional_samples)
    
    if hasattr(da2_copy, 'targets'):
        # Make sure to extend the targets with the labels corresponding to the randomly chosen samples
        additional_labels = [da2_copy.targets[random.randint(0, len_da2 - 1)] for _ in range(diff)]
        da2_copy.targets.extend(additional_labels)

    print(f'retain set has been extended!')
    print(f'current forget samples length is: {len(da1)}---new retain samples length is: {len(da2_copy)}.')
    
    # Return the new expanded dataset. The original da2 remains unchanged.
    return da2_copy

def main(args):
    misc.init_distributed_mode(args)

    print('job dir: {}'.format(os.path.dirname(os.path.realpath(__file__))))
    print("{}".format(args).replace(', ', ',\n'))

    device = torch.device(args.device)

    # fix the seed for reproducibility
    seed = args.seed + misc.get_rank()
    torch.manual_seed(seed)
    np.random.seed(seed)
    
    # get unlearning method
    method = args.method
    mask_ratio = args.mask_ratio

    cudnn.benchmark = True

    # simple augmentation
    # dataset_forget = build_dataset(is_forget=True, args=args)
    # dataset_retain = build_dataset(is_forget=False, args=args)
    
    transform = transforms.Compose([
            transforms.RandomResizedCrop(args.input_size, scale=(0.2, 1.0), interpolation=3),  # 3 is bicubic
            transforms.RandomHorizontalFlip(),
            transforms.ToTensor(),
            transforms.Normalize(mean=[0.485, 0.456, 0.406], std=[0.229, 0.224, 0.225])])
    
    dataset_forget = datasets.ImageFolder(args.data_path_forget, transform=transform)
    print(dataset_forget)
    
    dataset_retain = datasets.ImageFolder(args.data_path_retain, transform=transform)
    print(dataset_retain)
    
    dataset_gaussian = datasets.ImageFolder('/root/autodl-tmp/img2img_unlearning/mae-mage-data/gaussian_images_multi_10_100', transform=transform)
    print(dataset_gaussian)
    
    print(dataset_gaussian[0][0].size)
    
    retain_reduction = False
    retain_ratio = 0.8
    if retain_reduction:
        reduce_dataset(dataset_retain, retain_ratio)
        dataset_retain = expand_dataset(dataset_forget, dataset_retain)
    
    if True:  # args.distributed:
        num_tasks = misc.get_world_size()
        global_rank = misc.get_rank()
        
        sampler_forget = torch.utils.data.DistributedSampler(
            dataset_forget, num_replicas=num_tasks, rank=global_rank, shuffle=True
        )
        print("Sampler_forget = %s" % str(sampler_forget))

        sampler_retain = torch.utils.data.DistributedSampler(
        dataset_retain, num_replicas=num_tasks, rank=global_rank, shuffle=True)  # shuffle=True to reduce monitor bias
        
        print("Sampler_retain = %s" % str(sampler_forget))
        
        sampler_gaussian = torch.utils.data.DistributedSampler(
        dataset_gaussian, num_replicas=num_tasks, rank=global_rank, shuffle=True)  # shuffle=True to reduce monitor bias
        
        print("Sampler_gaussian = %s" % str(sampler_gaussian))
    else:
        sampler_forget = torch.utils.data.RandomSampler(dataset_forget)
        sampler_retain = torch.utils.data.RandomSampler(dataset_retain)
        sampler_gaussian = torch.utils.data.RandomSampler(dataset_gaussian)

    if global_rank == 0 and args.log_dir is not None:
        os.makedirs(args.log_dir, exist_ok=True)
        log_writer = SummaryWriter(log_dir=args.log_dir)
    else:
        log_writer = None

    # args.num_workers = 2
    # args.pin_mem = False
    data_loader_forget = torch.utils.data.DataLoader(
        dataset_forget, 
        sampler=sampler_forget,
        batch_size=args.batch_size,
        num_workers=args.num_workers,
        pin_memory=args.pin_mem,
        drop_last=True,
    )

    data_loader_retain = torch.utils.data.DataLoader(
        dataset_retain, 
        sampler=sampler_retain,
        batch_size=args.batch_size,
        num_workers=args.num_workers,
        pin_memory=args.pin_mem,
        drop_last=False
    )
    
    data_loader_gaussian = torch.utils.data.DataLoader(
        dataset_gaussian, 
        sampler=sampler_gaussian,
        batch_size=args.batch_size,
        num_workers=args.num_workers,
        pin_memory=args.pin_mem,
        drop_last=True,
    )
    
    # define the model
    model = models_mae.__dict__[args.model](norm_pix_loss=args.norm_pix_loss)
    
    if args.chkpt:
        checkpoint = torch.load(args.chkpt, map_location='cpu')
        print("Load pre-trained checkpoint from: %s" % args.chkpt)
        
        msg = model.load_state_dict(checkpoint['model'], strict=False)
        print(msg)

    model.to(device)

    model_without_ddp = model
    print("Model = %s" % str(model_without_ddp))

    eff_batch_size = args.batch_size * args.accum_iter * misc.get_world_size()
    
    if args.lr is None:  # only base_lr is specified
        args.lr = args.blr * eff_batch_size / 256

    print("base lr: %.2e" % (args.lr * 256 / eff_batch_size))
    print("actual lr: %.2e" % args.lr)

    print("accumulate grad iterations: %d" % args.accum_iter)
    print("effective batch size: %d" % eff_batch_size)

    if args.distributed:
        model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu], find_unused_parameters=True)
        model_without_ddp = model.module
    
    # following timm: set wd as 0 for bias and norm layers
    param_groups = optim_factory.param_groups_layer_decay(model_without_ddp, args.weight_decay)
    
    if method == 'ours':
        base_optimizer = torch.optim.AdamW(param_groups, lr=args.lr, betas=(0.9, 0.95))
        print(base_optimizer)
        optimizer = Constraint(model.parameters(), base_optimizer=base_optimizer)
    else:
        optimizer = torch.optim.AdamW(param_groups, lr=args.lr, betas=(0.9, 0.95))
        
    loss_scaler = NativeScaler()

    criterion = torch.nn.MSELoss(reduction='mean')

    misc.load_model(args=args, model_without_ddp=model_without_ddp, optimizer=optimizer, loss_scaler=loss_scaler)
    
    # freeze decoder params
    def freeze_decoder_parameters(model):
        for param in model.decoder_blocks.parameters():
            param.requires_grad = False
        for param in model.decoder_embed.parameters():
            param.requires_grad = False
        for param in model.decoder_norm.parameters():
            param.requires_grad = False
        for param in model.decoder_pred.parameters():
            param.requires_grad = False
        model.mask_token.requires_grad = False
    
    if method in ['iclr', 'ours']:
        freeze_decoder_parameters(model)
        
    if method != 'ours':
        model.mask_type = 'random'
        print(f'model mask type has been set to {model.mask_type}!')
        
    def clone_and_freeze_model(original_model: torch.nn.Module) -> torch.nn.Module:
        cloned_model = copy.deepcopy(original_model)

        for param in cloned_model.parameters():
            param.requires_grad = False

        return cloned_model
    model_clone = clone_and_freeze_model(model).to(device) 

    print(f"Start training for {args.epochs} epochs")
    start_time = time.time()

    for epoch in range(args.start_epoch, args.epochs):
        if args.distributed:
            data_loader_train.sampler.set_epoch(epoch)
        train_stats = train_one_epoch(
            model, model_clone, criterion, method, mask_ratio,
            data_loader_forget, data_loader_retain, data_loader_gaussian,
            optimizer, device, epoch, loss_scaler,
            log_writer=log_writer,
            args=args
        )
        if args.output_dir and (epoch % 1 == 0 or epoch + 1 == args.epochs):
            misc.save_model(
                args=args, model=model, model_without_ddp=model_without_ddp, optimizer=optimizer,
                loss_scaler=loss_scaler, epoch=epoch)

        log_stats = {**{f'train_{k}': v for k, v in train_stats.items()},
                        'epoch': epoch,}

        if args.output_dir and misc.is_main_process():
            if log_writer is not None:
                log_writer.flush()
            with open(os.path.join(args.output_dir, "log.txt"), mode="a", encoding="utf-8") as f:
                f.write(json.dumps(log_stats) + "\n")

    total_time = time.time() - start_time
    total_time_str = str(datetime.timedelta(seconds=int(total_time)))
    print('Training time {}'.format(total_time_str))


if __name__ == '__main__':
    args = get_args_parser()
    args = args.parse_args()
    if args.output_dir:
        Path(args.output_dir).mkdir(parents=True, exist_ok=True)
    main(args)
