import os
import torch
import torch.nn.functional as F
from torch.optim import Adam
from utils import soft_update, hard_update
from data_collection.model import GaussianPolicy, QNetwork


class SAC(object):
    def __init__(self, env, hidden_dim, alpha, lr, gamma, tau, device='cpu'):

        self.gamma = gamma
        self.tau = tau
        self.alpha = alpha

        self.device = torch.device(device if torch.cuda.is_available() else 'cpu') 

        self.critic = QNetwork(
            env.observation_space.shape[0], 
            env.action_space.shape[0], 
            hidden_dim
        ).to(self.device)
        self.critic_optim = Adam(self.critic.parameters(), lr=lr)

        self.critic_target = QNetwork(
            env.observation_space.shape[0], 
            env.action_space.shape[0], 
            hidden_dim
        ).to(self.device)
        hard_update(self.critic_target, self.critic)

        self.policy = GaussianPolicy(
            env.observation_space.shape[0], 
            env.action_space.shape[0], 
            hidden_dim, 
            env.action_space
        ).to(self.device)
        self.policy_optim = Adam(self.policy.parameters(), lr=lr)


    def select_action(self, state, evaluate=False):
        state = torch.FloatTensor(state).to(self.device).unsqueeze(0)
        if evaluate is False:
            action, _, _ = self.policy.sample(state)
        else:
            _, _, action = self.policy.sample(state)
        return action.detach().cpu().numpy()[0]

    def update_parameters(self, memory, batch_size):
        # Sample a batch from memory
        state_batch, action_batch, reward_batch, next_state_batch, done_batch, mask_match = memory.sample(batch_size=batch_size)

        state_batch = torch.FloatTensor(state_batch).to(self.device)
        next_state_batch = torch.FloatTensor(next_state_batch).to(self.device)
        action_batch = torch.FloatTensor(action_batch).to(self.device)
        reward_batch = torch.FloatTensor(reward_batch).to(self.device).unsqueeze(1)
        done_batch = torch.FloatTensor(done_batch).to(self.device).unsqueeze(1)
        mask_match = torch.FloatTensor(mask_match).to(self.device).unsqueeze(1)

        with torch.no_grad():
            next_state_action, next_state_log_pi, _ = self.policy.sample(next_state_batch)
            qf1_next_target, qf2_next_target = self.critic_target(next_state_batch, next_state_action)
            min_qf_next_target = torch.min(qf1_next_target, qf2_next_target) - self.alpha * next_state_log_pi
            next_q_value = reward_batch + mask_match * self.gamma * (min_qf_next_target)

        # Two Q-functions to mitigate positive bias in the policy improvement step
        qf1, qf2 = self.critic(state_batch, action_batch)  

        # JQ = 𝔼(st,at)~D[0.5(Q1(st,at) - r(st,at) - γ(𝔼st+1~p[V(st+1)]))^2]
        qf1_loss = F.mse_loss(qf1, next_q_value)  
        # JQ = 𝔼(st,at)~D[0.5(Q1(st,at) - r(st,at) - γ(𝔼st+1~p[V(st+1)]))^2]
        qf2_loss = F.mse_loss(qf2, next_q_value)  

        qf_loss = qf1_loss + qf2_loss

        self.critic_optim.zero_grad()
        qf_loss.backward()
        self.critic_optim.step()

        pi, log_pi, _ = self.policy.sample(state_batch)

        qf1_pi, qf2_pi = self.critic(state_batch, pi)
        min_qf_pi = torch.min(qf1_pi, qf2_pi)

        # Jπ = 𝔼st∼D,εt∼N[α * logπ(f(εt;st)|st) − Q(st,f(εt;st))]
        policy_loss = ((self.alpha * log_pi) - min_qf_pi).mean() 

        self.policy_optim.zero_grad()
        policy_loss.backward()
        self.policy_optim.step()

        soft_update(self.critic_target, self.critic, self.tau)

        return qf1_loss.item(), qf2_loss.item(), policy_loss.item()
    
    def save(self, path):
        torch.save(
            {
                'critic': self.critic.state_dict(),
                'policy': self.policy.state_dict(),
            },
            path
        )
   
    def load(self, path):
        ckp = torch.load(path, map_location=self.device)
        self.critic.load_state_dict(ckp['critic'])
        self.critic_target.load_state_dict(ckp['critic'])
        self.policy.load_state_dict(ckp['policy'])