import numpy as np
import torch
import torch.optim as optim
import torch.nn as nn
import os

from meta_test_algo.network import es_policy


class EvolutionStrategies_FULL:
    def __init__(self, 
                 obs_dim, 
                 action_dim,
                 net_size,
                 latent_action_dim,
                 device,
                 es_params,
                 **kwargs):

        self.max_path_length = kwargs['max_path_length']
        self.reward_scale = kwargs['reward_scale']

        self.obs_dim = obs_dim
        self.action_dim = action_dim
        self.device = device

        self.n_rollouts = es_params['n_rollouts']
        self.noise_sigma = es_params['noise_sigma']
        self.lr = es_params['lr']
        self.elite_frac = es_params['elite_frac']
        
        self.policy = es_policy(obs_dim,action_dim,net_size,latent_action_dim).to(self.device)
        self.head_param = nn.utils.parameters_to_vector(self.policy.last_layer.parameters()).detach().cpu().numpy()
        self.param_shape = self.head_param.shape[0]
        self.prev_elite_noise = []
        self.prev_elite_rewards = []

    def load_shared_network(self,policy_params):
        self.policy.shared_layer.load_state_dict(policy_params)
        for param in self.policy.shared_layer.parameters():
            param.requires_grad = False
    
    def evaluate_head(self,env):
        obs = env.reset()
        env_step = 0
        total_reward = 0
        while env_step<self.max_path_length:
            env_step += 1
            action = self.select_action(obs)
            obs, reward, done, env_info = env.step(action)
            total_reward += reward
            if done:
                break
        return total_reward

    def es_adapt_head_mixed_sampling(self, env, head_ratio=0.6):
        full_param_vec = torch.nn.utils.parameters_to_vector(self.policy.parameters()).detach().cpu().numpy()
        param_size = len(full_param_vec)

        # head 파라미터 구간 설정
        head_param_vec = torch.nn.utils.parameters_to_vector(self.policy.last_layer.parameters()).detach().cpu().numpy()
        head_size = len(head_param_vec)
        head_start = param_size - head_size
        head_indices = np.arange(head_start, param_size)
        non_head_indices = np.arange(0, head_start)

        # 샘플링 인덱스
        num_head = int(self.param_shape * head_ratio)
        num_other = self.param_shape - num_head

        sampled_head_idx = np.random.choice(head_indices, size=num_head, replace=False)
        sampled_other_idx = np.random.choice(non_head_indices, size=num_other, replace=False)
        sampled_idx = np.concatenate([sampled_head_idx, sampled_other_idx])

        pair_list = []

        for _ in range(self.n_rollouts // 2):
            epsilon = np.zeros_like(full_param_vec)
            sampled_noise = np.random.randn(self.param_shape)
            epsilon[sampled_idx] = sampled_noise

            # θ + ε
            perturbed_pos = full_param_vec + self.noise_sigma * epsilon
            torch.nn.utils.vector_to_parameters(
                torch.tensor(perturbed_pos, dtype=torch.float32).to(self.device),
                self.policy.parameters()
            )
            r_pos = self.evaluate_head(env)

            # θ - ε
            perturbed_neg = full_param_vec - self.noise_sigma * epsilon
            torch.nn.utils.vector_to_parameters(
                torch.tensor(perturbed_neg, dtype=torch.float32).to(self.device),
                self.policy.parameters()
            )
            r_neg = self.evaluate_head(env)

            r_diff = r_pos - r_neg
            r_mean = (r_pos + r_neg) / 2.0
            pair_list.append((epsilon, r_diff, r_mean))

        num_elites = max(1, int(len(pair_list) * self.elite_frac))
        elite_pairs = sorted(pair_list, key=lambda x: x[2])[-num_elites:]

        r_diffs = np.array([r_diff for _, r_diff, _ in elite_pairs])
        r_diffs = (r_diffs - r_diffs.mean()) / (r_diffs.std() + 1e-6)

        grad_estimate = np.zeros_like(full_param_vec)
        for (eps, _, _), r in zip(elite_pairs, r_diffs):
            grad_estimate += r * eps
        grad_estimate /= num_elites

        full_param_vec[sampled_idx] += self.lr * grad_estimate[sampled_idx]

        torch.nn.utils.vector_to_parameters(
            torch.tensor(full_param_vec, dtype=torch.float32).to(self.device),
            self.policy.parameters()
        )
    
    def select_action(self,obs):
        obs = torch.tensor(obs, dtype=torch.float32).to(self.device)
        action = self.policy(obs)
        return action.cpu().numpy()

    def collet_data_and_train_filter(self,env):
        self.es_adapt_head_mixed_sampling(env,head_ratio=0.5)

    def evaluation(self,env):
        with torch.no_grad():
            n_eval_epi = 3
            returns = 0
            for _ in range(n_eval_epi):
                episode_returns = 0
                env_step = 0
                o = env.reset()
                while env_step<self.max_path_length:
                    env_step += 1
                    a = self.select_action(o)
                    next_o, r, d, env_info = env.step(a)
                    episode_returns += r
                    o = next_o
                    if d:
                        break
                returns += episode_returns
        return returns/n_eval_epi   