import json 
import dataclasses
from enum import auto, Enum
from typing import List, Tuple, Any

import torch
from transformers import AutoTokenizer, AutoModelForCausalLM


class SeparatorStyle(Enum):
    """Different separator style."""
    SINGLE = auto()
    TWO = auto()


@dataclasses.dataclass
class Conversation:
    """A class that keeps all conversation history."""
    system: str
    roles: List[str]
    messages: List[List[str]]
    offset: int
    sep_style: SeparatorStyle = SeparatorStyle.SINGLE
    sep: str = "###"
    sep2: str = None

    skip_next: bool = False

    def get_prompt(self):
        if self.sep_style == SeparatorStyle.SINGLE:
            ret = self.system + self.sep
            for role, message in self.messages:
                if message:
                    ret += role + ": " + message + self.sep
                else:
                    ret += role + ":"
            return ret
        elif self.sep_style == SeparatorStyle.TWO:
            seps = [self.sep, self.sep2]
            ret = self.system + seps[0]
            for i, (role, message) in enumerate(self.messages):
                if message:
                    ret += role + ": " + message + seps[i % 2]
                else:
                    ret += role + ":"
            return ret
        else:
            raise ValueError(f"Invalid style: {self.sep_style}")

    def append_message(self, role, message):
        self.messages.append([role, message])

    def to_gradio_chatbot(self):
        ret = []
        for i, (role, msg) in enumerate(self.messages[self.offset:]):
            if i % 2 == 0:
                ret.append([msg, None])
            else:
                ret[-1][-1] = msg
        return ret

    def copy(self):
        return Conversation(
            system=self.system,
            roles=self.roles,
            messages=[[x, y] for x, y in self.messages],
            offset=self.offset,
            sep_style=self.sep_style,
            sep=self.sep,
            sep2=self.sep2,)

    def dict(self):
        return {
            "system": self.system,
            "roles": self.roles,
            "messages": self.messages,
            "offset": self.offset,
            "sep": self.sep,
            "sep2": self.sep2,
        }

prefixes = {
    'pick_and_place': 'put',
    'pick_clean_then_place': 'put',  # 
    'pick_heat_then_place': 'put',  # 
    'pick_cool_then_place': 'put',  # they are actually the same;
    'look_at_obj': 'put',  #
    'pick_two_obj': 'put'
}

empty_conv = Conversation(
    system="A chat between a curious human and an artificial intelligence assistant. "
           "The assistant gives helpful, detailed, and polite answers to the human's questions.",
    roles=("Human", "Assistant"),
    messages=(),
    offset=0,
    sep_style=SeparatorStyle.SINGLE,
    sep="###",
)


class HumanModel():

    def __init__(self, model_path, tokenizer_path, prompt_path):
        print("initializing human model")
        self.conv = None
        self.facts = None
        self.ask_first_time = None

        self.prompt_path = prompt_path

        self.tokenizer = AutoTokenizer.from_pretrained(tokenizer_path)
        self.model = AutoModelForCausalLM.from_pretrained(model_path, device_map='auto', load_in_8bit=True)

    
    def init_prompt(self, facts, recount=True):
        if recount:
            self.ask_cnt = 0
        self.ask_first_time = True
        self.facts = facts
        self.conv = empty_conv.copy()
        with open(self.prompt_path, 'r') as f:
            d = json.load(f)
        
        try:
            for i in range(20):
                self.conv.append_message(self.conv.roles[i % 2], d[str(i)])
        except:
            pass 
    
    def update_facts(self, facts):
        self.ask_first_time = True
        self.facts = facts
        self.init_prompt(facts, recount=False)
    
    @torch.inference_mode()
    def answer(self, question):

        self.ask_cnt += 1
        if self.ask_first_time:
            new_input = "Read the following paragraph and answer questions: " + self.facts + "The questions is: " + question
            self.ask_first_time = False
        else:
            new_input = "Another question is: " + question
        
        self.conv.append_message(self.conv.roles[0], new_input)

        prompt = self.conv.get_prompt()

        inputs = self.tokenizer([prompt])

        with torch.no_grad():
            output_ids = self.model.generate(
                torch.as_tensor(inputs.input_ids).to(self.model.device),
                do_sample=True,
                temperature=0.7,
                max_new_tokens=50)
        outputs = self.tokenizer.batch_decode(output_ids, skip_special_tokens=True)[0]
        try:
            index = outputs.index(self.conv.sep, len(prompt))
        except ValueError:
            outputs += self.conv.sep
            index = outputs.index(self.conv.sep, len(prompt))

        outputs = outputs[len(prompt) + len(self.conv.roles[1]) + 2:index].strip().split('\n')[0]

        self.conv.append_message(self.conv.roles[1], outputs)

        return outputs
