import torch
from torch.nn import Parameter
from torch_scatter import scatter_add
from torch_geometric.nn.conv import MessagePassing
from torch_geometric.utils import add_remaining_self_loops




class nonWeightedGCNConv(MessagePassing):
    def __init__(self, **kwargs):
        super(nonWeightedGCNConv, self).__init__(aggr='add', **kwargs)


    @staticmethod
    def norm(edge_index, num_nodes, edge_weight=None, dtype=None):
        if edge_weight is None:
            edge_weight = torch.ones((edge_index.size(1), ), dtype=dtype,
                                     device=edge_index.device)

        edge_index, edge_weight = add_remaining_self_loops(
            edge_index, edge_weight, 1., num_nodes)

        row, col = edge_index
        deg = scatter_add(edge_weight, row, dim=0, dim_size=num_nodes)
        deg_inv_sqrt = deg.pow(-0.5)
        deg_inv_sqrt[deg_inv_sqrt == float('inf')] = 0

        return edge_index, deg_inv_sqrt[row] * edge_weight * deg_inv_sqrt[col]

    def forward(self, x, edge_index, edge_weight=None, model_weight=None, model_bias=None):
        
        x = torch.matmul(x, model_weight)
        edge_index, norm = self.norm(edge_index, x.size(self.node_dim), edge_weight, x.dtype)

        return self.propagate(edge_index, x=x, norm=norm, model_bias=model_bias)

    def message(self, x_j, norm):
        return norm.view(-1, 1) * x_j if norm is not None else x_j

    def update(self, aggr_out, model_bias=None):
        if model_bias is not None:
            aggr_out = aggr_out + model_bias
        return aggr_out

    def __repr__(self):
        return '{}({}, {})'.format(self.__class__.__name__, self.in_channels,
                                   self.out_channels)
