import numpy as np
import torch
import json
import os
import glob

################################################################

def get_parent_directory(path:str)->str:
    return os.path.dirname(path)+"/"

def get_path_without_extension(path:str)->str:
    return os.path.splitext(path)[0]

def create_dir(path:str)->None:
    os.makedirs(path, exist_ok=True)
    
def collect_files(path:str, extension:str='.json')->list:
    json_files = glob.glob(os.path.join(path, '*'+extension))
    return [os.path.basename(file) for file in json_files]

def is_integer(n:str)->bool:
    return n.isdigit()

def is_float(n:str)->bool:
    return n.replace('.', '', 1).isdigit() or n.rfind('e-')!=-1

def get_dir(path:str):
    dir = [d for d in os.listdir(path) if os.path.isdir(os.path.join(path, d))]
    return dir
    
################################################################

class NumpyEncoder(json.JSONEncoder):
    """ 
        Custom encoder for numpy data types 
    """
    def default(self, obj):
        if isinstance(obj, (np.int_, np.intc, np.intp, np.int8,
                            np.int16, np.int32, np.int64, np.uint8,
                            np.uint16, np.uint32, np.uint64)):
            return int(obj)
        if isinstance(obj, (np.float_, np.float16, np.float32, np.float64)):
            return float(obj)
        if isinstance(obj, (np.complex_, np.complex64, np.complex128)):
            return {'real': obj.real, 'imag': obj.imag}
        if isinstance(obj, (np.ndarray,)):
            return obj.tolist()
        if isinstance(obj, (np.bool_)):
            return bool(obj)
        if isinstance(obj, (np.void)):
            return None
        return super(NumpyEncoder, self).default(obj)
    
    
def save_dict(path:str, data:dict) -> None:
    try:
        with open(path, 'w') as file:
            json.dump(data, file, indent=4, sort_keys=True, separators=(', ', ': '), ensure_ascii=False, cls=NumpyEncoder)
    except IOError as e:
        raise IOError(f"Unable to save data to '{path}'. Error: {e}")


def load_dict(path: str) -> dict:
    try:
        with open(path, 'r') as json_file:
            data = json.load(json_file)
        return data
    except FileNotFoundError:
        raise FileNotFoundError(f"File '{path}' not found.")
    except json.JSONDecodeError:
        raise ValueError(f"Invalid JSON data in '{path}'.")
    


  