# coding=utf-8
# Copyright 2020 The Google Research Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Pybullet simulation of a Aliengo robot."""
import sys
import os

curPath = os.path.abspath(os.path.dirname(__file__))
rootPath = os.path.split(curPath)[0]
sys.path.append(rootPath)

import math
import os
import re
import numpy as np
import pybullet as pyb

from robots import a1_pose_utils
# from robots import a1_constants
from robots import a1_motor
from robots import minitaur
from robots import robot_config
from build_envs import locomotion_gym_config
import h5py

NUM_MOTORS = 12
NUM_LEGS = 4
MOTOR_NAMES = [
    # "FR_hip_motor_2_chassis_joint",
    # "FR_upper_leg_2_hip_motor_joint",
    # "FR_lower_leg_2_upper_leg_joint",
    # "FL_hip_motor_2_chassis_joint",
    # "FL_upper_leg_2_hip_motor_joint",
    # "FL_lower_leg_2_upper_leg_joint",
    # "RR_hip_motor_2_chassis_joint",
    # "RR_upper_leg_2_hip_motor_joint",
    # "RR_lower_leg_2_upper_leg_joint",
    # "RL_hip_motor_2_chassis_joint",
    # "RL_upper_leg_2_hip_motor_joint",
    # "RL_lower_leg_2_upper_leg_joint",
    "FR_hip_joint",
    "FR_thigh_joint",
    "FR_calf_joint",

    "FL_hip_joint",
    "FL_thigh_joint",
    "FL_calf_joint",

    "RR_hip_joint",
    "RR_thigh_joint",
    "RR_calf_joint",

    "RL_hip_joint",
    "RL_thigh_joint",
    "RL_calf_joint",
]

INIT_RACK_POSITION = [0, 0, 1]
INIT_POSITION = [0, 0, 0.4]  # [0, 0, 0.4] ->
INIT_ORIENTATION = [0, 0, 0, 1]

# JOINT_DIRECTIONS = np.array([-1, 1, 1, 1, 1, 1, -1, 1, 1, 1, 1, 1])  # ??
JOINT_DIRECTIONS = np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1])  # ??

# HIP_JOINT_OFFSET = 0.0  # ??
# THIGH_JOINT_OFFSET = -0.6  # ??
# CALF_JOINT_OFFSET = 0.66  # ??
HIP_JOINT_OFFSET = 0.0  # ??
THIGH_JOINT_OFFSET = -0  # ??
CALF_JOINT_OFFSET = 0  # ??
DOFS_PER_LEG = 3
JOINT_OFFSETS = np.array(
    [HIP_JOINT_OFFSET, THIGH_JOINT_OFFSET, CALF_JOINT_OFFSET] * 4)
PI = math.pi

MAX_MOTOR_ANGLE_CHANGE_PER_STEP = 0.2  # ??
# _DEFAULT_HIP_POSITIONS = (
#     (0.21, -0.1157, 0),
#     (0.21, 0.1157, 0),
#     (-0.21, -0.1157, 0),
#     (-0.21, 0.1157, 0),
# )
_DEFAULT_HIP_POSITIONS = (
    (0.183, -0.047, 0),
    (0.183, 0.047, 0),
    (-0.183, -0.047, 0),
    (-0.183, 0.047, 0),
)

# ??
# ABDUCTION_P_GAIN = 220.0
# ABDUCTION_D_GAIN = 0.3
# HIP_P_GAIN = 220.0
# HIP_D_GAIN = 2.0
# KNEE_P_GAIN = 220.0
# KNEE_D_GAIN = 2.0
# motor_torque_limits = None

# ABDUCTION_P_GAIN = 100
# ABDUCTION_D_GAIN = 5
# HIP_P_GAIN = 300
# HIP_D_GAIN = 8
# KNEE_P_GAIN = 300
# KNEE_D_GAIN = 8
# motor_torque_limits = None

# ABDUCTION_P_GAIN = 80
# ABDUCTION_D_GAIN = 3
# HIP_P_GAIN = 250
# HIP_D_GAIN = 5
# KNEE_P_GAIN = 250
# KNEE_D_GAIN = 5
# motor_torque_limits = 200

ABDUCTION_P_GAIN = 100.0
ABDUCTION_D_GAIN = 1.
HIP_P_GAIN = 100.0
HIP_D_GAIN = 1
KNEE_P_GAIN = 100.0
KNEE_D_GAIN = 1
motor_torque_limits = 30

# ABDUCTION_P_GAIN = 5.0
# ABDUCTION_D_GAIN = 1.0
# HIP_P_GAIN = 5.0
# HIP_D_GAIN = 1.0
# KNEE_P_GAIN = 5.0
# KNEE_D_GAIN = 1.0
# motor_torque_limits = None

# Bases on the readings from Aliengo's default pose.

INIT_MOTOR_ANGLES = np.array([
                                 a1_pose_utils.A1_DEFAULT_ABDUCTION_ANGLE,
                                 a1_pose_utils.A1_DEFAULT_HIP_ANGLE,
                                 a1_pose_utils.A1_DEFAULT_KNEE_ANGLE
                             ] * NUM_LEGS)

# COM_OFFSET = -np.array([0.012731, 0.002186, 0.000515])
HIP_OFFSETS = np.array([[0.183, -0.047, 0.], [0.183, 0.047, 0.],
                        [-0.183, -0.047, 0.], [-0.183, 0.047, 0.]
                        ]) # + COM_OFFSET

def foot_position_in_hip_frame(angles, l_hip_sign=1):
  theta_ab, theta_hip, theta_knee = angles[0], angles[1], angles[2]
  l_up = 0.2
  l_low = 0.2
  l_hip = 0.08505 * l_hip_sign
  leg_distance = np.sqrt(l_up**2 + l_low**2 +
                         2 * l_up * l_low * np.cos(theta_knee))
  eff_swing = theta_hip + theta_knee / 2

  off_x_hip = -leg_distance * np.sin(eff_swing)
  off_z_hip = -leg_distance * np.cos(eff_swing)
  off_y_hip = l_hip

  off_x = off_x_hip
  off_y = np.cos(theta_ab) * off_y_hip - np.sin(theta_ab) * off_z_hip
  off_z = np.sin(theta_ab) * off_y_hip + np.cos(theta_ab) * off_z_hip
  return np.array([off_x, off_y, off_z])


def analytical_leg_jacobian(leg_angles, leg_id):
  """
  Computes the analytical Jacobian.
  Args:
  ` leg_angles: a list of 3 numbers for current abduction, hip and knee angle.
    l_hip_sign: whether it's a left (1) or right(-1) leg.
  """
  l_up = 0.2
  l_low = 0.2
  l_hip = 0.08505 * (-1)**(leg_id + 1)

  t1, t2, t3 = leg_angles[0], leg_angles[1], leg_angles[2]
  l_eff = np.sqrt(l_up**2 + l_low**2 + 2 * l_up * l_low * np.cos(t3))
  t_eff = t2 + t3 / 2
  J = np.zeros((3, 3))
  J[0, 0] = 0
  J[0, 1] = -l_eff * np.cos(t_eff)
  J[0, 2] = l_low * l_up * np.sin(t3) * np.sin(t_eff) / l_eff - l_eff * np.cos(
      t_eff) / 2
  J[1, 0] = -l_hip * np.sin(t1) + l_eff * np.cos(t1) * np.cos(t_eff)
  J[1, 1] = -l_eff * np.sin(t1) * np.sin(t_eff)
  J[1, 2] = -l_low * l_up * np.sin(t1) * np.sin(t3) * np.cos(
      t_eff) / l_eff - l_eff * np.sin(t1) * np.sin(t_eff) / 2
  J[2, 0] = l_hip * np.cos(t1) + l_eff * np.sin(t1) * np.cos(t_eff)
  J[2, 1] = l_eff * np.sin(t_eff) * np.cos(t1)
  J[2, 2] = l_low * l_up * np.sin(t3) * np.cos(t1) * np.cos(
      t_eff) / l_eff + l_eff * np.sin(t_eff) * np.cos(t1) / 2
  return J


# For JIT compilation
# foot_position_in_hip_frame_to_joint_angle(np.random.uniform(size=3), 1)
# foot_position_in_hip_frame_to_joint_angle(np.random.uniform(size=3), -1)


def foot_positions_in_base_frame(foot_angles):
  foot_angles = foot_angles.reshape((4, 3))
  foot_positions = np.zeros((4, 3))
  for i in range(4):
    foot_positions[i] = foot_position_in_hip_frame(foot_angles[i],
                                                   l_hip_sign=(-1)**(i + 1))
  # print(foot_positions + HIP_OFFSETS)
  # input()
  return foot_positions + HIP_OFFSETS

# _CHASSIS_NAME_PATTERN = re.compile(r"\w+_chassis_\w+")
# _MOTOR_NAME_PATTERN = re.compile(r"\w+_hip_motor_\w+")
# _KNEE_NAME_PATTERN = re.compile(r"\w+_lower_leg_\w+")
# _TOE_NAME_PATTERN = re.compile(r"jtoe\d*")
_HIP_NAME_PATTERN = re.compile(r"\w+_hip_\w+")
_THIGH_NAME_PATTERN = re.compile(r"\w+_thigh_\w+")
_CALF_NAME_PATTERN = re.compile(r"\w+_calf_\w+")
_FOOT_NAME_PATTERN = re.compile(r"\w+_foot_\w+")

ROOT_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
# URDF_FILENAME = ROOT_DIR + "/laikago_model/laikago_toes_limits.urdf"
# URDF_FILENAME = ROOT_DIR + "/laikago_model/laikago.urdf"

URDF_FILENAME = ROOT_DIR + "/a1/urdf/a1.urdf"
# URDF_FILENAME = ROOT_DIR + "/aliengo_model/urdf/aliengo.urdf"
# URDF_FILENAME = r"/home/amax/Westlake/laikago_pybullet/data/laikago/urdf/laikago.urdf"
# URDF_FILENAME = r"/home/amax/Westlake/aliengo_pybullet/data/aliengo/urdf/aliengo.urdf"

# ??
_BODY_B_FIELD_NUMBER = 2
_LINK_A_FIELD_NUMBER = 3

motor_pos_stddev = np.array([0.01078625925491068,
                             0.050329922827315056,
                             0.08194429842580836]*4)
motor_pos_vel_stddev = np.array([0.427526029776151 ,
                                 1.2235536518796661,
                                 2.321338267793847]*4)

# motor_torque_stddev = np.array([0.0,0.0,0.0]*4)

rpy_stddev = np.array([0.00526326,
                       0.00421531 ,
                       0.005])
rpy_rate_stddev = np.array([0.16191895,
                            0.12787766,
                            0.10788293])
com_vels_stddev = np.array([
                            0.01168824,
                            0.01268743,
                            0.00771031
                            ])

# f = h5py.File(ROOT_DIR + '/robots/obs_std_data.h5', 'r')
# # print('--iterms: ', len(f.keys()))
# # for i,key in enumerate(f.keys()):
# motor_pos_stddev = np.array(f["joint_std"])
# motor_pos_vel_stddev = np.array(f["joint_v_std"])
# rpy_stddev = np.array(f["rpy_std"])
# rpy_stddev = np.clip(rpy_stddev,a_min=0,a_max=0.02)
# rpy_rate_stddev = np.array(f["rpy_v_std"])
# com_vels_stddev = np.array(f["com_v_std"])
# f.close()

# motor_pos_stddev = np.array([0.01078625925491068,
#                              0.050329922827315056,
#                              0.08194429842580836]*4)
# motor_pos_vel_stddev = np.array([0.427526029776151 ,
#                                  1.2235536518796661,
#                                  2.321338267793847]*4)

# motor_torque_stddev = np.array([0.0,0.0,0.0]*4)

# rpy_stddev = np.array([0.00526326,
#                        0.00421531 ,
#                        0.005])
# rpy_rate_stddev = np.array([0.16191895,
#                             0.12787766,
#                             0.10788293])
# com_vels_stddev = np.array([
#                             0.01168824,
#                             0.01268743,
#                             0.00771031
#                             ])
SENSOR_NOISE_STDDEV = np.array([motor_pos_stddev,
                       motor_pos_vel_stddev,
                        # motor_torque_stddev,
                        rpy_stddev,
                        rpy_rate_stddev,
                        com_vels_stddev
                       ])*np.array([0.5])

# UPPER_BOUND = 6.28318548203
# LOWER_BOUND = -6.28318548203

# a1_Hip_max = 1.047
# a1_Hip_min = -0.873
# a1_Thigh_max = 3.927
# a1_Thigh_min = -0.524
# a1_Calf_max = -0.611
# a1_Calf_min = -2.775

# torque_max, torque_min = 9,-21
default_torque = 0
# torque_max, torque_min = 2,-2

# force_max, force_min = 0, -30
# force_z_max, force_z_min = 50,-20
# default_force = np.array([0,0,-60])
force_max, force_min = 30, -30
force_z_max, force_z_min = 60,-60
default_force = np.array([0,0,0])



L_half = 0.183
W_half = 0.047 + 0.08505
UPPER_BOUND,LOWER_BOUND = [0.3-L_half,-0.05+W_half,-0.15,
                           0.3-L_half,0.2-W_half,-0.15,
                           -0.1+L_half,-0.1+W_half,-0.15,
                           -0.1+L_half,0.2-W_half,-0.15],\
                          [0.05-L_half,-0.2+W_half,-0.35,
                           0.05-L_half,0.05-W_half,-0.35,
                           -0.25+L_half,-0.2+W_half,-0.35,
                           -0.25+L_half,0.1-W_half,-0.35]
foot_x_bound = [0.04, -0.04]
# foot_y_bound = [0.06, -0.06]
# foot_y_bound = [0.02, -0.02]
foot_y_bound = [0.04, -0.04]
foot_z_bound = [0.04, -0.]
UPPER_BOUND_fre, LOWER_BOUND_fre = 0.03, -0.03
# UPPER_BOUND_fre, LOWER_BOUND_fre = 0.1, -0.1

torque_max, torque_min = 2,-2

# q_dot_desired_bound = [5,-5]

class A1(minitaur.Minitaur):
    """A simulation for the A1 robot."""

    ACTION_CONFIG = [
        locomotion_gym_config.ScalarField(name="frequency_0", upper_bound=UPPER_BOUND_fre,
                                          lower_bound=LOWER_BOUND_fre),
        locomotion_gym_config.ScalarField(name="frequency_1", upper_bound=UPPER_BOUND_fre,
                                          lower_bound=LOWER_BOUND_fre),
        locomotion_gym_config.ScalarField(name="frequency_2", upper_bound=UPPER_BOUND_fre,
                                          lower_bound=LOWER_BOUND_fre),
        locomotion_gym_config.ScalarField(name="frequency_3", upper_bound=UPPER_BOUND_fre,
                                          lower_bound=LOWER_BOUND_fre),

        locomotion_gym_config.ScalarField(name="FF_x", upper_bound=foot_x_bound[0],
                                          lower_bound=foot_x_bound[1]),
        locomotion_gym_config.ScalarField(name="FF_y", upper_bound=foot_y_bound[0],
                                          lower_bound=foot_y_bound[1]),
        locomotion_gym_config.ScalarField(name="FF_z", upper_bound=foot_z_bound[0],
                                          lower_bound=foot_z_bound[1]),
        locomotion_gym_config.ScalarField(name="FL_x", upper_bound=foot_x_bound[0],
                                          lower_bound=foot_x_bound[1]),
        locomotion_gym_config.ScalarField(name="FL_y", upper_bound=foot_y_bound[0],
                                          lower_bound=foot_y_bound[1]),
        locomotion_gym_config.ScalarField(name="FL_z", upper_bound=foot_z_bound[0],
                                          lower_bound=foot_z_bound[1]),
        locomotion_gym_config.ScalarField(name="RR_x", upper_bound=foot_x_bound[0],
                                          lower_bound=foot_x_bound[1]),
        locomotion_gym_config.ScalarField(name="RR_y", upper_bound=foot_y_bound[0],
                                          lower_bound=foot_y_bound[1]),
        locomotion_gym_config.ScalarField(name="RR_z", upper_bound=foot_z_bound[0],
                                          lower_bound=foot_z_bound[1]),
        locomotion_gym_config.ScalarField(name="RL_x", upper_bound=foot_x_bound[0],
                                          lower_bound=foot_x_bound[1]),
        locomotion_gym_config.ScalarField(name="RL_y", upper_bound=foot_y_bound[0],
                                          lower_bound=foot_y_bound[1]),
        locomotion_gym_config.ScalarField(name="RL_z", upper_bound=foot_z_bound[0],
                                          lower_bound=foot_z_bound[1]),

        # locomotion_gym_config.ScalarField(name="FF_q_hip", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="FF_q_thigh", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="FF_q_calf", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        #
        # locomotion_gym_config.ScalarField(name="FL_q_hip", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="FL_q_thigh", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="FL_q_calf", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        #
        # locomotion_gym_config.ScalarField(name="RR_q_hip", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="RR_q_thigh", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="RR_q_calf", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        #
        # locomotion_gym_config.ScalarField(name="RL_q_hip", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="RL_q_thigh", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="RL_q_calf", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),

        # locomotion_gym_config.ScalarField(name="FF_x_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="FF_y_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="FF_z_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        #
        # locomotion_gym_config.ScalarField(name="FL_x_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="FL_y_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="FL_z_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        #
        # locomotion_gym_config.ScalarField(name="RR_x_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="RR_y_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="RR_z_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        #
        # locomotion_gym_config.ScalarField(name="RL_x_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="RL_y_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="RL_z_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),

    ]

    def __init__(self,
                 pybullet_client,
                 urdf_filename=URDF_FILENAME,
                 enable_clip_motor_commands=True,
                 time_step=0.0025,
                 action_repeat=8,  # 45
                 sensors=None,
                 control_latency=0.002,
                 on_rack=False,
                 enable_action_interpolation=True,
                 enable_action_filter=True,
                 height_map=None,
                 terrain_scale=0.1,
                 physical_parameters=None,
                 max_step_num=None,
                 FTG=None,
                 desired_yaw=None,
                 desired_turning=None
                 ):
        self.height_map = height_map
        self.terrain_scale = terrain_scale
        self._urdf_filename = urdf_filename
        self._enable_clip_motor_commands = enable_clip_motor_commands
        self.all_contacts = None

        self.JointPositionErrorHistory_real = np.array([0.] * 24)
        self.JointVelocityHistory_real = np.array([0.] * 24)
        self.FootTargetHistory_real = np.array([0.16840759, -0.13204999, -0.32405686,
                                                0.16840759, 0.13204999, -0.32405686,
                                                -0.19759241, -0.13204999, -0.32405686,
                                                -0.19759241, 0.13204999, -0.32405686] * 2)

        motor_kp = [ABDUCTION_P_GAIN, HIP_P_GAIN, KNEE_P_GAIN,
                    ABDUCTION_P_GAIN, HIP_P_GAIN, KNEE_P_GAIN,
                    ABDUCTION_P_GAIN, HIP_P_GAIN, KNEE_P_GAIN,
                    ABDUCTION_P_GAIN, HIP_P_GAIN, KNEE_P_GAIN]
        motor_kd = [ABDUCTION_D_GAIN, HIP_D_GAIN, KNEE_D_GAIN,
                    ABDUCTION_D_GAIN, HIP_D_GAIN, KNEE_D_GAIN,
                    ABDUCTION_D_GAIN, HIP_D_GAIN, KNEE_D_GAIN,
                    ABDUCTION_D_GAIN, HIP_D_GAIN, KNEE_D_GAIN]

        # motor_torque_limits = None  # jp hack
        super(A1, self).__init__(
            pybullet_client=pybullet_client,
            time_step=time_step,
            action_repeat=action_repeat,
            num_motors=NUM_MOTORS,
            dofs_per_leg=DOFS_PER_LEG,
            motor_direction=JOINT_DIRECTIONS,
            motor_offset=JOINT_OFFSETS,
            motor_overheat_protection=False,
            motor_model_class=a1_motor.A1MotorModel,
            motor_control_mode=robot_config.MotorControlMode.POSITION,
            sensors=sensors,
            motor_kp=motor_kp,
            motor_kd=motor_kd,
            motor_torque_limits=motor_torque_limits,
            control_latency=control_latency,
            observation_noise_stdev=SENSOR_NOISE_STDDEV,
            on_rack=on_rack,
            enable_action_interpolation=enable_action_interpolation,
            enable_action_filter=enable_action_filter,
            default_torque=default_torque,
            default_force=default_force,
            physical_parameters=physical_parameters,
            max_step_num=max_step_num,
            FTG=FTG,
            desired_yaw=desired_yaw,
            desired_turning=desired_turning
        )

        return

    def _LoadRobotURDF(self):
        a1_urdf_path = self.GetURDFFile()
        # print(aliengo_urdf_path)
        # input()

        if self._self_collision_enabled:
            self.quadruped = self._pybullet_client.loadURDF(
                a1_urdf_path,
                self._GetDefaultInitPosition(),
                self._GetDefaultInitOrientation(),
                flags=self._pybullet_client.URDF_USE_SELF_COLLISION)
        else:
            self.quadruped = self._pybullet_client.loadURDF(
                a1_urdf_path, self._GetDefaultInitPosition(),
                self._GetDefaultInitOrientation())
        # print(111)
        # input()

    def _SettleDownForReset(self, default_motor_angles, reset_time):
        self.ReceiveObservation()

        if reset_time <= 0:
            return

        for _ in range(500):
            self._StepInternal(
                INIT_MOTOR_ANGLES,
                motor_control_mode=robot_config.MotorControlMode.POSITION)
        if default_motor_angles is not None:
            num_steps_to_reset = int(reset_time / self.time_step)
            for _ in range(num_steps_to_reset):
                self._StepInternal(
                    default_motor_angles,
                    motor_control_mode=robot_config.MotorControlMode.POSITION)

    def GetHipPositionsInBaseFrame(self):
        return _DEFAULT_HIP_POSITIONS

    def GetFootContacts(self):
        self.all_contacts = self._pybullet_client.getContactPoints(bodyA=self.quadruped)

        foot_id = []
        for leg_idx in range(4):
            foot_id.append(self._foot_link_ids[leg_idx * 2 + 1])

        contacts = [False, False, False, False]
        for contact in self.all_contacts:
            # Ignore self contacts
            if contact[_BODY_B_FIELD_NUMBER] == self.quadruped:
                continue
            try:
                toe_link_index = foot_id.index(
                    contact[_LINK_A_FIELD_NUMBER])
                contacts[toe_link_index] = True
            except ValueError:
                continue

        return contacts

    def GetA1legsContacts(self):
        """Get A1's foot contact situation with the ground.

    Returns:
      A list of 4 booleans. The ith boolean is True if leg i is in contact with
      ground.
    """

        hip_contacts = []
        thigh_contacts = []
        shank_contacts = []
        foot_contacts = []

        for leg_idx in range(4):
            # hip_id = self._foot_link_ids[leg_idx * 2 - 2]
            # thigh_id = self._foot_link_ids[leg_idx * 2 - 1]
            # shank_id = self._foot_link_ids[leg_idx * 2]
            # foot_id = self._foot_link_ids[leg_idx * 2 + 1]
            hip_id = [3, 7, 11, 15][leg_idx]
            thigh_id = [4, 8, 12, 16][leg_idx]
            foot_id = [5, 9, 13, 17][leg_idx]
            # print("foot_id:-------",foot_id)
            hip_contacts.append(bool(
                self._pybullet_client.getContactPoints(
                    bodyA=0,
                    bodyB=self.quadruped,
                    linkIndexA=-1,
                    linkIndexB=hip_id)))
            thigh_contacts.append(bool(
                self._pybullet_client.getContactPoints(
                    bodyA=0,
                    bodyB=self.quadruped,
                    linkIndexA=-1,
                    linkIndexB=thigh_id)))
            foot_contacts.append(bool(
                self._pybullet_client.getContactPoints(
                    bodyA=0,
                    bodyB=self.quadruped,
                    linkIndexA=-1,
                    linkIndexB=foot_id)))
        leg_contact_dict = {
            "thigh_contacts": hip_contacts,
            "shank_contacts": thigh_contacts,
            "foot_contacts": foot_contacts
        }
        return leg_contact_dict

    def GetFootContactForces(self):
        foot_contact_forces = [0.,0.,0.,0.]
        for leg_idx in range(4):
            foot_id = self._foot_link_ids[leg_idx * 2 + 1]
            contact_list = self._pybullet_client.getContactPoints(
                    bodyA=0,
                    bodyB=self.quadruped,
                    linkIndexA=-1,
                    linkIndexB=foot_id)
            if len(contact_list) > 0:
                # print([contact_list[i][9] for i in range(len(contact_list))])
                for i in range(len(contact_list)):
                    if abs(contact_list[i][9]) > 0.:
                        foot_contact_forces[leg_idx] = contact_list[i][9]
                        break

        return foot_contact_forces

    def GetFootFriction(self):
        foot_lateral_friction = []
        for leg_idx in range(4):
            foot_id = self._foot_link_ids[leg_idx * 2 + 1]
            contact_info = self._pybullet_client.getDynamicsInfo(
                    bodyUniqueId=self.quadruped,
                    linkIndex=foot_id)[1]
            foot_lateral_friction.append(contact_info)
        return foot_lateral_friction

    def GetA1FootPositionsInWorldFrame(self):
        """Get the robot's foot position in the base frame."""
        # assert len(self._foot_link_ids) == self.num_legs
        a1_foot_link = [5, 9, 13, 17]
        assert len(a1_foot_link) == self.num_legs
        foot_positions = []
        for foot_id in a1_foot_link:
            link_state = self.pybullet_client.getLinkState(self.quadruped, foot_id)
            link_position = link_state[0][:-1]
            foot_positions.append(link_position)
            # foot_positions.append(
            #     kinematics.link_position_in_base_frame(
            #         robot=self,
            #         link_id=foot_id,
            #     ))
        return np.array(foot_positions)

    def GetXYZFootPositionsInWorldFrame(self):
        """Get the robot's foot position in the base frame."""
        # assert len(self._foot_link_ids) == self.num_legs
        a1_foot_link = [5, 9, 13, 17]
        assert len(a1_foot_link) == self.num_legs
        foot_positions = []
        for foot_id in a1_foot_link:
            link_state = self.pybullet_client.getLinkState(self.quadruped, foot_id)
            link_position = link_state[0]
            foot_positions.append(link_position)
            # foot_positions.append(
            #     kinematics.link_position_in_base_frame(
            #         robot=self,
            #         link_id=foot_id,
            #     ))
        return np.array(foot_positions)

    # def ComputeJacobian(self, leg_id):
    #     """Compute the Jacobian for a given leg."""
    #     # Because of the default rotation in the Laikago URDF, we need to reorder
    #     # the rows in the Jacobian matrix.
    #     return super(A1, self).ComputeJacobian(leg_id)[(2, 0, 1), :]

    def ResetPose(self, add_constraint):
        del add_constraint
        for name in self._joint_name_to_id:
            joint_id = self._joint_name_to_id[name]
            self._pybullet_client.setJointMotorControl2(
                bodyIndex=self.quadruped,
                jointIndex=(joint_id),
                controlMode=self._pybullet_client.VELOCITY_CONTROL,
                targetVelocity=0,
                force=0)
        for name, i in zip(MOTOR_NAMES, range(len(MOTOR_NAMES))):
            if "hip_joint" in name:
                angle = INIT_MOTOR_ANGLES[i] + HIP_JOINT_OFFSET
            elif "thigh_joint" in name:
                angle = INIT_MOTOR_ANGLES[i] + THIGH_JOINT_OFFSET
            elif "calf_joint" in name:
                angle = INIT_MOTOR_ANGLES[i] + CALF_JOINT_OFFSET
            else:
                raise ValueError("The name %s is not recognized as a motor joint." %
                                 name)
            self._pybullet_client.resetJointState(
                self.quadruped, self._joint_name_to_id[name], angle, targetVelocity=0)

        # ROLL = np.random.uniform(low=-np.pi, high=np.pi)
        # PITCH = np.random.uniform(low=-np.pi, high=np.pi)
        # YAW = np.random.uniform(low=-np.pi, high=np.pi)
        # # YAW = -np.pi
        # ROLL,PITCH = 0,0
        # orn = self._pybullet_client.getQuaternionFromEuler([ROLL,PITCH,YAW])
        # self._pybullet_client.resetBasePositionAndOrientation(
        #     self.quadruped, INIT_POSITION, orn)


    def GetURDFFile(self):
        return self._urdf_filename

    def _BuildUrdfIds(self):
        """Build the link Ids from its name in the URDF file.

    Raises:
      ValueError: Unknown category of the joint name.
    """
        num_joints = self._pybullet_client.getNumJoints(self.quadruped)
        self._chassis_link_ids = [-1]
        self._leg_link_ids = []
        self._motor_link_ids = []
        self._knee_link_ids = []
        self._foot_link_ids = []

        for i in range(num_joints):
            joint_info = self._pybullet_client.getJointInfo(self.quadruped, i)
            joint_name = joint_info[1].decode("UTF-8")
            # print("------------------------------------")
            # print(joint_name)

            joint_id = self._joint_name_to_id[joint_name]
            if _HIP_NAME_PATTERN.match(joint_name):
                self._chassis_link_ids.append(joint_id)
            elif _THIGH_NAME_PATTERN.match(joint_name):
                self._motor_link_ids.append(joint_id)
            # We either treat the lower leg or the toe as the foot link, depending on
            # the urdf version used.
            elif _CALF_NAME_PATTERN.match(joint_name):
                self._knee_link_ids.append(joint_id)
            elif _FOOT_NAME_PATTERN.match(joint_name):
                self._foot_link_ids.append(joint_id)
            # else:
            # raise ValueError("Unknown category of joint %s" % joint_name)
        # input()

        self._leg_link_ids.extend(self._knee_link_ids)
        self._leg_link_ids.extend(self._foot_link_ids)
        self._foot_link_ids.extend(self._knee_link_ids)

        self._chassis_link_ids.sort()
        self._motor_link_ids.sort()
        self._foot_link_ids.sort()
        self._leg_link_ids.sort()

        return

    def _GetMotorNames(self):
        return MOTOR_NAMES

    def _GetDefaultInitPosition(self):
        if self._on_rack:
            return INIT_RACK_POSITION
        else:
            return INIT_POSITION

    def _GetDefaultInitOrientation(self):
        # The Laikago URDF assumes the initial pose of heading towards z axis,
        # and belly towards y axis. The following transformation is to transform
        # the Laikago initial orientation to our commonly used orientation: heading
        # towards -x direction, and z axis is the up direction.
        # init_orientation = pyb.getQuaternionFromEuler([math.pi / 2.0, 0, math.pi / 2.0])
        # init_orientation = pyb.getQuaternionFromEuler([0, 0, 0])
        # return init_orientation
        return INIT_ORIENTATION

    def GetDefaultInitPosition(self):
        """Get default initial base position."""
        return self._GetDefaultInitPosition()

    def GetDefaultInitOrientation(self):
        """Get default initial base orientation."""
        return self._GetDefaultInitOrientation()

    def GetDefaultInitJointPose(self):
        """Get default initial joint pose."""
        joint_pose = (INIT_MOTOR_ANGLES + JOINT_OFFSETS) * JOINT_DIRECTIONS
        return joint_pose

    def ApplyAction(self, motor_commands, motor_control_mode=None):
        """Clips and then apply the motor commands using the motor model.

    Args:
      motor_commands: np.array. Can be motor angles, torques, hybrid commands,
        or motor pwms (for Minitaur only).N
      motor_control_mode: A MotorControlMode enum.
    """
        if self._enable_clip_motor_commands:
            motor_commands = self._ClipMotorCommands(motor_commands)

        super(A1, self).ApplyAction(motor_commands, motor_control_mode)
        return

    def _ClipMotorCommands(self, motor_commands):
        """Clips motor commands.

    Args:
      motor_commands: np.array. Can be motor angles, torques, hybrid commands,
        or motor pwms (for Minitaur only).

    Returns:
      Clipped motor commands.
    """

        # clamp the motor command by the joint limit, in case weired things happens
        max_angle_change = MAX_MOTOR_ANGLE_CHANGE_PER_STEP
        current_motor_angles = self.GetMotorAngles()
        motor_commands = np.clip(motor_commands,
                                 current_motor_angles - max_angle_change,
                                 current_motor_angles + max_angle_change)
        return motor_commands

    @classmethod
    def GetConstants(cls):
        del cls
        return a1_constants

    def GetFootPositionsInBaseFrame(self):
        """Get the robot's foot position in the base frame."""
        motor_angles = self.GetMotorAngles()
        return foot_positions_in_base_frame(motor_angles)

    def ComputeJacobian(self, leg_id):
        """Compute the Jacobian for a given leg."""
        # Does not work for Minitaur which has the four bar mechanism for now.
        motor_angles = self.GetMotorAngles()[leg_id * 3:(leg_id + 1) * 3]
        return analytical_leg_jacobian(motor_angles, leg_id)

    def height_map_converter(self, height_map, scale, x, y):
        if x != x:
            self.is_safe = False
        else:
            # print(np.shape(height_map))
            map_x = int((x * scale) + height_map.shape[0] / 2)
            map_y = int((y * scale) + height_map.shape[1] / 2)
            if map_x < 0 or map_x >= height_map.shape[0] or map_y < 0 or map_y >= height_map.shape[1]:
                map_x = map_y = 0
                self.is_safe = False

            return map_x, map_y

    def getHeightMap(self):
        foot_surrounding_height = np.array([0.]*36)
        for ind, f in enumerate(self.GetA1FootPositionsInWorldFrame()):
            foot_center_x, foot_center_y = f
            foothlod_height = self.height_map[self.height_map_converter(self.height_map,
                                                                               1 / self.terrain_scale,
                                                                               foot_center_x,
                                                                               foot_center_y)]
            gap_angle = 2 * np.pi / 9
            for i in range(9):
                i_x = foot_center_x + np.sin(gap_angle * i) * 0.1
                i_y = foot_center_y + np.cos(gap_angle * i) * 0.1
                scanner_target = self.height_map_converter(self.height_map, 1 / self.terrain_scale, i_x, i_y)
                fsh = self.height_map[scanner_target]
                foot_surrounding_height[ind * 9 + i] = fsh - foothlod_height
        return foot_surrounding_height

    def get_obs(self):
        self.JointPositionErrorHistory_real = np.squeeze(np.append(self.GetJointPositionErrorHistory(),
                                                                   self.JointPositionErrorHistory_real,
                                                        )[:24])
        # print( self.JointPositionErrorHistory_real)
        self.JointVelocityHistory_real = np.squeeze(np.append(self.GetJointVelocityHistory(),
                                                              self.JointVelocityHistory_real,
                                                   )[:24])

        self.FootTargetHistory_real = np.squeeze(np.append( self.GetFootTargetHistory(),
                                                            self.FootTargetHistory_real,
                                               )[:24])
        # self.TorqueTargetHiory_real = np.append(self.TorqueTargetHistory_real,
        #                                           self.GetTorqueTargetHistory())[12:]
        # print(self.GetFTGPhases())

        obs = np.concatenate((
            # self.getHeightMap(),
            self.GetDesiredDirection(),
            # self.GetDesiredTurningDirection(),
            self.GetBaseRollPitchYaw(),
            self.GetBaseRollPitchYawRate(),
            list(self.GetBaseVelocity()[0]),
            self.GetMotorAngles(),
            self.GetMotorVelocities(),
            self.GetFTGPhases(),
            self.GetFTGFrequencies(),
            self.GetBaseFrequency(),
            self.JointPositionErrorHistory_real,
            self.JointVelocityHistory_real,
            self.FootTargetHistory_real,
            # self.TorqueTargetHistory_real
        ))
        # print("heightmap:", self.getHeightMap())
        # print("DesiredDirection", self.GetDesiredDirection())
        # print("DesiredTurningDirection", self.GetDesiredTurningDirection())
        # print("BaseRollPitchYaw", self.GetBaseRollPitchYaw())
        #
        # print("BaseRollPitchYawRate", self.GetBaseRollPitchYawRate())
        # print("BaseVelocity",  list(self.GetBaseVelocity()[0]))
        # print("MotorAngles", self.GetMotorAngles())
        # print("MotorVelocities", self.GetMotorVelocities())
        #
        # print("FTGPhases", self.GetFTGPhases())
        # print("FTGFrequencies", self.GetFTGFrequencies())
        # print("BaseFrequency", self.GetBaseFrequency())
        # print("JointPositionErrorHistory_real", self.JointPositionErrorHistory_real)
        # print("JointVelocityHistory_real", self.JointVelocityHistory_real)
        # print("FootTargetHistory_real", self.FootTargetHistory_real)
        # print("TorqueTargetHistory_real", self.TorqueTargetHistory_real)
        return obs


if __name__ == "__main__":
    import pybullet_utils.bullet_client as bc
    import pybullet as p
    import time
    # p.connect(p.GUI)
    # p.loadURDF("/home/amax/Westlake/aliengo_py/envs/aliengo_model/plane.urdf")
    pybullet_client = bc.BulletClient(p.GUI)
    robot = A1(pybullet_client)
    quadruped = robot.quadruped
    p = pybullet_client

    # enable collision between lower legs
    for j in range(p.getNumJoints(quadruped)):
        print(p.getJointInfo(quadruped, j))

    lower_legs = [2, 5, 8, 11]
    for l0 in lower_legs:
        for l1 in lower_legs:
            if (l1 > l0):
                enableCollision = 1
                print("collision for pair", l0, l1, p.getJointInfo(quadruped, l0)[12],
                      p.getJointInfo(quadruped, l1)[12],
                      "enabled=", enableCollision)
                p.setCollisionFilterPair(quadruped, quadruped, 2, 5, enableCollision)

    jointIds = []
    paramIds = []

    maxForceId = p.addUserDebugParameter("maxForce", 0, 100, 20)

    for j in range(p.getNumJoints(quadruped)):
        p.changeDynamics(quadruped, j, linearDamping=0, angularDamping=0)
        info = p.getJointInfo(quadruped, j)
        # print(info)
        jointName = info[1]
        jointType = info[2]
        if (jointType == p.JOINT_PRISMATIC or jointType == p.JOINT_REVOLUTE):
            jointIds.append(j)

    # print(jointIds)

    p.getCameraImage(480, 320)
    p.setRealTimeSimulation(0)

    joints = []

    while (1):
        with open("mocap.txt", "r") as filestream:
            for line in filestream:
                maxForce = p.readUserDebugParameter(maxForceId)
                currentline = line.split(",")
                frame = currentline[0]
                t = currentline[1]
                joints = currentline[2:14]
                for j in range(12):
                    targetPos = float(joints[j])
                    p.setJointMotorControl2(quadruped, jointIds[j], p.POSITION_CONTROL, targetPos, force=maxForce)

                p.stepSimulation()
                for lower_leg in lower_legs:
                    # print("points for ", quadruped, " link: ", lower_leg)
                    pts = p.getContactPoints(quadruped, -1, lower_leg)
                    # print("num points=",len(pts))
                    # for pt in pts:
                    #    print(pt[9])
                time.sleep(1. / 500.)

    # for j in range (p.getNumJoints(quadruped)):
    #     p.changeDynamics(quadruped,j,linearDamping=0, angularDamping=0)
    #     info = p.getJointInfo(quadruped,j)
    #     js = p.getJointState(quadruped,j)
    #     #print(info)
    #     jointName = info[1]
    #     jointType = info[2]
    #     if (jointType==p.JOINT_PRISMATIC or jointType==p.JOINT_REVOLUTE):
    #             paramIds.append(p.addUserDebugParameter(jointName.decode("utf-8"),-4,4,(js[0]-jointOffsets[j])/jointDirections[j]))

    # p.setRealTimeSimulation(1)

    # while (1):
    #     for i in range(len(paramIds)):
    #         c = paramIds[i]
    #         targetPos = p.readUserDebugParameter(c)
    #         maxForce = p.readUserDebugParameter(maxForceId)
    #         p.setJointMotorControl2(quadruped,jointIds[i],p.POSITION_CONTROL,jointDirections[i]*targetPos+jointOffsets[i], force=maxForce)
