# coding=utf-8
# Copyright 2020 The Google Research Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# pytype: disable=attribute-error
"""Real robot interface of A1 robot."""

import os
import inspect

currentdir = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
parentdir = os.path.dirname(os.path.dirname(currentdir))
os.sys.path.insert(0, parentdir)

from absl import logging
import math
import re
import numpy as np
import time

from envs.robots import a1_pose_utils
from envs.robots import a1
from real_uti import a1_robot_velocity_estimator
from envs.robots import minitaur
from motion_imitation.robots import robot_config
from envs.build_envs import locomotion_gym_config
# from robot_interface import RobotInterface  # pytype: disable=import-error
# import lcm
# from pybullet_vison_mapping.exlcm import aroundfeet_map_t
import select
import numpy as np

NUM_MOTORS = 12
NUM_LEGS = 4
MOTOR_NAMES = [
    "FR_hip_joint",
    "FR_upper_joint",
    "FR_lower_joint",
    "FL_hip_joint",
    "FL_upper_joint",
    "FL_lower_joint",
    "RR_hip_joint",
    "RR_upper_joint",
    "RR_lower_joint",
    "RL_hip_joint",
    "RL_upper_joint",
    "RL_lower_joint",
]
INIT_RACK_POSITION = [0, 0, 1]
INIT_POSITION = [0, 0, 0.345]
# JOINT_DIRECTIONS = np.ones(12)
JOINT_DIRECTIONS = np.array([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1])

HIP_JOINT_OFFSET = 0.0
UPPER_LEG_JOINT_OFFSET = 0.0
KNEE_JOINT_OFFSET = 0.0

DOFS_PER_LEG = 3

JOINT_OFFSETS = np.array(
    [HIP_JOINT_OFFSET, UPPER_LEG_JOINT_OFFSET, KNEE_JOINT_OFFSET] * 4)
PI = math.pi

MAX_MOTOR_ANGLE_CHANGE_PER_STEP = 0.2

_DEFAULT_HIP_POSITIONS = (
    (0.183, -0.047, 0),
    (0.183, 0.047, 0),
    (-0.183, -0.047, 0),
    (-0.183, 0.047, 0),
)

ABDUCTION_P_GAIN = 120  # 50
ABDUCTION_D_GAIN = 1
HIP_P_GAIN = 120  # 150
HIP_D_GAIN = 1
KNEE_P_GAIN = 120
KNEE_D_GAIN = 1
motor_torque_limits = 30

COMMAND_CHANNEL_NAME = 'LCM_Low_Cmd'
STATE_CHANNEL_NAME = 'LCM_Low_State'

# Bases on the readings from A1's default pose.
INIT_MOTOR_ANGLES = np.array([
                                 a1_pose_utils.A1_DEFAULT_ABDUCTION_ANGLE,
                                 a1_pose_utils.A1_DEFAULT_HIP_ANGLE,
                                 a1_pose_utils.A1_DEFAULT_KNEE_ANGLE
                             ] * NUM_LEGS)
ROOT_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
URDF_FILENAME = ROOT_DIR + "/a1/urdf/a1.urdf"

_BODY_B_FIELD_NUMBER = 2
_LINK_A_FIELD_NUMBER = 3

# UPPER_BOUND_fre, LOWER_BOUND_fre = 0.1,-0.1
# torque_max, torque_min = 5, -5
# torque_z_max, torque_z_min = 10, -10
# horizon frame
L_half = 0.183
W_half = 0.047 + 0.08505
UPPER_BOUND, LOWER_BOUND = [0.3 - L_half, -0.05 + W_half, -0.15,
                            0.3 - L_half, 0.2 - W_half, -0.15,
                            -0.1 + L_half, -0.1 + W_half, -0.15,
                            -0.1 + L_half, 0.2 - W_half, -0.15], \
                           [0.05 - L_half, -0.2 + W_half, -0.35,
                            0.05 - L_half, 0.05 - W_half, -0.35,
                            -0.25 + L_half, -0.2 + W_half, -0.35,
                            -0.25 + L_half, 0.1 - W_half, -0.35]
# a1_Hip_max = 1.047
# a1_Hip_min = -0.873
# a1_Thigh_max = 3.927
# a1_Thigh_min = -0.524
# a1_Calf_max = -0.611
# a1_Calf_min = -2.775
#
# foot_x_bound = [0.08, -0.08]
# foot_y_bound = [0.04, -0.04]
# foot_z_bound = [0.04, -0.]
#
# torque_max, torque_min = 2,-2
a1_Hip_max = 0.25
a1_Hip_min = -0.25
a1_Thigh_max = 0.5
a1_Thigh_min = -0.5
a1_Calf_max = 0.5
a1_Calf_min = -0.5

a1_action_max = np.array(1) * np.array([a1_Hip_max, a1_Thigh_max, a1_Calf_max])
a1_action_min = np.array(1) * np.array([a1_Hip_min, a1_Thigh_min, a1_Calf_min])

UPPER_BOUND_fre, LOWER_BOUND_fre = 0.1, -0.1
foot_x_bound = [0.1, -0.1]
foot_y_bound = [0.06, -0.06]
foot_z_bound = [0.04, -0.]


class A1Robot(a1.A1):
    """Interface for real A1 robot."""

    ACTION_CONFIG = [
        locomotion_gym_config.ScalarField(name="frequency_0", upper_bound=UPPER_BOUND_fre,
                                          lower_bound=LOWER_BOUND_fre),
        locomotion_gym_config.ScalarField(name="frequency_1", upper_bound=UPPER_BOUND_fre,
                                          lower_bound=LOWER_BOUND_fre),
        locomotion_gym_config.ScalarField(name="frequency_2", upper_bound=UPPER_BOUND_fre,
                                          lower_bound=LOWER_BOUND_fre),
        locomotion_gym_config.ScalarField(name="frequency_3", upper_bound=UPPER_BOUND_fre,
                                          lower_bound=LOWER_BOUND_fre),

        locomotion_gym_config.ScalarField(name="FF_x", upper_bound=foot_x_bound[0],
                                          lower_bound=foot_x_bound[1]),
        locomotion_gym_config.ScalarField(name="FF_y", upper_bound=foot_y_bound[0],
                                          lower_bound=foot_y_bound[1]),
        locomotion_gym_config.ScalarField(name="FF_z", upper_bound=foot_z_bound[0],
                                          lower_bound=foot_z_bound[1]),
        locomotion_gym_config.ScalarField(name="FL_x", upper_bound=foot_x_bound[0],
                                          lower_bound=foot_x_bound[1]),
        locomotion_gym_config.ScalarField(name="FL_y", upper_bound=foot_y_bound[0],
                                          lower_bound=foot_y_bound[1]),
        locomotion_gym_config.ScalarField(name="FL_z", upper_bound=foot_z_bound[0],
                                          lower_bound=foot_z_bound[1]),
        locomotion_gym_config.ScalarField(name="RR_x", upper_bound=foot_x_bound[0],
                                          lower_bound=foot_x_bound[1]),
        locomotion_gym_config.ScalarField(name="RR_y", upper_bound=foot_y_bound[0],
                                          lower_bound=foot_y_bound[1]),
        locomotion_gym_config.ScalarField(name="RR_z", upper_bound=foot_z_bound[0],
                                          lower_bound=foot_z_bound[1]),
        locomotion_gym_config.ScalarField(name="RL_x", upper_bound=foot_x_bound[0],
                                          lower_bound=foot_x_bound[1]),
        locomotion_gym_config.ScalarField(name="RL_y", upper_bound=foot_y_bound[0],
                                          lower_bound=foot_y_bound[1]),
        locomotion_gym_config.ScalarField(name="RL_z", upper_bound=foot_z_bound[0],
                                          lower_bound=foot_z_bound[1]),

        # locomotion_gym_config.ScalarField(name="FF_q_hip", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="FF_q_thigh", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="FF_q_calf", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        #
        # locomotion_gym_config.ScalarField(name="FL_q_hip", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="FL_q_thigh", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="FL_q_calf", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        #
        # locomotion_gym_config.ScalarField(name="RR_q_hip", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="RR_q_thigh", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="RR_q_calf", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        #
        # locomotion_gym_config.ScalarField(name="RL_q_hip", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="RL_q_thigh", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),
        # locomotion_gym_config.ScalarField(name="RL_q_calf", upper_bound=q_dot_desired_bound[0],
        #                                   lower_bound=q_dot_desired_bound[1]),

        # locomotion_gym_config.ScalarField(name="FF_x_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="FF_y_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="FF_z_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        #
        # locomotion_gym_config.ScalarField(name="FL_x_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="FL_y_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="FL_z_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        #
        # locomotion_gym_config.ScalarField(name="RR_x_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="RR_y_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="RR_z_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        #
        # locomotion_gym_config.ScalarField(name="RL_x_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="RL_y_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),
        # locomotion_gym_config.ScalarField(name="RL_z_torque", upper_bound=torque_max,
        #                                   lower_bound=torque_min),

    ]

    def __init__(self, pybullet_client, time_step=0.001, motor_kps=None, motor_kds=None, **kwargs):
        """Initializes the robot class."""
        # Initialize pd gain vector
        # self.motor_kps = np.array([ABDUCTION_P_GAIN, HIP_P_GAIN, KNEE_P_GAIN] * 4)
        # self.motor_kds = np.array([ABDUCTION_D_GAIN, HIP_D_GAIN, KNEE_D_GAIN] * 4)
        self.motor_kps = motor_kps
        self.motor_kds = motor_kds
        self._pybullet_client = pybullet_client
        self.time_step = time_step
        self._motor_control_mode = robot_config.MotorControlMode.POSITION

        # Robot state variables
        self._init_complete = True
        self._base_orientation = None
        self._raw_state = None
        self._last_raw_state = None
        self._motor_angles = np.zeros(12)
        self._motor_velocities = np.zeros(12)
        self._joint_states = None
        self._last_reset_time = time.time()
        self._velocity_estimator = a1_robot_velocity_estimator.VelocityEstimator(
            self)

        self.action_list = np.zeros(12)

        self.JointPositionErrorHistory_real = np.array([0.] * 24)
        self.JointVelocityHistory_real = np.array([0.] * 24)

        self.FootTargetHistory_real = np.array([0.16840759, -0.13204999, -0.32405686,
                                                0.16840759, 0.13204999, -0.32405686,
                                                -0.19759241, -0.13204999, -0.32405686,
                                                -0.19759241, 0.13204999, -0.32405686] * 2)
        # self.TorqueTargetHistory_real = np.array([0.] * 24)

        # Initiate UDP for robot state and actions
        self._robot_interface = RobotInterface()
        self._robot_interface.send_command(np.zeros(60, dtype=np.float32))

        kwargs['on_rack'] = True
        super(A1Robot, self).__init__(pybullet_client,
                                      time_step=time_step,
                                      # motor_control_mode=robot_config.MotorControlMode.HYBRID,
                                      **kwargs)

    def ReceiveObservation(self):
        """Receives observation from robot.

    Synchronous ReceiveObservation is not supported in A1,
    so changging it to noop instead.
    """
        state = self._robot_interface.receive_observation()
        self._raw_state = state
        # Convert quaternion from wxyz to xyzw, which is default for Pybullet.
        q = state.imu.quaternion
        self._base_orientation = np.array([q[1], q[2], q[3], q[0]])
        self._motor_angles = np.array([motor.q for motor in state.motorState[:12]])
        self._motor_velocities = np.array(
            [motor.dq for motor in state.motorState[:12]])
        self._joint_states = np.array(
            list(zip(self._motor_angles, self._motor_velocities)))
        if self._init_complete:
            # self._SetRobotStateInSim(self._motor_angles, self._motor_velocities)
            self._velocity_estimator.update(self._raw_state)

        # self.JointPositionErrorHistory_real = np.tile(self.GetJointPositionErrorHistory(), 2)
        # self.JointVelocityHistory_real = np.tile(self.GetJointVelocityHistory(), 2)
        # self.FootTargetHistory_real = np.tile(self.GetFootTargetHistory(), 2)
        # self.TorqueTargetHistory_real = np.tile(self.GetTorqueTargetHistory(), 2)

        self.BaseRollPitchYawHistory = np.tile(self.GetIMU(), 3)
        self.LastActionHistory = np.tile(self.action_list, 3)
        self.JointPositionHistory = np.tile(self.GetMotorAngles(), 3)

        self._observation_history.appendleft(self.GetTrueObservation())

    def _SetRobotStateInSim(self, motor_angles, motor_velocities):
        self._pybullet_client.resetBasePositionAndOrientation(
            self.quadruped, self.GetBasePosition(), self.GetBaseOrientation())
        for i, motor_id in enumerate(self._motor_id_list):
            self._pybullet_client.resetJointState(self.quadruped, motor_id,
                                                  motor_angles[i],
                                                  motor_velocities[i])

    def GetTrueMotorAngles(self):
        return self._motor_angles.copy()

    def GetMotorAngles(self):
        return minitaur.MapToMinusPiToPi(self._motor_angles).copy()

    def GetMotorVelocities(self):
        return self._motor_velocities.copy()

    def GetBasePosition(self):
        return self._pybullet_client.getBasePositionAndOrientation(
            self.quadruped)[0]

    def GetBaseRollPitchYaw(self):
        return self._pybullet_client.getEulerFromQuaternion(self._base_orientation)

    def GetIMU(self):
        return np.concatenate((self.GetBaseRollPitchYaw(), self.GetBaseRollPitchYawRate()))

    def GetTrueBaseRollPitchYaw(self):
        return self._pybullet_client.getEulerFromQuaternion(self._base_orientation)

    def GetBaseRollPitchYawRate(self):
        return self.GetTrueBaseRollPitchYawRate()

    def GetTrueBaseRollPitchYawRate(self):
        return np.array(self._raw_state.imu.gyroscope).copy()

    def GetBaseVelocity(self):
        return self._velocity_estimator.estimated_velocity.copy()

    def GetFootContacts(self):
        return np.array(self._raw_state.footForce) > 20

    def GetTimeSinceReset(self):
        return time.time() - self._last_reset_time

    def GetBaseOrientation(self):
        return self._base_orientation.copy()

    @property
    def motor_velocities(self):
        return self._motor_velocities.copy()

    def ApplyAction(self, motor_commands, motor_control_mode=None):
        """Clips and then apply the motor commands using the motor model.

      Args:
        motor_commands: np.array. Can be motor angles, torques, hybrid commands,
          or motor pwms (for Minitaur only).
        motor_control_mode: A MotorControlMode enum.
      """
        if motor_control_mode is None:
            motor_control_mode = self._motor_control_mode
        motor_commands = self._ClipMotorCommands(motor_commands)

        command = np.zeros(60, dtype=np.float32)

        for motor_id in range(NUM_MOTORS):
            command[motor_id * 5] = motor_commands[motor_id]
            command[motor_id * 5 + 1] = self.motor_kps[motor_id]
            command[motor_id * 5 + 3] = self.motor_kds[motor_id]

        self._robot_interface.send_command(command)

    # def ApplyAction(self, motor_commands, motor_control_mode=None):
    #     """Clips and then apply the motor commands using the motor model.
    #
    # Args:
    #   motor_commands: np.array. Can be motor angles, torques, hybrid commands,
    #     or motor pwms (for Minitaur only).
    #   motor_control_mode: A MotorControlMode enum.
    # """
    #     # if motor_control_mode is None:
    #     #   motor_control_mode = self._motor_control_mode
    #     motor_commands = self._ClipMotorCommands(motor_commands)
    #
    #     self.action_list = motor_commands
    #
    #     motor_commands_HYBRID = np.array([0.] * self.num_motors * 5)
    #     qdot_compensation = np.array([0.] * 12)
    #     for i in range(self.num_motors):
    #         motor_commands_HYBRID[5 * i] = motor_commands[i]
    #         motor_commands_HYBRID[5 * i + 1] = self.motor_kps[i]  # self.GetMotorPositionGains()[i]
    #         motor_commands_HYBRID[5 * i + 2] = qdot_compensation[i]
    #         motor_commands_HYBRID[5 * i + 3] = self.motor_kds[i]  # self.GetMotorVelocityGains()[i]
    #         motor_commands_HYBRID[5 * i + 4] = 0.
    #     command = motor_commands_HYBRID
    #
    #     # motor_commands = np.asarray(motor_commands)
    #     # print("1111---->",motor_commands)
    #     # q, qdot = self._GetPDObservation()
    #     # qdot_true = self.GetTrueMotorVelocities()
    #     # actual_torque, _ = self._motor_model.convert_to_torque(
    #     #     motor_commands, q, qdot, qdot_true)  # motor_control_mode)
    #     #
    #     #
    #     # command = np.zeros(60, dtype=np.float32)
    #     # for motor_id in range(NUM_MOTORS):
    #     #     command[motor_id * 5 + 4] = actual_torque[motor_id]
    #     self._robot_interface.send_command(command)

    # def robot_stand_up(self):
    #     initial_motor_angle = np.array([0., 0.67, -1.25] * 4)
    #     for t in range(1000):
    #         blend_ratio = np.minimum(t / 700., 1)
    #         if t == 0:
    #             action = (1 - blend_ratio
    #                       ) * np.array([0, 1.4, -2.8] * 4) + blend_ratio * initial_motor_angle
    #         else:
    #             action = (1 - blend_ratio
    #                       ) * self.GetMotorAngles() + blend_ratio * initial_motor_angle
    #
    #         # for i in range(self._action_repeat):
    #         self.ApplyAction(action)
    #         time.sleep(0.001)

    def Reset(self, reload_urdf=True, default_motor_angles=None, reset_time=3.0):
        """Reset the robot to default motor angles."""
        super(A1Robot, self).Reset(reload_urdf=reload_urdf,
                                   default_motor_angles=default_motor_angles,
                                   reset_time=-1)
        logging.warning(
            "About to reset the robot, make sure the robot is hang-up.")

        if not default_motor_angles:
            default_motor_angles = a1.INIT_MOTOR_ANGLES

        # print(self.GetMotorAngles())
        # print("--------")
        # [-0.24320337  1.12962484 - 2.7052424   0.31067324  1.10164237 - 2.69630837
        #  - 0.26288384  1.11871004 - 2.71076298  0.30005339  1.17079806 - 2.72479653]
        # print("default_motor_angles",default_motor_angles)
        current_motor_angles = self.GetMotorAngles()

        # Stand up in 1.5 seconds, and keep the behavior in this way.
        # standup_time = min(reset_time, 1.5)
        # for t in np.arange(0, reset_time, self.time_step * self._action_repeat):
        #   blend_ratio = min(t / standup_time, 1)
        #   action = blend_ratio * default_motor_angles + (
        #       1 - blend_ratio) * current_motor_angles
        #   self.Step(action) #, robot_config.MotorControlMode.POSITION)
        #   time.sleep(self.time_step * self._action_repeat)

        # standup_time = 3
        # for t in np.arange(0, standup_time, self.time_step * self._action_repeat):
        #     print(111111111)
        #     blend_ratio = min(t / standup_time, 1)
        #     action = blend_ratio * default_motor_angles + (
        #       1 - blend_ratio) * current_motor_angles
        #     # self.Step(action) #, robot_config.MotorControlMode.POSITION)
        #     for i in range(self._action_repeat):
        #       self.ApplyAction(action)
        #
        #     time.sleep(self.time_step * self._action_repeat)
        # Stand up in 1.5 seconds, and keep the behavior in this way.
        standup_time = 1.5
        for t in np.arange(0, 1.5, self.time_step * self._action_repeat):
            # print(self.GetMotorAngles())
            blend_ratio = min(t / standup_time, 1)
            action = blend_ratio * default_motor_angles + (
                    1 - blend_ratio) * current_motor_angles
            self._Step(action)
            time.sleep(self.time_step * self._action_repeat)

        # print(self.GetMotorAngles())

        if self._enable_action_filter:
            self._ResetActionFilter()

        self._velocity_estimator.reset()
        self._state_action_counter = 0
        self._step_counter = 0
        self._last_reset_time = time.time()

    def _Step(self, action):
        """Steps simulation."""
        # if self._enable_action_filter:
        #     action = self._FilterAction(action)

        for i in range(self._action_repeat):
            proc_action = self.ProcessAction(action, i)
            self._StepInternal(proc_action, self._motor_control_mode)
            self._step_counter += 1

        self._last_action = action

    def Terminate(self):
        self._is_alive = False

    # def getHeightMap(self):
    #     timeout = 0.0001  # amount of time to wait, in seconds
    #     while True:
    #         rfds, wfds, efds = select.select([lc1.fileno()], [], [], timeout)
    #         if rfds:
    #             lc1.handle()
    #         else:
    #             break
    #         #     print("Do something else...")
    #     print(aroundmap)
    #     return aroundmap

    def _StepInternal(self, action, motor_control_mode=None):
        self.ApplyAction(action, motor_control_mode)
        self.ReceiveObservation()
        self._state_action_counter += 1

    def get_obs(self):
        self.JointPositionErrorHistory_real = np.squeeze(np.append(self.GetJointPositionErrorHistory(),
                                                                   self.JointPositionErrorHistory_real,
                                                                   )[:24])
        # print( self.JointPositionErrorHistory_real)
        self.JointVelocityHistory_real = np.squeeze(np.append(self.GetJointVelocityHistory(),
                                                              self.JointVelocityHistory_real,
                                                              )[:24])

        self.FootTargetHistory_real = np.squeeze(np.append(self.GetFootTargetHistory(),
                                                           self.FootTargetHistory_real,
                                                           )[:24])
        # self.TorqueTargetHiory_real = np.append(self.TorqueTargetHistory_real,
        #                                           self.GetTorqueTargetHistory())[12:]
        # print(self.GetFTGPhases())

        obs = np.concatenate((
            # self.getHeightMap(),
            self.GetDesiredDirection(),
            # self.GetDesiredTurningDirection(),
            self.GetBaseRollPitchYaw(),
            self.GetBaseRollPitchYawRate(),
            self.GetBaseVelocity(),
            self.GetMotorAngles(),
            self.GetMotorVelocities(),
            self.GetFTGPhases(),
            self.GetFTGFrequencies(),
            self.GetBaseFrequency(),
            self.JointPositionErrorHistory_real,
            self.JointVelocityHistory_real,
            self.FootTargetHistory_real,
            # self.TorqueTargetHistory_real
        ))
        # print("heightmap:", self.getHeightMap())
        # print("DesiredDirection", self.GetDesiredDirection())
        # print("DesiredTurningDirection", self.GetDesiredTurningDirection())
        # print("BaseRollPitchYaw", self.GetBaseRollPitchYaw())
        #
        # print("BaseRollPitchYawRate", self.GetBaseRollPitchYawRate())
        # print("BaseVelocity",  list(self.GetBaseVelocity()[0]))
        # print("MotorAngles", self.GetMotorAngles())
        # print("MotorVelocities", self.GetMotorVelocities())
        #
        # print("FTGPhases", self.GetFTGPhases())
        # print("FTGFrequencies", self.GetFTGFrequencies())
        # print("BaseFrequency", self.GetBaseFrequency())
        # print("JointPositionErrorHistory_real", self.JointPositionErrorHistory_real)
        # print("JointVelocityHistory_real", self.JointVelocityHistory_real)
        # print("FootTargetHistory_real", self.FootTargetHistory_real)
        # print("TorqueTargetHistory_real", self.TorqueTargetHistory_real)
        return obs
