# Install packages if not already installed with:
#install.packages("tidyverse")
#install.packages("ggplot2")
#install.packages("dplyr")
#install.packages("reshape2")
#install.packages("glue")
#install.packages("extrafont")
#install.packages("scales")

library(tidyverse)
library(ggplot2)
library(dplyr)
library(reshape2)
library(glue)
library(scales)

# ******* Import is slow, only use for final graphs. *******
# Font code to use Libertine on Graphs
# Taken from: https://stackoverflow.com/questions/27689222/changing-fonts-for-graphs-in-r
# library(extrafont)
# font_import()
# loadfonts(device="win")       #Register fonts for Windows bitmap output
# **********************************************************


#******** Read In Files ********
TDPA_dat <- read.csv("C://Projects//aamas2024//TDP_Asymptotic_data.results", header=TRUE, comment.char = '#')
regular_dat <- read.csv("C://Projects//aamas2024//non_monotonic_data.results", header=TRUE, comment.char = '#')
regular_dat %>% na.omit() # Remove all rows with NAs
#*******************************


#******** Parameters ********
k_num <- 3
links_num <- 2
n_limit <- 14
experiment_binary_table_size <- 1000 # Used for TDP Asymptotic Procedure 2
#****************************


#******** Process Table Dynamic Program Asymptotic ********
TDPA_filtered <- filter(TDPA_dat, num_links == links_num & k == k_num & rand_range == 1 & n <= n_limit)
# Convert data into long form for charting
TDPA_long <- melt(TDPA_filtered, id.vars = c("algorithm","n","num_links","k","rand_range"))
#Compute Averages
TDPA_averages <- aggregate(TDPA_long$value, list(TDPA_long$algorithm, TDPA_long$rand_range, TDPA_long$n), FUN = mean)
colnames(TDPA_averages)[colnames(TDPA_averages) == "Group.1"] <- "algorithm"
colnames(TDPA_averages)[colnames(TDPA_averages) == "Group.2"] <- "rand_range"
colnames(TDPA_averages)[colnames(TDPA_averages) == "Group.3"] <- "n"
colnames(TDPA_averages)[colnames(TDPA_averages) == "x"] <- "average_time_raw"

TDPA_averages_procedure_1 <- TDPA_averages[which(TDPA_averages$algorithm == "TDP_Asymptotic_Procedure_1"),]
colnames(TDPA_averages_procedure_1)[colnames(TDPA_averages_procedure_1) == "average_time_raw"] <- "procedure_1_average_time"
TDPA_averages_procedure_1$algorithm <- NULL # We already know that this is TDP procedure 1

TDPA_averages_procedure_2 <- TDPA_averages[which(TDPA_averages$algorithm == "TDP_Asymptotic_Procedure_2"),]
colnames(TDPA_averages_procedure_2)[colnames(TDPA_averages_procedure_2) == "average_time_raw"] <- "procedure_2_average_time"
TDPA_averages_procedure_2$algorithm <- NULL # We already know that this is TDP procedure 2

TDPA_averages_list <- list(TDPA_averages_procedure_1, TDPA_averages_procedure_2)
TDPA_averages <- Reduce(function(x, y) merge(x, y, all=TRUE), TDPA_averages_list)

# (number of total configurations) * (procedure 1 time + (procedure 2 time * true_binary_table_size) / experiment_binary_table_size)
# w_max assumed to be n*q
TDPA_averages$w_max <- TDPA_averages$n * TDPA_averages$rand_range
# true_binary_table_size = (w_max + 1)^(km)
TDPA_averages$true_binary_table_size <- (TDPA_averages$w_max + 1)^(links_num*k_num)
TDPA_averages$intermediate1 <- TDPA_averages$procedure_2_average_time * (TDPA_averages$true_binary_table_size / experiment_binary_table_size)
TDPA_averages$intermediate2 <- TDPA_averages$procedure_1_average_time + TDPA_averages$intermediate1 
# total number of configurations = w_max^(km)
TDPA_averages$total_configurations <- TDPA_averages$w_max^(links_num*k_num)
TDPA_averages$average_time <- TDPA_averages$total_configurations * TDPA_averages$intermediate2

# Fill in algorithm column
TDPA_averages <- TDPA_averages %>% mutate(algorithm = "TDPA")
# Drop extra columns so that all dataframes have the same column names.
TDPA_averages$procedure_1_average_time <- NULL
TDPA_averages$procedure_2_average_time <- NULL
TDPA_averages$w_max <- NULL
TDPA_averages$true_binary_table_size <- NULL
TDPA_averages$intermediate1 <- NULL
TDPA_averages$intermediate2 <- NULL
TDPA_averages$total_configurations <- NULL
#************************************************


#******** Process Regular Run ********
regular_filtered <- filter(regular_dat, num_links == links_num & k == k_num)
# Remove whether NE was found from the data to just graph time.
regular_just_time <- regular_filtered[,c("algorithm","n","num_links","k","rand_range","trial_0_runtime","trial_1_runtime","trial_2_runtime","trial_3_runtime","trial_4_runtime","trial_5_runtime","trial_6_runtime","trial_7_runtime","trial_8_runtime","trial_9_runtime")]
# Convert data into long form for charting
regular_long <- melt(regular_just_time, id.vars = c("algorithm","n","num_links","k","rand_range"))
#Compute Averages
regular_averages <- aggregate(regular_long$value, list(regular_long$algorithm, regular_long$rand_range, regular_long$n), FUN = mean)
colnames(regular_averages)[colnames(regular_averages) == "Group.1"] <- "algorithm"
colnames(regular_averages)[colnames(regular_averages) == "Group.2"] <- "rand_range"
colnames(regular_averages)[colnames(regular_averages) == "Group.3"] <- "n"
colnames(regular_averages)[colnames(regular_averages) == "x"] <- "average_time"
#***************************************


#******** Merge Dataframes ********
averages <- rbind(regular_averages) 
#*********************************


#******** Plot Formatting ********
title_string <- glue('m = {links_num}, k = {k_num}')
# https://r-charts.com/colors/
algorithm_colours <- c("BF" = "black", "TDPA" = "red", "TDP" = "red", "SDP" = "blue")
# "solid", "longdash", "dashed", "dotted"
algorithm_linetypes <- c("BF" = "solid", "TDPA" = "dashed", "TDP" = "solid", "SDP" = "solid")
# https://ggplot2.tidyverse.org/reference/scale_shape.html
algorithm_shapes <- c("BF" = 5, "TDPA" = 0, "TDP" = 4, "SDP" = 2)
pdf_width <- 15
pdf_height <-10
#*********************************


#******** Plot ********
# Linux Libertine font is commented out below. Before uncommenting, be sure to correctly enable it (See top of file.).
chart <- ggplot(averages , aes(x = n, y = average_time)) +
  geom_point(aes(colour = algorithm, shape = algorithm), size = 2) +
  geom_line(aes(colour = algorithm, linetype = algorithm), size = 0.5) +
  theme_bw() +
  scale_color_manual(values = algorithm_colours) +
  scale_linetype_manual(values = algorithm_linetypes) +
  scale_shape_manual(values = algorithm_shapes) +
  labs(colour = "Algorithm", shape = "Algorithm", linetype = "Algorithm", y = "Running Time (Seconds)") +
  #theme(text=element_text(size=16,  family="Linux Libertine"), plot.title.position = "plot", plot.title = element_text(hjust = 0.5, size = 13), legend.title=element_text(size=13), axis.title=element_text(size=13)) +
  theme(plot.title.position = "plot", plot.title = element_text(hjust = 0.5, size = 13), legend.title=element_text(size=13), axis.title=element_text(size=13)) +
  scale_x_continuous(breaks=seq(2, 45, 8)) +
  scale_y_log10(breaks = trans_breaks("log10", function(x) 10^x), labels = trans_format("log10", math_format(10^.x)))
chart
#*********************

ggsave("regular.pdf", width = pdf_width, height = pdf_height, units = "cm")

#******** Merge Dataframes ********
averages <- rbind(regular_averages, TDPA_averages) #Create averages which will have all the merged data.
averages <- filter(averages, algorithm != "BF")
#**********************************

#******** Plot ********
# Linux Libertine font is commented out below. Before uncommenting, be sure to correctly enable it (See top of file.).
chart <- ggplot(averages , aes(x = n, y = average_time)) +
  geom_point(aes(colour = algorithm, shape = algorithm), size = 2) +
  geom_line(aes(colour = algorithm, linetype = algorithm), size = 0.5) +
  theme_bw() +
  scale_color_manual(values = algorithm_colours) +
  scale_linetype_manual(values = algorithm_linetypes) +
  scale_shape_manual(values = algorithm_shapes) +
  labs(colour = "Algorithm", shape = "Algorithm", linetype = "Algorithm", y = "Running Time (Seconds)") +
  #theme(text=element_text(size=16,  family="Linux Libertine"), plot.title.position = "plot", plot.title = element_text(hjust = 0.5, size = 13), legend.title=element_text(size=13), axis.title=element_text(size=13)) +
  theme(text=element_text(size=16,  family="Linux Libertine"), plot.title.position = "plot", plot.title = element_text(hjust = 0.5, size = 13), legend.title=element_text(size=13), axis.title=element_text(size=13)) +
  scale_x_continuous(breaks=seq(2, 45, 8)) +
  scale_y_log10(limits = c(1e-4, 1e6), breaks = trans_breaks("log10", function(x) 10^x), labels = trans_format("log10", math_format(10^.x)))
chart
#*********************

ggsave("asymptotic.pdf", width = pdf_width, height = pdf_height, units = "cm")
