import os
import re
import json
from typing import ClassVar, List, Dict, Any, Optional, Tuple
from metagpt.actions import Action
from metagpt.logs import logger
from utils.global_vars import PROMPT_BASE_DIR
from utils.global_vars import DATASET_NAME
from collections import OrderedDict


class A5_ReasoningJudge_New(Action):
    name: str = "A5_ReasoningJudge_New"

    async def run(
        self,
        inputs: Optional[Dict[str, Any]] = None,
        semiotic_square: dict = None,
        context: str = "",
        ablation_type: str = "full"
    ) -> dict:
        if inputs:
            ablation_type = inputs.get("ablation_type", ablation_type)
            semiotic_square = inputs.get("semiotic_square", semiotic_square)
            context = inputs.get("context", context)

        self.ablation_type = ablation_type

        def normalize_square(square):
            key_aliases = {"¬S1": "not_S1", "¬S2": "not_S2"}
            square = dict(square)
            for unicode_key, ascii_key in key_aliases.items():
                if unicode_key in square and ascii_key not in square:
                    square[ascii_key] = square[unicode_key]
            for k in {"S1", "S2", "not_S1", "not_S2"}:
                if k in square and not isinstance(square[k], dict):
                    square[k] = {"statement": square[k], "FOL": square[k]}
            return square

        semiotic_square = normalize_square(semiotic_square)
        s2_type = semiotic_square.get("S2_type", "").lower()
        trace = {}

        def final_output(verdict: str):
            return {"s1_truth": verdict, "reasoning_trace": trace}

        # Step 1: Run S1 and ¬S2
        trace["S1"] = await self._run_reasoning_pipeline("S1", context, semiotic_square)
        s1_verdict = trace["S1"]["verdict"]

        trace["not_S1"] = await self._run_reasoning_pipeline("not_S1", context, semiotic_square)
        not_s1_verdict = trace["not_S1"]["verdict"]

        return await self._verify_and_return(semiotic_square, trace, s2_type, s1_verdict, not_s1_verdict)

    async def _run_reasoning_pipeline(self, statement_key: str, context: str, square: dict) -> dict:
        statement = square.get(statement_key, {})

        # 根据 ablation_type 动态构造 target_statement
        if self.ablation_type == "woStatement":
            fol = statement.get("FOL", "")
            target_statement = f"FOL: {fol}"
        elif self.ablation_type == "woFOL":
            stmt_text = statement.get("statement", "")
            target_statement = f"Statement: {stmt_text}"
        else:  # full
            stmt_text = statement.get("statement", "")
            fol = statement.get("FOL", "")
            target_statement = f"Statement: {stmt_text}\nFOL: {fol}"

        # 🧠 Step 1: A4_ContextHandle —— 依据 ablation_type 选择对应的 prompt 路径
        if self.ablation_type == "woStatement":
            a4_path = os.path.join(PROMPT_BASE_DIR, "A4_ContextHandle_woStatement.txt")
        elif self.ablation_type == "woFOL":
            a4_path = os.path.join(PROMPT_BASE_DIR, "A4_ContextHandle_woFOL.txt")
        else:
            a4_path = os.path.join(PROMPT_BASE_DIR, "A4_ContextHandle.txt")

        try:
            with open(a4_path, "r", encoding="utf-8") as f:
                a4_prompt = f.read()
        except Exception as e:
            logger.error(f"[A4] Failed to load prompt from {a4_path}: {e}")
            return {"error": str(e), "verdict": "Uncertain"}


        # a4_filled = a4_prompt.replace("{context}", context.strip()) \
        #                     .replace("{target_statement}", target_statement)
        a4_filled = a4_prompt.replace("{context}", context.strip())
        a4_rsp = await self._log_and_ask(a4_filled)

        try:
            match = re.search(r"```json(.*?)```", a4_rsp, re.DOTALL)
            a4_json = json.loads(match.group(1).strip())
            premises = a4_json
        except Exception as e:
            logger.warning(f"[A5] A4_ContextHandle parsing failed: {e}")
            return {"verdict": "Uncertain", "steps": [], "error": f"A4_ContextHandle error: {e}"}

        # 🧠 Step 2: A5_Plan —— 推理计划
        # plan_path = os.path.join(PROMPT_BASE_DIR, "A5_Plan.txt")
        # with open(plan_path, "r", encoding="utf-8") as f:
        #     plan_prompt = f.read()
        # plan_filled = plan_prompt.replace("{target_statement}", target_statement) \
        #                         .replace("{premises}", self._format_premises(dataset_name=DATASET_NAME, ablation_type=self.ablation_type, premises=premises))
        # plan_rsp = await self._log_and_ask(plan_filled)

        # try:
        #     match = re.search(r"```json(.*?)```", plan_rsp, re.DOTALL)
        #     plan_json = json.loads(match.group(1).strip())
        #     plan_steps = plan_json.get("plan", [])
        # except Exception as e:
        #     logger.warning(f"[A5] Plan parsing failed: {e}")
        #     return {"verdict": "Uncertain", "steps": [], "error": f"Plan error: {e}"}

        # 🧠 Step 3: A5_Reasoning —— 执行推理步骤
        reasoning_path = os.path.join(PROMPT_BASE_DIR, "A5_Reasoning_noPlan.txt") #ablation A5_Reasoning_noPlan.txt
        with open(reasoning_path, "r", encoding="utf-8") as f:
            reasoning_prompt = f.read()
        
        # filled_reasoning = reasoning_prompt.replace("{premises}", self._format_premises(dataset_name=DATASET_NAME, ablation_type=self.ablation_type, premises=premises)) \
        #                                 .replace("{target_statement}", target_statement) \
        #                                 .replace("{PLAN}", json.dumps(plan_steps, indent=2))

        # no_Plan
        filled_reasoning = reasoning_prompt.replace("{premises}", self._format_premises(dataset_name=DATASET_NAME, ablation_type=self.ablation_type, premises=premises)) \
                                        .replace("{target_statement}", target_statement)             
        rsp = await self._log_and_ask(filled_reasoning)

        try:
            match = re.search(r"```json(.*?)```", rsp, re.DOTALL)
            reasoning_json = json.loads(match.group(1).strip())
            verdict = reasoning_json.get("verdict", "Uncertain")
            steps = reasoning_json.get("steps", [])
        except Exception as e:
            logger.warning(f"[A5] Reasoning parsing failed: {e}")
            return {"verdict": "Uncertain", "steps": [], "error": f"Reasoning error: {e}"}

        result = OrderedDict()

        if self.ablation_type != "woStatement":
            result["statement"] = stmt_text
        if self.ablation_type != "woFOL":
            result["FOL"] = fol

        result["premises"] = premises
        # result["plan_steps"] = plan_steps
        result["reasoning_steps"] = steps
        result["verdict"] = verdict

        return result

    async def _verify_and_return(self, square, trace, s2_type, s1_verdict, not_s1_verdict):
        direct_result = self.direct_resolution(s1_verdict, not_s1_verdict)
        if direct_result is not None:
            trace["final_decision"] = {"method": "Direct Resolution", "verdict": direct_result}
            return {"s1_truth": direct_result, "reasoning_trace": trace}
        
        if s2_type == "contrary":
            deep_result = await self.deep_reflection(trace, square)
            if deep_result is not None:
                trace["final_decision"] = {"method": "Deep Reflection", "verdict": deep_result}
                return {"s1_truth": deep_result, "reasoning_trace": trace}

        quick_result = await self.quick_reflection(trace, square, s1_verdict, not_s1_verdict)
        if quick_result is not None:
            trace["final_decision"] = {"method": "Quick Reflection", "verdict": quick_result}
            return {"s1_truth": quick_result, "reasoning_trace": trace}



        trace["final_decision"] = {"method": "Fallback", "verdict": "Uncertain"}
        return {"s1_truth": "Uncertain", "reasoning_trace": trace}


        # # Step 2: Quick Reflection
        # quick_result = await self.quick_reflection(trace, square, s1_verdict, not_s1_verdict)
        # if quick_result is not None:
        #     trace["final_decision"] = {"method": "Quick Reflection", "verdict": quick_result}
        #     return {"s1_truth": quick_result, "reasoning_trace": trace}

        # # Step 3: Deep Reflection (only for contrary)
        # if s2_type == "contrary":
        #     deep_result = await self.deep_reflection(trace, square)
        #     if deep_result is not None:
        #         trace["final_decision"] = {"method": "Deep Reflection", "verdict": deep_result}
        #         return {"s1_truth": deep_result, "reasoning_trace": trace}

        # # If all fail, fallback
        # trace["final_decision"] = {"method": "Fallback", "verdict": "Uncertain"}
        # return {"s1_truth": "Uncertain", "reasoning_trace": trace}
    
    def direct_resolution(self, s1, not_s1):
        if s1 == "True" and not_s1 == "False":
            return "True"
        if s1 == "False" and not_s1 == "True":
            return "False"
        if s1 == "Uncertain" and not_s1 == "Uncertain":
            return "Uncertain"
        return None  # Cannot resolve directly
    
    async def quick_reflection(self, trace, square, s1_verdict, not_s1_verdict):
        s1_block = trace.get("S1", {})
        not_s1_block = trace.get("not_S1", {})
        s2_type = square.get("S2_type", "")

        reflection_verdict, reflection_reasoning = await self.run_reflection_pass(
            s1_block=s1_block,
            s2_block=not_s1_block,
            s2_type=s2_type,
            initial_verdict=s1_verdict
        )

        trace["quick_reflection_detail"] = reflection_reasoning

        reason_text = reflection_reasoning.get("reason", "")
        match = re.search(r'Type (\d+):', reason_text)
        reason_type = f"type{match.group(1)}" if match else ""

        if reason_type != "type6" and reason_type != "type3":
            if s1_verdict == "Uncertain":
                if not_s1_verdict == "True":
                    reflection_verdict = "False"
                    reflection_reasoning = "Type 4: S1 incorrect, ¬S1 correct with True verdict → Return False"
                elif not_s1_verdict == "False":
                    reflection_verdict = "True"
                    reflection_reasoning = "Type 5: S1 incorrect, ¬S1 correct with False verdict → Return True"

        # 通用判断逻辑，只返回 verdict（不要包 trace dict）
        if s1_verdict in {"True", "False"} and not_s1_verdict == "Uncertain":
            if reason_type == "type1":
                return s1_verdict
            elif reason_type == "type2":
                return "Uncertain"
            elif reason_type == "type6" and reflection_verdict in {"True", "False", "Uncertain"}:
                return reflection_verdict
            else:
                return s1_verdict

        elif s1_verdict == "Uncertain" and not_s1_verdict in {"True", "False"}:
            if reason_type == "type3":
                return "Uncertain"
            elif reason_type == "type4":
                return "False"
            elif reason_type == "type5":
                return "True"
            elif reason_type == "type6" and reflection_verdict in {"True", "False", "Uncertain"}:
                return reflection_verdict
            else:
                return "Uncertain"

        # fallback：直接用 reflection_verdict
        return reflection_verdict


        
    async def deep_reflection(self, trace, square):
        s1 = trace.get("S1", {}).get("verdict")
        not_s1 = trace.get("not_S1", {}).get("verdict")

        # Case 1: both True
        if s1 == "True" and not_s1 == "True":
            # Step 1: Check S2
            s2_block = trace.get("S2") or await self._run_reasoning_pipeline("S2", "", square)
            s2_verdict = s2_block.get("verdict")
            if s2_verdict == "True":
                return "False"  # S2 ⇒ ¬S1 → ¬S1 正确 → S1 False

            # Step 2: Check ¬S2
            not_s2_block = trace.get("not_S2") or await self._run_reasoning_pipeline("not_S2", "", square)
            not_s2_verdict = not_s2_block.get("verdict")
            if not_s2_verdict == "False":
                return "False"  # S1 ⇒ ¬S2 → S1 错误 → S1 False

            # Unclear → fallback
            return None

        # Case 2: both False
        elif s1 == "False" and not_s1 == "False":
            # Check S2 only
            s2_block = trace.get("S2") or await self._run_reasoning_pipeline("S2", "", square)
            s2_verdict = s2_block.get("verdict")
            if s2_verdict == "True":
                return "False"  # S2 ⇒ ¬S1 → ¬S1 错误 → S1 False

            # Unclear → fallback
            return None

        # Other cases: skip deep reflection
        return None





    # async def _verify_and_return(self, square, trace, initial_verdict, dataset_name: str):
    #     s1_block = trace.get("S1", {})
    #     s2_block = trace.get("S2", {})  # 对于 FOLIO 是 "¬S1"
    #     s2_type = square.get("S2_type", "")

    #     reflection_verdict, reflection_reasoning = await self.run_reflection_pass(
    #         s1_block=s1_block,
    #         s2_block=s2_block,
    #         s2_type=s2_type,
    #         initial_verdict=initial_verdict
    #     )

    #     # ===== 非 ProofWriter 和 FOLIO：直接返回 =====
    #     if dataset_name not in {"ProofWriter", "FOLIO"}:
    #         trace["reflection"] = reflection_reasoning
    #         return {"s1_truth": reflection_verdict, "reasoning_trace": trace}

    #     # ===== ProofWriter =====
    #     if dataset_name == "ProofWriter":
    #         s1_verdict = s1_block.get("verdict")
    #         s2_verdict = s2_block.get("verdict")

    #         reason_text = reflection_reasoning.get("reason", "")
    #         match = re.search(r'Type (\d+):', reason_text)
    #         reason_type = f"type{match.group(1)}" if match else ""

    #         if s1_verdict == "Uncertain" and s2_verdict == "True":
    #             if reason_type == "type1" and reflection_reasoning.get("verdict") == "Uncertain":
    #                 trace["reflection"] = {
    #                     "verdict": "Uncertain",
    #                     "reason": "Type 1: S1 reasoning correct → Return S1's verdict"
    #                 }
    #                 return {"s1_truth": "Uncertain", "reasoning_trace": trace}
    #             else:
    #                 trace["reflection"] = {
    #                     "verdict": "False",
    #                     "reason": "Type 2: S1 incorrect, S2 correct with True verdict → Return False"
    #                 }
    #                 return {"s1_truth": False, "reasoning_trace": trace}

    #         elif s1_verdict == "Uncertain" and s2_verdict == "False":
    #             if reason_type == "type1" and reflection_reasoning.get("verdict") == "Uncertain":
    #                 trace["reflection"] = {
    #                     "verdict": "Uncertain",
    #                     "reason": "Type 1: S1 reasoning correct → Return S1's verdict"
    #                 }
    #                 return {"s1_truth": "Uncertain", "reasoning_trace": trace}
    #             else:
    #                 trace["reflection"] = {
    #                     "verdict": "True",
    #                     "reason": "Type 3: S1 incorrect, S2 correct with False verdict → Return True"
    #                 }
    #                 return {"s1_truth": True, "reasoning_trace": trace}

    #         elif s1_verdict == "False" and s2_verdict == "Uncertain":
    #             if reason_type == "type4" and reflection_reasoning.get("verdict") == "Uncertain":
    #                 trace["reflection"] = {
    #                     "verdict": "Uncertain",
    #                     "reason": "Type 4: S1 incorrect, S2 correct with Uncertain verdict → Return Uncertain"
    #                 }
    #                 return {"s1_truth": "Uncertain", "reasoning_trace": trace}
    #             else:
    #                 trace["reflection"] = {
    #                     "verdict": "False",
    #                     "reason": "Type 1: S1 reasoning correct → Return S1's verdict"
    #                 }
    #                 return {"s1_truth": False, "reasoning_trace": trace}

    #         elif s1_verdict == "True" and s2_verdict == "Uncertain":
    #             if reason_type not in {"type4", "type5"}:
    #                 trace["reflection"] = {
    #                     "verdict": "True",
    #                     "reason": "Type 1: S1 reasoning correct → Return S1's verdict"
    #                 }
    #                 return {"s1_truth": True, "reasoning_trace": trace}

    #         # 默认返回
    #         trace["reflection"] = reflection_reasoning
    #         return {"s1_truth": reflection_verdict, "reasoning_trace": trace}

    #     # ===== FOLIO =====
    #     else:
    #         s1_verdict = s1_block.get("verdict")
    #         s2_block = trace.get("S2", {})  # FOLIO 中的非S1
    #         s2_verdict = s2_block.get("verdict")
    #         not_s1_block = trace.get("not_S1", {})  # FOLIO 中的非S1
    #         not_s1_verdict = not_s1_block.get("verdict")

    #         reason_text = reflection_reasoning.get("reason", "")
    #         match = re.search(r'Type (\d+):', reason_text)
    #         reason_type = f"type{match.group(1)}" if match else ""

    #         def folio_return(verdict_str):
    #             trace["reflection"] = {"verdict": verdict_str}
    #             return {"s1_truth": verdict_str, "reasoning_trace": trace}

    #         if s1_verdict == "Uncertain" and not_s1_verdict == "True":
    #             if reason_type == "type1" and reflection_verdict == "Uncertain":
    #                 return folio_return("Uncertain")
    #             else:
    #                 return folio_return("False")

    #         elif s1_verdict == "Uncertain" and not_s1_verdict == "False":
    #             if reason_type == "type1" and reflection_verdict == "Uncertain":
    #                 return folio_return("Uncertain")
    #             else:
    #                 return folio_return("True")

    #         elif s1_verdict == "False" and not_s1_verdict == "Uncertain":
    #             if reason_type == "type4" and reflection_verdict == "Uncertain":
    #                 return folio_return("Uncertain")
    #             else:
    #                 return folio_return("False")

    #         elif s1_verdict == "True" and not_s1_verdict == "Uncertain":
    #             if reason_type not in {"type4", "type5"}:
    #                 return folio_return("True")

    #         # 默认 fallback
    #         return folio_return(reflection_verdict)


    

    async def run_reflection_pass(
        self,
        s1_block: dict,
        s2_block: dict,
        s2_type: str,
        initial_verdict: str
    ) -> Tuple[str, dict]:

        verify_path = os.path.join(PROMPT_BASE_DIR, "A5_Verify.txt")
        try:
            with open(verify_path, "r", encoding="utf-8") as f:
                verify_prompt = f.read()
        except Exception as e:
            logger.warning(f"[A5] Failed to load A5_Verify.txt: {e}")
            return "Uncertain", {"error": str(e)}

        # 构造 markdown 格式的输入（带代码块）
        execution_block = f"""### S1

```json
{json.dumps(s1_block, indent=2)}
```

S2_type: \"{s2_type}\"

```json
{json.dumps(s2_block, indent=2)}
```
"""

        prompt_filled = verify_prompt.replace("[[EXECUTION]]", execution_block)

        rsp = await self._log_and_ask(prompt_filled)

        try:
            match = re.search(r"```json(.*?)```", rsp, re.DOTALL)
            json_str = match.group(1).strip() if match else rsp.strip()
            reflection = json.loads(json_str)
            return reflection.get("verdict", "Uncertain"), reflection
        except Exception as e:
            logger.warning(f"[A5] Reflection parsing failed: {e}")
            return "Uncertain", {"error": str(e), "raw_response": rsp}

   
    def _format_premises(self, dataset_name: str, ablation_type: str, premises: Optional[Dict[str, List[Any]]] = None) -> str:
        if not premises:
            return "No explicit premises provided."

        formatted = []

        def format_line(i, item):
            # 对于字典型（RepublicQA）
            if isinstance(item, dict):
                stmt = item.get('statement', '').strip()
                fol = item.get('FOL', '').strip()
                print(f"DEBUG: item {i + 1} | statement='{stmt}' | FOL='{fol}'")

                if ablation_type == "woFOL":
                    content = stmt or fol or "No content"
                    return f"{i + 1}. {content}"
                elif ablation_type == "woStatement":
                    content = fol or stmt or "No content"
                    return f"{i + 1}. {content}"
                else:  # full
                    combined = f"{i + 1}."
                    if stmt:
                        combined += f" {stmt}"
                    if fol:
                        combined += f"\nFOL: {fol}"
                    return combined

            # 对于字符串型（FOLIO / ProofWriter）
            else:
                parts = item.split(":::")
                fol = parts[0].strip()
                desc = parts[1].strip() if len(parts) > 1 else ""
                print(f"DEBUG: item {i + 1} | fol='{fol}' | desc='{desc}'")

                if ablation_type == "woFOL":
                    content = desc or fol or "No content"
                    return f"{i + 1}. {content}"
                elif ablation_type == "woStatement":
                    content = fol or desc or "No content"
                    return f"{i + 1}. {content}"
                else:
                    combined = f"{i + 1}."
                    if fol:
                        combined += f" {fol}"
                    if desc:
                        combined += f"\nDescription: {desc}"
                    return combined

        if dataset_name == "FOLIO":
            for section, title in [("Predicates", "Predicates"), ("Premises", "Premises"), ("Conclusion", "Conclusion")]:
                items = premises.get(section, [])
                if items:
                    formatted.append(f"### {title}:")
                    for i, item in enumerate(items):
                        formatted.append(format_line(i, item))
                    formatted.append("")

        elif dataset_name == "ProofWriter":
            for section in ["Predicates", "Facts", "Rules", "Conditional rules", "Rules with compound predicates by comma"]:
                items = premises.get(section, [])
                if items:
                    formatted.append(f"### {section}:")
                    for i, item in enumerate(items):
                        formatted.append(format_line(i, item))
                    formatted.append("")

        elif dataset_name == "RepublicQA":
            # ① 如果传进来本身就是 list，直接当作一个 section 处理
            if isinstance(premises, list):
                formatted.append("### Premises:")
                for i, item in enumerate(premises):
                    formatted.append(format_line(i, item))
                formatted.append("")
            else:
                # ② 向后兼容原来 dict 结构的写法
                for section_name, items in premises.items():
                    if items:
                        formatted.append(f"### {section_name}:")
                        for i, item in enumerate(items):
                            formatted.append(format_line(i, item))
                        formatted.append("")
        else:
            print("❗Unrecognized dataset_name, nothing processed.")

        result = "\n".join(formatted)
        print("DEBUG: Final formatted premises:\n", result)
        return result






    def _format_target_statement(self, statement: str, fol: str) -> str:
        if self.ablation_type == "woFOL":
            return statement
        elif self.ablation_type == "woStatement":
            return f"FOL: {fol}"
        else:
            return f"{statement}\nFOL: {fol}"

    async def _log_and_ask(self, prompt: str) -> str:
        logger.debug("==== Prompt to LLM ====")
        logger.debug(prompt)
        rsp = await self._aask(prompt)
        logger.debug("==== Raw LLM Response ====")
        logger.debug(rsp)
        return rsp










# async def _verify_and_return(self, square, trace, initial_verdict, dataset_name: str):
#         s1_block = trace.get("S1", {})
#         s2_block = trace.get("S2", {})
#         s2_type = square.get("S2_type", "")

#         # # 删除 plan_steps 字段（如果存在）
#         # s1_block.pop("plan_steps", None)
#         # s2_block.pop("plan_steps", None)

#         reflection_verdict, reflection_reasoning = await self.run_reflection_pass(
#             s1_block=s1_block,
#             s2_block=s2_block,
#             s2_type=s2_type,
#             initial_verdict=initial_verdict
#         )

#         # 如果不是 ProofWriter 数据集，直接返回
#         if dataset_name != "ProofWriter":
#             trace["reflection"] = reflection_reasoning
#             return {"s1_truth": reflection_verdict, "reasoning_trace": trace}

#         s1_verdict = s1_block.get("verdict")
#         s2_verdict = s2_block.get("verdict")

#         # 从 reason 中提取 type 信息
#         reason_text = reflection_reasoning.get("reason", "")
#         match = re.search(r'Type (\d+):', reason_text)
#         reason_type = f"type{match.group(1)}" if match else ""

#         # ========== 处理规则 ==========
#         if s1_verdict == "Uncertain" and s2_verdict == "True":
#             if reason_type == "type1" and reflection_reasoning.get("verdict") == "Uncertain":
#                 trace["reflection"] = {
#                     "verdict": "Uncertain",
#                     "reason": "Type 1: S1 reasoning correct → Return S1's verdict"
#                 }
#                 return {"s1_truth": "Uncertain", "reasoning_trace": trace}
#             else:
#                 trace["reflection"] = {
#                     "verdict": "False",
#                     "reason": "Type 2: S1 incorrect, S2 correct with True verdict → Return False"
#                 }
#                 return {"s1_truth": False, "reasoning_trace": trace}

#         elif s1_verdict == "Uncertain" and s2_verdict == "False":
#             if reason_type == "type1" and reflection_reasoning.get("verdict") == "Uncertain":
#                 trace["reflection"] = {
#                     "verdict": "Uncertain",
#                     "reason": "Type 1: S1 reasoning correct → Return S1's verdict"
#                 }
#                 return {"s1_truth": "Uncertain", "reasoning_trace": trace}
#             else:
#                 trace["reflection"] = {
#                     "verdict": "True",
#                     "reason": "Type 3: S1 incorrect, S2 correct with False verdict → Return True"
#                 }
#                 return {"s1_truth": True, "reasoning_trace": trace}

#         elif s1_verdict == "False" and s2_verdict == "Uncertain":
#             if reason_type == "type4" and reflection_reasoning.get("verdict") == "Uncertain":
#                 trace["reflection"] = {
#                     "verdict": "Uncertain",
#                     "reason": "Type 4: S1 incorrect, S2 correct with Uncertain verdict → Return Uncertain"
#                 }
#                 return {"s1_truth": "Uncertain", "reasoning_trace": trace}
#             else:
#                 trace["reflection"] = {
#                     "verdict": "False",
#                     "reason": "Type 1: S1 reasoning correct → Return S1's verdict"
#                 }
#                 return {"s1_truth": False, "reasoning_trace": trace}

#         elif s1_verdict == "True" and s2_verdict == "Uncertain":
#             if reason_type not in {"type4", "type5"}:
#                 trace["reflection"] = {
#                     "verdict": "True",
#                     "reason": "Type 1: S1 reasoning correct → Return S1's verdict"
#                 }
#                 return {"s1_truth": True, "reasoning_trace": trace}

#         # ========== 默认返回 ==========
#         trace["reflection"] = reflection_reasoning
#         return {"s1_truth": reflection_verdict, "reasoning_trace": trace}