
import torch

"""
This script provides classes for simulating sensor failures
on PyTorch tensors. These tensors are typically expected to represent
time series data with dimensions (batch_size, sequence_length, num_features).

Each class simulates a specific failure mode, as desribed in the paper.

Classes:
    bias_mask_T: Applies a constant additive bias to a specified feature.
    mean_mask_T: Replaces the values of a specified feature with a constant mean value.
    performance_degradation_mask_T: Adds random noise to a specified feature,
                                   simulating performance degradation. The noise
                                   is scaled by the feature's standard deviation
                                   and a given noise level.
    scaling_mask_T: Multiplies the values of a specified feature by a constant
                    scaling factor.
"""

# Bias
class bias_mask_T():
    def __init__(self, bias):
        self.bias = bias

    def apply_mask(self, X, feature_to_mask):
        tensor = torch.clone(X)
        tensor[:,:,feature_to_mask] = self.bias + tensor[:,:,feature_to_mask]
        return tensor

# Mean
class mean_mask_T():
    def __init__(self, mean):
        self.mean = mean

    def apply_mask(self, X, feature_to_mask):
        tensor = torch.clone(X)
        tensor[:,:,feature_to_mask] = self.mean
        return tensor
    
# Noise
class performance_degradation_mask_T():
    def __init__(self, std, noise_level = 0.4):
        self.std = std
        self.noise_level = noise_level

    def apply_mask(self, X, feature_to_mask):
        tensor = torch.clone(X)
        noise = torch.normal(mean=0, std=self.std * self.noise_level, size=(tensor.size(0), tensor.size(1), tensor.size(2))).to(X.device)
        tensor[:,:,feature_to_mask] = noise[:,:,feature_to_mask] + tensor[:,:,feature_to_mask]
        return tensor
    
# Scaling
class scaling_mask_T():
    def __init__(self, mult):
        self.mult = mult

    def apply_mask(self, X, feature_to_mask):
        tensor = torch.clone(X)
        tensor[:,:,feature_to_mask] = tensor[:,:,feature_to_mask] * self.mult
        return tensor

