import argparse
import json
import os
import time
import itertools
import pickle
import copy

import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim
import torchvision
import torchvision.datasets as datasets
from datasets import load_metric
from torch.utils.data import DataLoader, Dataset, TensorDataset
from torch.utils.tensorboard import SummaryWriter

import numpy as np
from transformers import AutoModelForSequenceClassification
from train_cluster_mnist import *
from util import *


# LR_DECAY = True
LR_DECAY = False

def main():


    config = get_config()
    print("config:",config)
    config['train_seed'] = config['data_seed']

    exp = TrainMNISTClusterCentral(config)
    exp.setup()
    if config['inference_mode']==0:
        exp.run()
    else:
        print("inference mode")
        exp.inference()


def get_config():
    parser = argparse.ArgumentParser()
    parser.add_argument("--project-dir","--projdir" , type=str,default="output/localtrain")
    # parser.add_argument("--dataset-dir",type=str,default="output")
    # parser.add_argument("--num-epochs",type=float,default=)
    parser.add_argument("--lr",type=float,default=0.1)
    # parser.add_argument("--num-epochs",type=int,default=10)
    
    parser.add_argument("--train-seed",type=int,default=0)
    parser.add_argument("--config-override",type=str,default="")

    parser.add_argument("--inference-mode",type=int,default=0)
    parser.add_argument("--checkpoint",type=int,default=0)
    args = parser.parse_args()
    # read config json and update the sysarg
    with open("config_central.json", "r") as read_file:
        config = json.load(read_file)

    args_dict = vars(args)
    config.update(args_dict)

    if config["config_override"] == "":
        del config['config_override']
    else:
        print(config['config_override'])
        config_override = json.loads(config['config_override'])
        del config['config_override']
        config.update(config_override)

    return config

class TrainMNISTClusterCentral(TrainMNISTCluster):
    def setup(self):
        self.epoch = -1 
        os.makedirs(self.config['project_dir'], exist_ok = True)

        if self.config['checkpoint']:
            self.load_checkpoint_fname = os.path.join(self.config['project_dir'], 'checkpoint_local.pt')

        self.result_fname = os.path.join(self.config['project_dir'], 'results_local.pickle')

        
        self.checkpoint_fname = os.path.join(self.config['project_dir'], 'checkpoint_local.pt')
        self.writer = SummaryWriter(os.path.join(self.config['project_dir'],'runs'))

        self.setup_datasets()
        self.setup_models()

        self.cluster2client = {}

    
        for ind, cluster in enumerate(self.dataset['train']['cluster_assign']):  # for cifar: 50000; for tinyimagenet: 100000
            if cluster in self.cluster2client:
                self.cluster2client[cluster].append(ind)
            else:
                self.cluster2client[cluster] = [ind]
        
        for key in self.cluster2client.keys():
            print("cluster", key,  len(self.cluster2client[key]))


        self.lr = None

    def setup_models(self):
        np.random.seed(self.config['train_seed'])
        torch.manual_seed(self.config['train_seed'])

        p = self.config['p']
        self.models = [AutoModelForSequenceClassification.from_pretrained("bert-base-uncased", num_labels=2) for p_i in range(p)] # p models
        metric = load_metric("accuracy")


        def compute_metrics(eval_pred):
            logits, labels = eval_pred
            predictions = np.argmax(logits, axis=-1)
            return metric.compute(predictions=predictions, references=labels)

        self.criterion = compute_metrics

        if self.config['checkpoint']:
            print('loading checkpoint from',self.load_checkpoint_fname )

            states = torch.load(self.load_checkpoint_fname)
        
            for p_i in range(p):
                state_dict = states['models'][p_i]
                model = self.models[p_i]
                model.load_state_dict(state_dict)
                # self.epoch = states['epoch']

    def inference(self):
        cfg = self.config
        m = cfg['m_test']
        num_data = 0
        losses = []
        corrects = []
        num_samples =[]
        accuracys = []


        for m_i in range(m):
            cluster_index = self.dataset['test']['cluster_assign'][m_i]

            dataset = self.load_data(m_i, train=False) # load batch data rotated
            training_args = TrainingArguments(output_dir=f"inference_client_{m_i}",
                                              num_train_epochs=1,
                                              per_device_train_batch_size=32,
                                              per_device_eval_batch_size=32,
                                              local_rank=-1
                                              )
            trainer = Trainer(
                model=self.models[cluster_index],
                args=training_args,
                eval_dataset=dataset,
                compute_metrics=self.criterion,
            )
            eval_log = trainer.evaluate()
            print(eval_log)
            print(len(dataset))

            losses.append(eval_log['eval_loss'])
            corrects.append(eval_log['eval_accuracy'] * len(dataset))
            num_samples.append(len(dataset))
            accuracys.append(eval_log['eval_accuracy'])
            num_data += len(dataset)


        loss = np.mean(losses)
        acc = np.sum(corrects) / num_data
        print("loss", loss, "acc",  acc)

       
        inference_stats ={'losses':losses, 'corrects':corrects , 'num_samples':num_samples,  'accuracys':accuracys}
        print(inference_stats)
        with open(os.path.join(self.config['project_dir'], 'inference_stats.json'),'w') as out:
            out.write(json.dumps(inference_stats))


    def run(self):
        num_epochs = self.config['num_epochs']
        print("num_epochs", num_epochs)
        lr = self.config['lr']
       
        results = []
        result = {}
        result['epoch'] = self.epoch 

        t0 = time.time()
        res = self.test(train=True)
        t1 = time.time()
        res['infer_time'] = t1-t0
        result['train'] = res

        self.print_epoch_stats(res)

        t0 = time.time()
        res = self.test(train=False)
        t1 = time.time()
        res['infer_time'] = t1-t0
        result['test'] = res
        self.print_epoch_stats(res)
        results.append(result)

        # this will be used in next epoch
        cluster_assign = result['train']['cluster_assign']
        start_epoch = self.epoch + 1 

        for epoch in range(start_epoch, num_epochs):
            self.epoch = epoch

            result = {}
            result['epoch'] = epoch

            lr = self.lr_schedule(epoch)
            result['lr'] = lr

            t0 = time.time()
            result['train'] = self.train(cluster_assign, lr = lr)
            t1 = time.time()
            train_time = t1-t0

            t0 = time.time()
            res = self.test(train=True)
            t1 = time.time()
            res['infer_time'] = t1-t0
            res['train_time'] = train_time
            res['lr'] = lr
            result['train'] = res

            self.print_epoch_stats(res)

            t0 = time.time()
            res = self.test(train=False)
            t1 = time.time()
            res['infer_time'] = t1-t0
            result['test'] = res
            self.print_epoch_stats(res)

            results.append(result)

            # this will be used in next epoch's gradient update
            cluster_assign = result['train']['cluster_assign']

            if epoch == num_epochs - 1 :
                with open(self.result_fname, 'wb') as outfile:
                    pickle.dump(results, outfile)
                    print(f'result written at {self.result_fname}')
            self.save_checkpoint(epoch)
            print(f'checkpoint written at {self.checkpoint_fname}')

    def get_inference_stats(self, train = True):
        VERBOSE=1

        cfg = self.config
        if train:
            m = cfg['m']
            data_cluster_assign =  self.dataset['train']['cluster_assign']
        else:
            m = cfg['m_test']
            data_cluster_assign =  self.dataset['test']['cluster_assign']



        num_data = 0
        losses = []
        corrects = []
        for m_i in range(m):
            dataset = self.load_data(m_i, train=train)
            print(len(dataset))

            training_args = TrainingArguments(output_dir=f"get_stat_client_{m_i}",
                                              num_train_epochs=1,
                                              per_device_train_batch_size=32,
                                              per_device_eval_batch_size=32,
                                              local_rank=-1
                                              )
            gt_cluster_index = data_cluster_assign[m_i]
            model = self.models[gt_cluster_index]  # th

            trainer = Trainer(
                model=model,
                args=training_args,
                train_dataset=dataset,
                eval_dataset=dataset,
                compute_metrics=self.criterion,
            )
            # trainer.train()
            eval_log = trainer.evaluate()
            print(eval_log)

            losses.append(eval_log['eval_loss'])
            corrects.append(eval_log['eval_accuracy'])


        loss = np.mean(losses)
        acc = np.mean(corrects)


        cluster_assign = []
     

        # check cluster assignment acc
        cl_acc = 0
        cl_ct = []

        res = {} # results
        res['cluster_assign'] = cluster_assign # dummy
        res['loss'] = loss
        res['acc'] = acc
        res['cl_acc'] = cl_acc # dummy
        res['cl_ct'] = cl_ct # dummy
        res['is_train'] = train

        return res


    def train(self, cluster_assign, lr):
        # cluster_assign is dummy
        VERBOSE = 0

        cfg = self.config
        m = cfg['m']
        p = cfg['p']
        tau = cfg['tau']

        for p_i in range(p):
            client_set = self.cluster2client[p_i]

            model = self.models[p_i]            
            for step_i in range(tau):
                for m_i in client_set:
                    dataset = self.load_data(m_i)

                    training_args = TrainingArguments(output_dir=f"train_client_{m_i}",
                                                      num_train_epochs=1,
                                                      per_device_train_batch_size=32,
                                                      per_device_eval_batch_size=32,
                                                      local_rank=-1
                                                      )
                    trainer = Trainer(
                        model=model,
                        args=training_args,
                        train_dataset=dataset,
                        compute_metrics=self.criterion,
                    )
                    trainer.train()


if __name__ == '__main__':
    start_time = time.time()
    main()
    duration = (time.time() - start_time)
    print("---train cluster single Ended in %0.2f hour (%.3f sec) " % (duration/float(3600), duration))