import utils.preprocess as u_prep
import torch_geometric.data as data
import torch_geometric as pyg
import logging
from ogb.graphproppred import PygGraphPropPredDataset
import torch
import shutil
from ogb.utils.torch_util import replace_numpy_with_torchtensor
from ogb.utils.url import decide_download
from torch_geometric.data import InMemoryDataset, download_url
import pandas as pd
import hashlib
import os.path as osp
from tqdm import tqdm
import pickle
from torch_geometric.data import Data

try:
    # Code you want to try to run
    from ogb.utils.mol import smiles2graph
except Exception as e:
    # Code to run in case of a ZeroDivisionError
    print("An error occurred:", e)
    print("Can't import  smile2graph!")
    smiles2graph = 1



def get_dataloader(cfg, sample_idx=0):

    if "Peptides" in cfg.data.name:
        transform_suffix = '_sample_keep_prob_' + str(float(cfg.data.sampling.keep_subgraph_prob)) + '_sample_index_' + str(sample_idx) 
    else:
        transform_suffix = ''
        
        # transform_suffix = transform_suffix + '__F__'
        
    if "product" in cfg.model.model_name:
        transforms = u_prep.product_graph_construction(cfg)
        transform_suffix = transform_suffix + '_product_' + str(cfg.model.aggs) + '_n_cluster_' + str(cfg.data.preprocess.n_cluster) + '_dim_laplas_' + str(cfg.data.preprocess.dim_laplacian)
    else:
        transforms = None
        transform_suffix = transform_suffix + '_Mpnn_' + str(cfg.model.aggs) + '_' + str(cfg.data.preprocess.n_cluster)

    if 'zinc' in cfg.data.name:
        if "12k" in cfg.data.name:
            dataloader, num_elements_in_target = get_zinc12k_dataloader(cfg=cfg, transform_suffix=transform_suffix, transforms=transforms)
        elif "full" in cfg.data.name:
            dataloader, num_elements_in_target = get_zincfull_dataloader(cfg=cfg, transform_suffix=transform_suffix, transforms=transforms)
        else:
            raise ValueError(f"No dataset available for: {cfg.data.name}.")
            
    elif cfg.data.name == 'ogbg-moltox21' or cfg.data.name == 'ogbg-molesol' or cfg.data.name == 'ogbg-moltoxcast' or cfg.data.name == 'ogbg-molbace' or cfg.data.name == 'ogbg-molhiv':
        dataloader, num_elements_in_target = get_ogb_dataloader(cfg=cfg, transform_suffix=transform_suffix, transforms=transforms)
    elif cfg.data.name + "_full" == 'alchemy_full':
        dataloader, num_elements_in_target = get_alchemy_dataloader(cfg=cfg, transform_suffix=transform_suffix, transforms=transforms)
    elif cfg.data.name == 'Peptides-func':
        dataloader, num_elements_in_target = get_peptides_func_dataloader(cfg=cfg, transform_suffix=transform_suffix, transforms=transforms)
    elif cfg.data.name == 'Peptides-struc':
        dataloader, num_elements_in_target = get_peptides_struct_dataloader(cfg=cfg, transform_suffix=transform_suffix, transforms=transforms)  
    else:
        raise ValueError(f"No dataset available for: {cfg.data.name}")
    assert num_elements_in_target == cfg.model.final_dim, f"The final dim: {cfg.model.final_dim} should match the number of target elements: {num_elements_in_target}!"

    return dataloader, num_elements_in_target


# ----------------------------- dataloaders ----------------------------- #

def get_zinc12k_dataloader(cfg, transform_suffix, transforms):
    dataloader = {
                name: data.DataLoader(
                    pyg.datasets.ZINC(
                        split=name,
                        subset=True,
                        root=cfg.data.dir + transform_suffix,
                        pre_transform=transforms
                        # ============ DEBUG ============ #
                        # root=cfg.data.dir + 'DEBUG',
                        # transform=transforms,
                        ),
                    batch_size=cfg.data.bs,
                    num_workers=cfg.data.num_workers,
                    shuffle=(name == "train"),
                )
                for name in ["train", "val", "test"]
            }
    num_elements_in_target = 1
    return dataloader, num_elements_in_target

def get_zincfull_dataloader(cfg, transform_suffix, transforms):
    dataloader = {
                name: data.DataLoader(
                    pyg.datasets.ZINC(
                        split=name,
                        subset=False,
                        root=cfg.data.dir + transform_suffix,
                        pre_transform=transforms,
                        # ============ DEBUG ============ #
                        # root=cfg.data.dir + 'DEBUG',
                        # transform=transforms,
                        ),
                    batch_size=cfg.data.bs,
                    num_workers=cfg.data.num_workers,
                    shuffle=(name == "train"),
                )
                for name in ["train", "val", "test"]
            }
    num_elements_in_target = 1
    return dataloader, num_elements_in_target

def get_ogb_dataloader(cfg, transform_suffix, transforms):
    dataset = PygGraphPropPredDataset(
            name=cfg.data.name, 
            root=cfg.data.dir + transform_suffix,
            pre_transform=transforms
            # ============ DEBUG ============ #
            # root=cfg.data.dir + 'DEBUG',
            # transform=transforms,
        )
    dataloader = {
            "train": data.DataLoader(
                dataset[dataset.get_idx_split()["train"]],
                        # [dataset.get_idx_split()["train"] < 1000]],
                batch_size=cfg.data.bs,
                num_workers=cfg.data.num_workers,
                shuffle=True
            ),
            "val": data.DataLoader(
                dataset[dataset.get_idx_split()["valid"]],
                        # [dataset.get_idx_split()["valid"] < 11500]],
                batch_size=cfg.data.bs,
                num_workers=cfg.data.num_workers,
                shuffle=False
            ),
            "test": data.DataLoader(
                dataset[dataset.get_idx_split()["test"]],
                        # [dataset.get_idx_split()["test"] < 1000]],
                batch_size=cfg.data.bs,
                num_workers=cfg.data.num_workers,
                shuffle=False
            ),
    }
    num_elements_in_target = dataset.num_tasks
    return dataloader, num_elements_in_target

def get_alchemy_dataloader(cfg, transform_suffix, transforms):
    dataset = pyg.datasets.TUDataset(root = cfg.data.dir + transform_suffix, name=cfg.data.name + "_full", 
    # transform=transforms
    pre_transform=transforms
    )
    indices_test = [42285,183639,151286,169347,154894,115987,95866,168685,177320,197940,38116,162823,149649,142803,116646,38562,3549,50567,136869,163292,172214,58171,84459,23534,112357,195793,132478,23693,144534,80206,188172,152709,89872,121018,104332,79470,123439,171861,180287,100779,94651,116710,118914,194599,28115,140825,41999,103236,143699,49328,53850,195181,95935,39674,91693,44519,177741,144196,118495,62691,181694,26286,67857,16474,115998,95857,25876,61247,46710,67904,179766,98369,156269,53656,68710,151581,126442,128319,16359,97764,108516,85862,145031,94967,162432,58826,77291,109601,176475,152978,177672,91512,70638,99556,8020,57023,28261,154705,147594,176838,50816,40559,91556,92497,62539,24884,24305,72518,89608,150789,47101,199085,81953,181004,122918,121520,9722,32656,149506,77700,62154,160588,162993,21135,22025,184092,109373,107302,27379,150638,35183,125675,24458,192939,182886,96482,123119,68849,177266,139922,150125,113302,46352,108215,133583,147972,132431,97440,116258,186600,178634,49239,38742,139774,13506,170586,113130,27833,147887,117433,188390,181165,92637,71403,187301,156624,30815,77156,153466,91278,128860,162393,186753,113353,197972,145025,22359,161509,52656,18997,103572,10997,75399,84397,9923,112845,11230,82807,117948,46431,157180,93879,119550,102914,131457,177183,8944,32039,91898,6982,89523,55777,183381,174236,192569,28734,67971,143105,91620,2525,63393,4361,59174,135033,28613,111465,123764,91906,172358,186828,50153,131431,2912,158849,147129,161931,175440,181519,66969,188177,88927,71401,128740,173142,155471,168091,28371,88442,195014,108624,108679,2365,5242,122031,9955,85483,126054,10039,12072,180436,104351,72245,79704,124247,96382,129459,32419,100154,95115,59553,93697,174253,82395,153293,133977,29559,116619,12788,63773,168084,50828,83265,136787,165902,46621,93053,81793,98843,30866,133453,171979,132150,158162,151173,60876,109920,164531,29790,187005,158311,51571,162585,88817,179055,47089,2787,56024,45381,82907,124251,114022,165081,168098,148886,99161,159002,2611,74634,134476,100699,130733,126980,105744,187303,64698,182050,162727,49227,139075,2842,129788,164192,77933,96951,116981,53383,91109,154245,2435,106842,125031,33578,72448,12469,164878,191676,70588,55482,71375,74797,111662,48613,124070,168661,41112,174770,115269,53696,101929,182610,128175,192298,156892,90151,167101,77452,98905,49843,12243,3136,112754,27167,126929,131388,3209,175187,188150,26202,48770,8499,81183,7495,159367,39821,84804,145597,54495,154679,161159,117159,49921,55394,57541,181583,29914,10949,68798,191126,45683,164682,40356,121182,46897,189033,175213,139446,184612,95596,48226,82559,23610,133556,22391,119079,65775,70529,140697,10545,4511,159493,90043,192599,188445,4210,40102,16989,86907,32335,121987,169317,29404,59881,80337,80599,125510,138636,131224,91252,188270,15731,75688,55019,89993,111550,27705,132261,94894,86036,36854,169924,59044,11180,149671,45285,18515,78640,123326,45513,132600,174067,174302,66649,122674,47945,144261,31350,128559,147722,109697,117558,77647,34941,117236,197991,127039,153269,151421,67446,137621,78101,183961,64360,50381,65024,23965,189769,86624,157259,184909,113028,116215,67030,53377,113609,75113,168525,187888,134583,106184,172813,101641,194636,132159,50603,10974,6888,87101,163373,96727,1766,43886,158155,154355,113411,102684,196053,52592,108029,186003,7342,157737,38428,173507,35097,152575,4397,39997,146681,72335,66725,4255,31861,111108,64996,55940,79414,159379,30435,156714,90297,78382,70194,120827,13199,79336,17390,138307,108549,128368,104037,77816,125078,11712,158454,36847,69780,116972,78778,139812,107269,88998,193012,184878,137106,124213,136124,163908,165806,135768,194832,162338,20925,26316,15434,40107,20354,121035,147528,184270,123539,165055,130528,127663,72499,194375,162303,176502,199627,115864,96666,73133,89174,45790,76613,119124,55073,109880,118330,160324,121969,4744,15728,131398,142734,4727,153687,115907,114001,191450,6321,198349,108802,127630,721,161983,58535,86371,112982,55198,180629,167312,79410,123251,23750,81014,43538,52780,30585,138797,148267,70230,64004,171841,99027,131310,4302,44020,156871,58055,155363,114959,90791,169812,129457,104918,2620,53589,171386,65671,23018,132590,24481,182915,166852,192676,185587,80421,35137,84638,195336,151922,17772,138454,58555,31527,161813,143935,162976,69093,50481,128691,24053,12829,8077,175098,12639,80665,31957,85923,21053,146140,192197,23423,194319,73506,144096,180609,15928,65312,119497,115459,150599,58883,137013,176145,159311,85165,198506,163627,79774,116970,18706,131127,151694,158685,128450,195781,74419,90421,70185,123840,199201,61109,77336,191093,123433,6831,70169,149619,19197,35950,79267,173517,155211,35739,145229,99905,175712,133364,9237,159324,8039,184281,132845,156715,128098,42376,126741,130033,59324,163425,135089,129420,7813,69116,30909,24774,176237,56667,126261,7472,106033,154460,39293,184504,182091,19448,124093,69740,73780,32860,143683,172789,120209,39216,80407,145646,160823,41106,186065,69642,154826,10117,120501,114699,22086,148248,157789,92887,147901,102050,118586,10788,24508,53725,188944,170627,66069,183390,106130,171141,62742,82114,76900,32972,26377,173442,171387,112993,57548,45411,117374,63903,95390,93005,136296,130018,107323,103954,189231,74559,173706,10403,147471,7084,190356,35146,177303,72644,141615,101675,16940,6412,94775,176159,93430,92448,63604,54659,147636,50028,137459,158822,152283,16900,100056,40855,99727,28070,103747,171843,14913,20608,172813,27110,154823,187551,153032,101793,121443,185429,31787,44940,90504,26531,179801,118216,146470,157022,171994,38187,173884,32143,188045,191059,108446,156914,72480,178910,190470,107950,126011,145464,152260,49459,183564,16387,13508,92724,108046,157603,86501,8756,3334,50113,126767,51254,67415,23536,86107,175469,116913,76049,121780,154255,81311,163446,41270,157860,179791,1959,90808,145510,161998,128303,134124,80200,186477,53690,23022,123027,105817,17932,93459,183008,186788,45196,65679,171124,39739,64839,63960,56041,156928,48764,147031,130429,194299,79601,140698,16196,90087,72646,101105,174441,121554,91508,140701,112759,110865,50123,193322,167801,80709,191648,20516,14558,187168,54364,121341,128274,14326,189464,163574,89105,180854,95880,137282,53857,175682,177746,59662,71942,88686,60720,88000,92288,11232,75581,24062,48705,28694,139080,163431,30093,152740,43562,29851,84260,72832,48220,98010,150023,194193,176121,165079,195914,175336,32903,17840,145140,135809,117260,39625,111393,34010,137450,43035,16211,6650,93900,5683,18419,115590,189343,120188,13645,146139,52305,187689]
    indices_train = [44597,117684,162970,50552,196703,47896,101534,175036,121407,40496,187884,98176,87911,190348,20385,160757,324,55813,64278,191854,96187,73596,39104,1821,10909,64913,145455,82508,83309,146554,90197,103018,21916,50561,96272,89768,150970,126578,121562,90519,41249,68335,74484,181616,183810,101502,95547,156875,182181,173443,186305,32284,107428,109753,23669,4748,198408,142494,111880,155131,168550,60461,111154,109910,193762,123788,139647,183026,196037,100753,112312,18902,114140,6200,113393,44613,72825,191921,32020,18655,193110,152992,41163,52122,66127,134469,164348,159546,70853,35843,105654,182483,23642,3494,31378,134850,40769,86720,160919,37726,145788,113584,69226,16136,172005,147080,131740,37820,52154,176812,158445,37153,119179,152374,73028,112490,187891,136330,141974,90006,106973,43577,137879,79719,144443,56360,17120,78439,15223,118546,141315,188330,139140,79468,42395,132572,133888,96166,55191,114410,196705,26133,55891,109208,6947,193876,3680,134762,65976,101759,81934,90692,145454,126926,144854,83316,72439,113326,114782,93078,150876,178750,113777,145481,140118,165339,115835,187569,749,63771,7541,157776,47724,7274,53182,140971,117766,77164,14383,79063,192480,191056,86985,42416,164883,182518,116811,174156,93442,182577,70487,128099,117215,147036,175201,188702,155536,15084,60104,175934,184436,145116,197098,71035,148993,50736,159232,187785,151366,119082,25745,44645,166250,7940,148052,134250,113379,171069,96764,133284,39168,143940,121822,117862,72855,146714,198754,158028,31334,71552,113161,135495,88489,34010,52545,133610,180197,14545,13780,93745,60616,182656,38023,37823,2262,101050,80739,192351,5803,197803,51198,113841,149022,46896,62000,76876,156513,93706,100639,155247,50649,92459,197088,171500,187744,36276,131378,188766,124898,71904,25138,67168,169105,171632,68587,156194,35906,12080,57320,182405,66815,138953,51088,185841,45245,14979,111085,195908,101313,33044,456,185840,179239,42425,48796,69990,83762,139034,97582,38604,122855,119784,185679,73853,59094,149708,53298,93250,23051,160673,21357,90555,30047,186443,92040,90059,194395,56077,198393,8406,34160,134283,132712,185047,40836,117528,160385,60049,61526,134348,137442,169485,42791,186134,110170,145706,190978,109402,103325,133483,180477,196447,72847,81602,27072,125670,83140,153451,67497,182107,158951,155451,99478,145724,31073,12246,100047,39157,128572,144399,21079,7104,148902,45241,181628,127562,127325,124330,86516,72351,150172,170321,35776,8047,187646,84821,172635,30203,145789,137298,9512,195111,14542,112427,106847,185702,148609,161656,97386,56311,144406,175580,106353,150046,39899,101461,66210,32814,158133,190882,40509,90613,116968,172601,83615,21200,74736,108048,18800,141562,188492,63177,180680,114028,108576,5261,16597,107735,183840,111680,117141,155612,96653,112206,129685,198759,145820,17000,67076,196734,181602,59975,157900,188009,69331,56606,132527,146273,136577,75233,61940,37337,170995,19214,6205,163117,45765,146941,161480,169266,48293,122121,89752,49555,96876,171140,105456,36853,83262,76929,130630,65331,21746,76043,68085,165156,152783,190833,161098,36234,108737,116969,112058,158245,117371,51534,67258,84222,130765,199056,190165,163313,84022,148563,31753,59469,84281,126278,33516,94536,71027,17316,21111,60808,11392,13032,164081,182038,5795,28357,197481,113698,63331,4997,102302,125906,193758,26013,127509,2142,30496,27928,131282,46323,173333,198517,148922,21561,42438,166362,145647,129487,120585,28616,122732,40917,151597,139475,62626,184349,152822,90313,61126,62204,19896,133667,191332,13175,83611,43832,196108,137566,145747,148432,52113,66208,8805,90479,111869,51619,73693,104242,150385,121642,158905,60332,70761,42358,125581,56647,188476,7256,128161,128796,10896,54420,135329,192361,172622,142207,142350,28329,117149,179185,162578,162359,49762,156824,165404,120442,32446,74991,91227,21094,169926,91796,38647,133920,77481,60649,22992,46184,57876,160322,55575,26899,70256,99279,54611,153020,177350,123616,187847,5516,192186,93260,67650,164388,197,75105,67200,108568,184546,12726,30217,137643,75009,127064,40372,62593,31352,142753,97587,175025,70902,11075,45783,14988,154016,160761,173492,103321,148967,83074,18739,122843,168617,197453,67072,169619,166221,173707,128588,151365,155088,84661,53573,30384,1207,83689,185151,3408,67084,133333,10710,161851,56588,75039,144378,190196,154320,24145,24071,22687,1581,146265,199438,73328,70675,26349,189206,112476,60027,6052,85639,16351,54436,155685,40618,51483,25941,162860,91286,77763,160638,56798,78837,145320,128104,151883,125385,197130,195088,27789,86906,169934,81635,60500,74195,188962,174775,42680,126466,49537,65809,177063,154307,83562,72000,196456,146370,117877,64524,154674,95828,143155,185887,47682,80669,99958,171337,97245,196711,144773,109579,174520,49576,98139,20596,41306,163098,21080,169813,196171,115456,150841,60926,140356,184678,127450,95144,99834,28509,47532,127584,117443,3897,96154,57917,199108,114016,36202,122702,36065,151387,124985,133958,53,161965,107811,198547,77085,111914,191173,103406,150975,87175,49323,195954,163193,143904,46080,47319,184244,126785,197352,118980,47145,32963,24385,189096,110883,124862,77334,70657,69765,16089,196246,19736,61744,178486,164921,46519,181344,145833,198320,57909,12273,193256,191086,51702,178083,18831,131084,176574,46510,84516,159932,52232,25535,176233,188730,37123,71442,55920,153461,156346,116644,168321,41214,62294,169411,6556,197155,152795,97160,133378,105126,99599,104786,144065,189462,145699,63674,129374,92435,31618,149,70098,179102,132263,137964,137626,152636,84068,62389,153685,57181,191329,171177,164175,28076,142941,73809,23375,13121,199740,194014,196973,159711,136280,175768,18051,165608,103590,97396,156486,56690,186873,161138,122535,155135,127246,98561,90773,40995,12983,165220,33943,68505,154317,57415,151509,134082,83689,92318,6106,134556,68678,168499,58700,49839,174707,124390,124463,19214,186911,70862,46278,178781,52271,189774,24707,24416,88781,174188,124986,169961,91046,79873,27658,22034,82076,66908,29602,135078,85896,90221,49400,88442,142897,146796,133560,20286,6831,55525,135835,168087,175376,46848,4878,69225,175371,34784,63220,56883,99014,128575,57446,74867,20507,21111,199212,198841,132282,74872,75161,162294,53270,108896,62266,158161,2700,99616,177138,173319,195577,8455,101635,18048,90399,117682,90770,142180,32362,95191,77282,29165,16161,186979,33976,78853,50291,38358,74634,14337,69655,134078,98811,170547,21328,128596,51123,127292,81496,132954,166407,114948,190078,120367,190451,103293,17212,108254,174867,122802,183836,163323,147192,96911,40169,100534,101792,114397,59064,3374,81455,78272,62906,134478,55666,2252,84922,126021,56209,102647,131158,22832,122151,138888,85663,54602,138211,59875,36019,197734,94065,189912,102940,46899,58387,18026,12209,33726,118795,129553,6216,199752,140886,165047,102591,147425,44406,29548,128348,139931,99453,162334,80355,41279,34949,91718,117899,23553,23783,18366,30798,80129,198042,127517,50933,78735,81791,56759,98697,192798,68482,1562,194352,50703,39354,168625,183131,132532,118517,90475,111641,136993,139350,121200,157941,89286,153944,182854,83057,110033,135810,163929,130495,75216,182031,132264,167342,198246,131334,150762,146005,95094,47034,20353,199185,146527,114709,16662,182908,124784,47652,145348,93983,20357,144571,33542,66859,114999,130493,186745,12940,93501,177928,150989,15983,585,23405,136476,175321,41503,72186,97015,46900,6429,151355,39769,88278,97640,127255,171274,76881,48899,63869,8641,113995,4660,198111,51192,33230,162817,174479,39603,142419,178395,171311,74638,85102,9416,93,119374,74013,126162,19480,29405,187880,135868,62733,111809,126833,80362,173292,29363,192317,39043,121847,72978,70513,52080,66329,143420,18723,82032,190611,67328,52310,67019,8537,115488,66390,163841,32878,193564,81267,85951,41724,73266,178006,139403,164595,149208,137219,31151,48904,45559,191603,92417,46296,38885,199221,11906,74910,120994,69223,57122,160551,194062,168300,114386,85578,31948,166201,42852,109409,11446,63063,104690,54720,153228,180680,144651,156282,121982,127514,59148,14514,6242,127392,106747,61500,41903,100605,185182,35867,168166,4088,150237,11444,99243,131138,47015,149264,40758,69871,66171,102666,86599,701,154201,79238,167809,58564,33066,182821,39120,22296,73232,63850,27497,100968,154136,32677,117886,99973,89967,185573,81860,44828,193194,120548,21192,56624,107922,133491,192418,101443,188648,124433,109267,112192,72281,90742,179650,80398,57678,97568,161739,2052,198521,127106,179529,116558,104896,128190,138979,8596,164281,179051,98549,148768,34709,190578,105209,51164,96551,39092,20828,144679,194552,194104,164752,49921,87535,192358,82482,101724,161726,178331,39448,103915,120918,190366,146208,102320,138820,99632,95039,193732,152836,196294,161851,69972,199938,166803,96070,152176,75962,23382,109361,159483,67783,2284,6639,175887,137871,162530,91821,167100,89876,102474,192877,135787,143614,169284,16596,702,124192,140499,162071,155768,155853,197244,40919,47670,184763,169503,89088,166738,76542,185361,79843,85273,113083,49979,79349,103389,101024,153177,36977,136354,57354,87456,117847,157012,155087,67619,117965,56315,97590,113430,38000,194410,48341,148391,17293,199647,48726,171225,191287,13564,64707,31060,192889,104783,119813,14500,32955,122183,18947,172126,138723,91064,80561,166917,41694,93855,156262,192336,94018,79633,81838,16225,45030,177834,71592,57291,32830,58155,187834,39392,78803,19548,156675,166693,21356,112035,25999,1298,190010,76216,92365,183629,123851,2316,83899,13634,110088,77296,116429,112804,199712,196478,67392,35205,179329,193867,161010,54961,198747,193708,154929,67349,130335,5071,144757,43762,129510,143291,96583,56480,113157,109226,194222,199148,133190,110850,51829,197190,60992,98279,12418,192338,1929,184912,33125,29579,162544,136618,131120,93,42412,75825,119292,34996,92920,167824,113353,112170,61292,179709,120091,30587,85169,84743,51900,38167,94812,123095,181603,48295,11991,87773,91435,55130,128580,82123,4720,137239,55864,44026,109296,54776,175067,188344,44360,89133,24470,17388,113246,17407,55786,64794,40388,174672,117261,192890,66990,167440,41294,550,82741,193219,199804,161075,180012,95146,89942,15699,7279,174978,152889,136516,115517,109096,18556,186414,184924,97937,140786,179791,8697,85368,167003,149395,134585,171947,154932,105970,181382,105594,27752,69603,75383,38462,14230,188382,76399,197312,99235,63618,197469,42344,114051,31932,93591,86468,117886,127778,117887,164289,86918,29296,177029,57704,59464,108256,73604,63517,142732,134681,189405,114941,81766,83363,22733,105737,88637,38167,144822,12226,50386,106514,125419,8124,170193,91823,193478,36560,132720,16431,82683,64021,144641,58009,14680,78104,124414,20975,192442,10328,43480,48979,97984,139388,52527,63842,165136,84841,169689,103522,67345,12834,126532,146264,115454,142861,177668,154687,91416,97468,185260,36491,180629,113151,82019,83369,164667,26594,91922,180432,178752,132109,41134,85438,161705,192023,35067,127137,112335,171154,188025,92199,152409,108230,118279,75832,23042,110715,76450,152900,18733,23574,150787,174198,160311,159127,123230,56252,55023,121429,166354,60228,126420,134080,168680,16222,91275,16017,98803,4506,23940,120298,177670,9192,107290,193983,106598,114626,56869,117839,151687,35616,104463,82614,106272,52195,46149,8120,87075,57277,1183,14324,61511,80922,184239,109149,67646,159131,194722,192953,63950,112759,56917,53128,91310,86734,37546,140123,72735,34274,39688,32304,73988,131343,136264,48844,138227,98,73449,141919,70756,27199,133814,66118,141728,152879,182256,64065,166997,30751,91989,158503,12592,157467,158097,24980,178382,3392,127521,116151,180155,30733,26130,137632,7091,70424,12200,96871,48463,171656,28846,87849,145722,118204,53272,9446,67198,41726,11317,109665,20067,98052,62454,186553,59486,57655,175778,56677,184123,81486,149925,191669,198704,106388,181486,183008,177365,166563,36725,123206,145093,135174,179226,78892,126781,59006,199724,143771,131265,88389,149842,182733,122797,1118,37416,40964,154652,116397,66411,177313,133715,84638,125428,87106,104752,153562,18072,138195,52536,77433,22535,39987,94171,46421,104768,12736,111901,101795,19636,110712,146521,141045,89752,28195,20405,187950,170483,57141,141411,142431,137159,17650,52318,16251,176757,170604,168021,43991,40779,191796,73300,139259,14563,101807,161437,36359,183828,51743,115474,33901,58980,123412,138819,94553,29226,67749,187736,78550,54279,185205,182155,41902,12183,92552,96466,177754,170045,48793,17260,86104,194432,28210,177719,67718,90368,85186,166960,174081,148289,192518,193821,86541,116545,120390,186462,35351,77306,121037,7226,66057,172142,19855,119655,83794,123091,177359,119362,136574,141922,111797,122199,124022,191091,41259,74581,173591,173610,160217,43429,144551,138598,36461,150465,96399,130023,63093,24673,113666,81227,6792,149597,184850,87495,171910,83060,11891,69939,189681,80829,184139,979,106931,3804,42068,63972,160442,116368,124297,110525,3780,36885,106753,33851,19661,110450,27208,196037,141251,46314,149456,96020,148006,196434,29148,76187,9527,115068,624,115888,173446,157278,94247,24122,111852,191835,55997,2868,41334,113337,30043,170426,25961,33824,83665,69157,150164,187184,127248,74701,191168,106335,172168,121951,69212,151995,155343,24234,142495,11217,185736,43603,44344,52936,60839,54708,119501,33592,179115,39591,28256,136323,44932,15597,95840,190963,6168,117357,113799,156659,44505,159087,84142,113893,153942,164892,150361,96067,164719,104782,55843,198360,129424,38141,90754,142161,9996,7282,28293,179307,155055,90000,187173,79225,23695,77930,193711,104428,109357,46741,135541,89137,71931,181698,73557,98549,192184,16983,38997,123639,194796,190974,173971,193599,124687,143675,126377,133290,15163,89747,65398,160813,75915,80419,34661,46717,161359,61562,122430,178791,38984,85391,172838,45296,36270,130463,153802,74211,45251,88163,15289,47201,93140,188169,26909,88319,63887,21290,191846,20024,168447,95065,67004,174257,12270,140932,18580,73304,9345,96380,143510,144013,51294,162053,80003,137421,104242,7130,73874,102043,4515,165983,157280,42004,81384,69763,140076,12581,178041,94264,88902,35158,127804,38621,47142,54862,134197,112067,97322,160248,105965,164231,119820,68100,15197,89986,160578,177605,68982,68800,58268,90420,73921,84182,135685,87431,43226,27266,37444,59499,45264,90913,111791,184662,151697,49116,196349,84158,129428,5867,167764,139880,36663,116208,41786,139548,190919,88659,174944,111870,125480,158468,40674,66812,181299,173911,135172,30022,57498,56358,158204,139008,47420,41123,114145,34891,64451,58119,159992,82574,193151,175795,119878,153776,118035,29429,99194,164324,111211,157034,25230,140465,88286,17832,37472,50446,188969,50309,136158,138108,194851,61550,39777,21934,156938,150664,194524,181251,117620,111935,103837,53802,178603,198942,8261,75318,69503,100672,98639,19011,83283,81198,105951,103566,164202,69237,91486,4874,159306,57680,44397,106602,11587,136004,64592,166535,102314,62499,72942,160786,118266,26278,41529,170342,92982,100415,162331,58091,3812,198305,56757,87048,9694,22432,18841,181035,50089,128497,101589,159609,145794,120814,192103,195793,146433,17622,101540,77459,20170,100109,194211,191493,21975,22106,53820,79778,146976,74421,144940,139796,184034,3382,56649,163437,80526,45683,3162,26723,87740,174301,144058,38029,31714,70414,47919,28939,72575,101274,100069,28279,59562,110298,29181,13879,22256,92292,112178,129572,98053,108741,178117,89668,10961,121438,57671,2896,95889,128649,42753,9898,182235,130206,26642,4600,21506,35201,94085,100060,128119,5047,17551,18037,37448,122346,87821,155471,74501,177530,193383,157441,107909,15045,94128,177486,142444,155751,7110,173529,141340,71423,62373,127917,50621,34955,43480,18872,136044,110241,149468,52143,71107,169148,144904,135758,71294,155543,180172,127847,120967,193780,152328,187973,111466,183457,187096,184895,104259,115503,42688,180312,180481,195775,152531,91940,64946,105503,119102,139845,129312,129049,165585,140250,141282,115609,122615,131682,196573,163365,846,37434,15288,86454,167826,16971,30401,167844,189878,37586,181208,20136,124936,175893,149680,109190,93941,60370,115791,167552,109681,159030,94765,93097,167541,111434,148317,3177,130809,43342,110907,168721,130782,2963,22079,51964,107885,184307,158641,1056,146891,77045,157174,113256,198405,176930,144359,47035,29900,140282,94517,9828,85360,109156,171641,184601,104139,48050,98120,95050,97766,58633,3066,51291,106677,153448,4916,22300,162089,176211,171457,181004,155814,180860,11614,5511,79099,1013,20500,114101,10107,15593,177180,55758,169482,25053,48938,7731,173605,16624,113130,170433,190034,104620,8702,92841,9680,163398,40570,106726,15063,106891,18328,117517,112150,61980,20107,50854,43234,91591,149351,159031,77367,31195,114412,85917,117707,63393,102105,180203,112690,8764,114369,4990,197487,51686,164674,8945,4990,137832,176791,30779,75204,10196,112305,55967,121179,78870,168823,144261,145281,118716,37381,170771,79687,128181,81382,124595,97508,102305,1732,122946,31567,64733,168713,162914,167278,27574,71440,130842,35583,99445,55648,120812,139422,48902,45431,120414,81262,84468,75355,61564,190478,150724,1939,44328,80648,92507,86334,25120,129260,41703,60717,118351,10369,33539,161383,61261,148148,160456,50545,86196,43900,33202,195920,122650,75893,28207,100769,56500,144018,35442,127161,108055,158585,128677,59928,48541,127161,199515,67888,183815,72699,17034,81267,67620,95646,179618,146339,98024,105666,24831,141714,84623,177132,154765,185387,137223,47774,35925,11626,93593,140265,116893,161987,14155,125347,139776,42528,195813,71923,186170,171479,73034,123612,17249,154393,131013,68441,186914,60683,19271,143858,12457,95064,170815,178465,175993,85381,41663,131520,99628,126748,80037,183914,144051,70416,100362,64695,192505,126735,172473,82027,46705,18199,10255,102948,61992,58879,132622,76128,83,118310,98147,197763,44781,35043,17575,105129,47497,91249,150504,187621,61764,115771,171019,111680,63861,87947,86061,129868,135726,64147,127488,91175,166202,33837,163824,158960,22744,61670,144078,124360,8419,175284,60854,73388,189134,9767,108744,139814,31936,158961,94251,60553,70004,59612,141690,99813,164231,52035,113235,34537,188109,149993,160099,77956,5933,62144,169045,90756,173906,21100,44283,127104,176635,20093,104651,42049,6205,32879,194328,76190,176385,43784,39323,20625,98310,87263,30110,16716,89453,160143,23980,183663,111724,121764,193300,46218,62344,2017,145148,101027,35315,113390,5158,113333,28381,47843,99140,69677,51215,136449,96745,13587,91662,58857,157986,147312,158648,160681,19635,126614,88941,153747,70469,193683,94530,156972,81323,148037,154011,23013,106086,145861,81234,121601,150526,82859,32937,5343,36029,10053,153807,187212,196618,175901,17594,124647,159489,131878,163480,77315,182309,143083,173174,156334,196921,104772,106548,122504,69532,28362,128120,74111,139815,103940,13830,185168,123605,181091,2002,125588,83310,184455,14279,57886,52380,88843,141291,51667,6153,107793,95169,56135,139441,15694,110126,24039,189143,69180,85595,97342,88458,32357,128699,26059,92613,79683,197597,166533,62499,159189,176446,154700,102382,122401,118939,188731,47069,30118,110227,69972,87998,138064,47534,174259,90980,44530,87509,184218,102194,160541,68265,48630,191911,169228,130022,10578,179185,144402,131892,87209,26889,57191,107661,102385,20928,86424,189898,186410,184361,139187,198798,167249,57386,7036,429,59217,56835,72262,132165,100645,123921,49268,163719,192907,8551,88962,41941,65809,156579,199891,152604,5918,28325,114067,53154,31209,172224,187334,9640,71756,12059,80044,6961,133031,104835,94469,183272,177776,118159,191007,92998,105970,75115,137197,143123,34822,142622,105372,167134,33335,109106,72277,169146,15969,135079,83558,48617,56051,128440,152882,91829,38464,36565,178768,6149,119948,57771,27560,164853,42878,13710,26480,101680,47319,159177,108090,34088,53462,62259,167700,28259,151636,70345,50018,131368,174121,192667,191379,38782,9456,120662,138810,24362,143220,118345,180389,61284,142144,198309,87905,68014,106510,66951,14393,145686,108045,153320,125222,146322,188974,87461,67972,191107,115818,116841,40740,178126,64727,51809,114701,1064,197028,173867,168328,191316,182059,64927,64454,39946,133281,162460,31247,136117,13246,146040,103026,132593,136348,86160,23337,131613,145801,65101,99149,194334,179352,108067,136484,42599,51379,80533,159246,42559,110429,20190,48922,162153,88401,78530,165338,159502,168082,22847,119315,165063,127881,188596,160246,39838,158402,69095,37338,6726,188436,111596,122315,10177,52646,143026,133249,82528,87627,182435,114817,6041,145824,29411,155563,19507,135269,130242,194953,157415,177266,179393,143177,23863,57727,63080,56592,2436,70744,149688,37052,179125,98389,70529,90625,132461,62699,81078,59986,119640,187217,9969,12026,184617,89229,2638,35231,63073,178397,73950,33939,76306,146918,133488,189491,181749,36282,163266,117909,123508,151930,19880,181306,196332,104301,16100,190425,36009,195581,142294,110104,17821,147507,58348,156682,198483,110925,84394,60627,90755,120995,24994,56569,143968,2083,42447,100109,102961,13802,181640,89436,178200,7771,50903,141024,167737,3529,124520,101782,182694,6456,104542,103234,71252,127494,116008,143983,90609,158549,33277,137030,9545,198098,60241,10164,21600,65009,73150,2326,137285,29538,80675,79791,173053,194145,137136,30908,84660,128184,194101,154310,163919,39451,77756,95201,35193,196962,191960,75896,98411,49007,61853,77768,159041,36361,184616,47905,124274,145508,111690,199697,32783,18190,44907,117111,113199,127689,178454,67114,104313,20617,25719,198745,52235,152521,139555,10867,160564,148079,1752,172645,49289,192406,167141,90421,77056,44059,183322,92813,5802,78376,88062,125293,174342,188525,137726,81021,121618,140390,9647,127036,37232,73635,197237,127972,96526,29087,129700,174253,131931,166178,198503,192131,171837,159796,144930,177676,37144,148517,8825,21303,127107,16428,170818,14989,160578,36480,25631,188486,65923,126566,56267,31166,71372,96853,136538,119961,64745,149127,56588,194119,192195,104703,166174,100609,43170,37509,113681,49615,190462,157730,39219,12769,93889,7772,116119,15505,147782,117044,157583,99776,152312,158957,129591,68473,60212,57057,96611,82357,161767,19104,111063,59058,45378,52922,9254,29010,112142,144440,16476,41643,167432,33855,47931,63441,143549,117123,31842,57067,15464,43280,5864,52001,44992,85993,94251,50344,140133,36675,114054,131155,112449,5817,146680,113725,5098,63924,77756,5967,31064,192187,145400,15806,179798,161665,68712,175624,198286,10554,27398,13882,123638,76257,194886,189642,197968,80845,9242,83726,9809,168067,97700,179700,80594,46312,178264,36289,161835,176921,146158,6023,4608,147330,128259,120910,12095,19323,145901,140489,93873,26694,98776,3436,61010,8887,102777,168395,34247,122325,61686,156897,134416,138952,76067,103718,104164,33556,198858,117987,4498,61939,137505,150269,37318,95304,144805,4087,43535,108297,135414,140674,51713,82201,130649,73207,182575,113014,73677,177809,8759,53038,85648,36907,152026,164634,43693,112641,109526,66437,133255,45588,22573,121007,93724,89629,145780,119829,182765,152874,11588,97478,58941,126194,87088,12355,125480,56029,11446,149399,48289,199400,45753,184438,32333,135989,63102,63596,89247,75186,106954,64381,171538,158681,21878,173396,6219,175844,24119,35481,59096,50486,95818,46723,65523,31032,175073,168257,47518,27329,186350,93269,29508,179501,119107,43260,196027,122334,17766,129544,136934,190744,193705,164182,76773,21374,148353,132213,167508,62446,196981,51132,15987,131675,114250,1439,49404,191406,60450,123509,104089,78769,84688,34033,55058,33404,180693,131093,149623,133847,71747,196179,156786,105706,137347,182900,188323,58946,62076,142569,143090,31107,80935,176796,193429,163731,173864,11555,190631,112556,101349,199297,122302,137598,159597,22622,133222,22049,132256,169332,120631,91506,85255,12322,129427,182716,41848,41160,101955,124677,21871,56438,154964,127922,192479,129983,26425,17509,146858,3725,112582,141524,73181,114632,44152,80416,71468,76425,110135,17963,10007,4612,152273,45074,165918,64838,88322,100727,120448,16562,191994,109180,124968,186644,111108,147369,60931,57124,4332,186856,101876,168446,184555,120331,163572,138682,154978,385,117775,178615,93591,71716,118714,132202,147482,58178,32486,31563,18658,178679,81077,186680,154530,186527,123886,179339,30694,105397,3536,35037,183310,60608,140598,58455,168719,89117,153082,48313,178126,171379,132686,7714,43981,37666,133803,175359,54337,99743,131734,25446,6768,160007,74462,103184,14979,50234,29301,51865,102911,93054,69061,150009,63012,158967,104185,175243,138854,193822,160367,55885,54106,53386,176674,94586,43786,83378,93923,174300,13601,144448,12106,162135,67567,75250,199335,41071,62225,186561,101402,118003,50287,109649,177808,134279,2587,161875,81787,26166,2769,178433,164484,67704,152790,96417,156467,135353,134945,67516,141985,156239,171137,186323,177735,139560,8622,48644,19653,5761,21541,181355,176388,74383,18143,181836,14126,138871,116794,21695,134511,143608,195282,120312,148133,53033,48146,21480,137506,67630,40482,19223,174095,94985,46616,175904,46102,89752,38945,108364,719,31481,198147,199271,19537,16846,62690,79517,6916,193305,124783,116268,3226,191494,101296,15464,8248,9206,164168,100030,144941,131970,68954,153979,151829,70460,40379,48011,135155,72537,175994,165205,57293,168689,58097,36340,107415,21449,63869,95547,156450,31795,194824,129162,18357,14085,3427,167777,33824,37416,128894,12411,140327,171402,131464,123377,174844,105229,69884,163082,150770,161160,149223,118819,26412,28421,118685,48775,167497,30757,169477,28324,177124,103842,59102,125623,58040,138658,9542,194878,121990,180435,35506,132287,86449,154919,65141,183228,181897,85340,23545,134760,89551,85152,164680,186110,34215,52306,151154,92108,151964,20038,181373,104063,178725,127055,193047,181059,86258,176671,28252,26546,4187,97242,88852,138420,173213,118178,17035,149980,27269,31989,102980,172403,96674,198316,173007,181780,145939,138159,73059,68792,73387,128884,158025,49925,64779,146650,162660,107808,50650,145491,77365,164929,105727,33155,23985,3757,156295,179123,38319,163948,171711,187372,78173,111637,5609,115712,191191,154615,9691,87817,46436,198739,170217,121056,181926,105168,87515,29210,166314,20846,68864,15258,164570,58337,140863,69810,7462,192409,196699,58442,1980,128345,133832,2269,115921,88071,42531,116351,95155,111718,119555,172569,61003,82383,105911,51519,147831,161670,66889,113138,63426,56131,23414,153872,199373,164175,117595,63065,134133,118718,37645,54509,107369,100281,95292,184168,114023,97695,175924,97588,46453,51110,188205,31453,149474,80216,45323,143044,38094,166707,128432,182311,9312,52398,63789,144877,55403,150843,180643,75104,151491,149444,42708,88550,128573,7531,119628,183275,31056,65240,67251,83928,59415,104781,115454,56195,151427,47568,73815,181855,167845,79511,102210,71126,148824,138318,183268,156667,131078,174175,179077,156142,4081,23438,150895,50771,199266,68280,168180,108795,91439,100201,193272,37006,74804,188591,126638,39767,151438,99227,9153,172826,171101,51353,99253,26459,14568,18647,125352,56734,151561,13760,199504,102788,119878,194149,61655,58761,55659,30877,38808,37865,172806,170177,127946,31671,40133,123222,7898,38908,78644,68712,154275,94206,9522,86190,182291,197033,76037,40901,26188,190157,4581,15567,191823,10623,121005,57778,115944,91712,63095,57157,78534,7042,181925,53886,115492,25653,119714,69519,112375,48861,2274,147647,55893,146282,124733,182035,92287,161503,98037,160925,197445,170775,150656,85549,173296,119859,11944,18146,74148,14390,27179,172484,197732,157279,182065,74626,115141,138011,106995,166330,130033,191063,101696,114175,115468,86857,27462,122467,53871,55786,193429,45171,13306,112837,92139,28763,75892,90935,8069,100823,6227,184443,10276,26507,191549,157873,11756,138525,179547,166704,184325,10225,46552,41700,106192,130162,178566,43022,102369,144294,168171,99398,137852,18988,199447,38051,23786,185129,148979,119628,150403,128049,9494,154999,136938,99426,117116,102449,86657,10232,181276,38510,53402,52968,68882,68103,138710,86355,138305,129522,16382,120303,175189,20033,109838,21753,13983,159128,62459,7179,108424,2644,112548,25296,36790,49733,131016,43595,153205,6041,65647,85445,197291,104905,59738,53583,92451,167376,123644,190687,66189,174582,81396,180142,87214,63588,163463,75193,45449,190678,160672,77289,114421,36711,98542,100560,156118,135397,143440,144825,115861,174389,177978,24226,178829,38741,166866,84040,64218,199251,139669,147810,2012,123954,135075,110209,68281,23268,160318,66002,149430,71957,159512,78596,93925,45698,54369,175651,22844,60524,116118,147708,72099,31081,78738,41097,52496,65330,156973,45332,112044,140804,151864,165494,26576,101186,176529,99726,72434,7398,25300,25814,112206,61097,100747,75029,139269,186687,155153,36308,184742,190579,142177,112337,193907,191425,199404,42837,139916,55120,121259,113954,116097,199954,17211,105378,52457,74521,57837,38668,99689,177801,194146,20015,26562,98268,100361,44081,70309,11054,53526,153382,18850,6476,94830,185129,146085,76354,189335,78520,146051,36529,108707,18988,178314,39830,44450,195053,104150,35267,168278,82639,19439,3962,137754,172213,110667,86237,80092,69369,156658,187564,105769,107036,147755,35704,174784,115761,172331,8799,72601,141427,7649,120267,19716,37717,106697,82560,40580,101601,124871,72270,90083,195260,70261,120707,114550,163044,156170,115857,17550,97500,178718,80824,142545,199920,22967,141740,102862,58637,129321,48776,34268,38758,61844,131984,29360,52426,66253,35342,88814,46708,42776,17018,127231,86972,134306,43798,154644,94410,57776,185179,168138,133952,156327,156366,126314,51303,169561,94736,153387,151154,146455,3451,43711,1678,21243,95514,9417,29627,106109,134897,35263,49496,132897,147194,26413,81452,172937,162290,119223,92166,178439,94367,159273,196159,9532,25889,24510,94698,26585,13492,165526,11500,29808,69845,150557,194420,15345,33704,13100,46853,126073,113910,59410,166295,101813,79468,61011,177609,177619,179019,72952,65495,42284,183390,16578,25459,41886,128372,133808,159525,193153,95312,138735,153776,199075,78845,75208,155189,141440,29728,86730,172621,161435,188694,67808,143871,118368,102231,139063,81410,147648,42042,130897,198803,125071,3833,67482,69746,16834,165292,117810,22526,73809,41483,110300,186991,185483,125316,181663,69360,171933,65228,181571,60760,30600,115032,180145,148437,91761,56231,179193,27706,128547,117644,5604,4715,193077,22114,34192,161632,21960,174305,123917,138481,32786,155750,26153,19849,151049,165832,148663,104219,169589,75444,156114,47981,11617,145143,168249,105801,64907,148923,40226,134102,105638,5013,89608,162832,84322,172917,133022,192559,104458,3456,5508,107022,144246,22888,180986,164276,38386,105658,58315,95337,71246,175555,67316,133707,185880,190457,66211,88192,177367,186095,35056,40279,103581,136847,13812,31451,10508,135334,47380,74936,40988,98620,84082,81264,192496,72696,77006,180680,40192,79428,59082,33791,76374,164555,117230,111577,38834,118663,160055,29209,94540,132350,155666,19220,109451,34393,85660,2712,174807,149639,15070,174739,147024,42744,149846,67779,90714,148366,106099,102713,156318,83822,30726,154270,58008,151297,87083,6184,143608,105238,46046,26004,28742,56310,90348,135469,47373,183134,38754,25058,21994,186522,125872,193813,175700,46518,83861,188270,157488,94268,103021,106621,160155,9304,90117,132611,438,120654,185713,11272,163986,123939,177464,138843,90453,118215,180627,32412,60385,3218,7920,160664,110970,9867,3391,140898,28672,153133,72597,187570,44309,87826,188405,142292,188338,37817,162551,105358,182920,85004,173162,99204,183822,92618,42233,89191,30773,19182,145499,22903,29355,81681,130501,78566,143465,800,10659,130482,72472,64783,97364,199727,85054,54228,115197,58604,116027,33138,67201,193318,67033,102435,3695,183554,71724,105436,117111,32979,34791,142022,128838,150750,34636,116929,74660,153912,30558,18860,49113,80099,124723,41393,113750,132224,135135,140916,75476,74516,11663,163065,99707,169393,113389,112304,107381,113325,176345,112138,141404,195912,90241,172710,25717,18129,93768,99996,68946,171038,8873,59195,115086,133837,176442,142721,5347,111013,96802,72884,131670,37127,155578,43996,46371,59003,164834,199514,21443,187369,67975,154281,132596,111024,76429,12492,31075,166201,147983,76474,56262,36073,78277,46835,102567,34686,3984,69657,711,133408,159255,178678,135720,112350,179771,4895,60841,8431,141268,82998,81681,51032,157777,191227,10513,2916,49764,128115,106954,30288,114037,118968,40959,75500,58562,91398,16724,35645,161364,199290,136443,67149,127757,62003,51055,14452,98708,99970,178896,85957,3900,96815,152620,2334,151178,3927,4354,14292,125588,2368,182400,47596,59312,105059,178067,31507,193279,174535,147440,70597,73814,143444,105829,107005,134446,34183,69548,10821,90777,95453,147890,125363,84374,148298,15847,42505,99190,16825,702,193794,89019,62403,124933,71064,13484,148655,114708,154572,130033,152956,145221,67513,71695,162431,188639,8540,81307,87527,14510,189123,81002,42074,154640,5481,182482,81659,51650,65346,48750,175984,74716,115264,154142,77295,117604,431,92132,66088,37703,113336,12390,140140,191827,180140,40012,155403,68391,53615,68568,143045,81129,108653,98468,112847,4476,150547,191513,101161,98467,18779,188568,171748,88759,125972,190695,192059,172963,47349,72251,110096,146941,107713,99266,209,128296,108566,117386,99352,92755,108903,179171,73215,15737,167232,49029,118350,188200,63332,132319,4931,19376,154040,38692,133944,92029,101675,147164,86495,42714,75106,80131,105274,61786,117773,8181,15114,39170,155715,141668,142459,108927,5501,114910,185771,88654,80969,20406,41585,13047,191076,158726,62800,180653,106745,78269,5712,197537,22281,158054,81478,175890,158414,78540,63380,189470,106647,113602,148166,182961,141488,193477,53172,127081,185519,101347,7859,70737,78670,140244,114052,20857,91589,2222,92794,179170,3659,20770,116891,128615,66283,134863,116501,67949,161305,123484,97895,85073,156255,74376,137802,128843,9717,41612,170960,121898,106115,36118,44228,128442,2703,67413,56440,125932,145293,159300,157243,199511,185660,24859,179645,34383,149410,58375,118002,122212,171699,124210,97919,46741,192570,106005,150872,49417,104676,172830,129017,57238,145342,114880,70972,113788,132118,183677,11042,137345,76871,43893,118717,86535,169601,33919,177341,111635,102020,28959,178203,166339,43612,143818,2768,123658,197650,136535,198344,137877,184710,69279,39463,140168,195614,99745,41239,57653,52755,65295,115780,71720,105285,148144,90990,79590,54772,7281,10675,71466,82057,89210,158005,149267,122141,98310,169661,73344,24679,66819,39313,141343,62423,186177,192253,35595,92041,131827,143335,171313,123255,96882,179604,103109,151661,133914,16276,68451,112308,176580,37140,188714,37490,1105,27368,145749,111272,51413,56574,39921,29785,145506,9547,166453,170720,115564,7605,9712,97230,91088,191109,46671,4171,94699,114996,192783,62707,40995,100608,136786,151112,19481,132105,196998,191409,146845,49999,159918,37271,99003,163237,150656,92314,13691,57029,45466,196741,164972,62257,113054,167114,135532,43617,92286,73687,6658,117031,38527,137398,109025,154213,41021,121196,58086,156842,86213,86456,107312,154786,144010,83808,51076,138433,117266,157733,132891,100680,104649,146229,107109,166709,197565,101790,114979,148301,15010,136593,179471,158322,52485,46190,180859,95281,17386,43053,32466,130181,23133,23964,116410,116962,150484,161233,82131,32816,90654,21271,29198,198041,112704,4852,52220,136656,63821,42224,84267,106685,54736,193402,145013,180486,109084,80421,116883,94533,66880,183091,81851,186760,184190,149974,4819,133174,25242,79242,10815,45864,113673,72156,63986,2167,24019,94315,150513,126896,10806,3490,149098,129932,145461,21362,116445,159690,97646,65705,99385,159279,122202,20271,165788,145235,86703,46365,187740,174989,147956,183259,69779,1676,152301,189535,181408,79408,62450,83925,26386,57126,72876,80464,29150,137449,72890,125771,15535,183052,8041,108473,81943,182085,174490,177321,108338,18095,86587,87396,178666,105994,93537,166495,106557,176615,31706,44029,172031,42587,160566,87464,114907,143860,99692,156677,7891,143335,4737,37718,82021,114096,15867,99963,32057,96326,174403,167268,74953,126603,112048,168191,186383,189351,28868,9903,63859,13834,40396,142316,178124,74311,103852,90079,126542,29904,89440,167753,62104,37257,62275,182286,27002,107673,167311,129138,83298,160029,147697,151909,173152,65576,118784,12953,46648,15276,58162,47992,72629,178082,104208,117208,127347,7953,23042,79584,90364,4328,8039,25496,155077,110583,65555,162750,123726,135572,181168,107011,36265,113385,173448,11519,158649,60879,84264,68617,171469,187016,35242,48692,75195,139055,174813,116756,93748,187948,90029,159569,76207,15563,80184,154158,141369,92193,762,1140,177634,179110,149851,16901,66694,171242,130874,34388,197673,14861,106703,165915,59680,156172,51719,103727,133068,178802,195896,96503,84375,166841,132187,63834,148235,41181,65147,16029,167032,19627,31971,188702,97880,43412,68601,66448,17649,101464,47159,18842,131926,24306,185141,165819,19618,46566,188247,72296,90998,159943,66970,25433,151613,60956,85395,77689,67432,180746,191334,174065,88262,72242,55158,53190,145299,157815,30449,69982,7632,32967,99693,51704,59940,12619,142701,56993,322,175666,13917,22926,35206,177107,5843,125333,12354,75918,62530,170572,19578,105818,12733,76970,44853,146230,176866,29124,152373,109432,166156,131339,18849,184179,52093,77459,163472,128577,137105,42456,195344,80894,31650,45540,161684,142804,188564,132311,11170,28861,27357,113509,166174,131138,78113,87182,139535,155723,118158,26350,163980,45702,165116,14915,88656,28437,69299,9021,103754,90611,175492,43462,3427,183277,100862,125585,19516,163625,165001,48879,137454,129894,7908,43909,108809,141628,130461,53385,1412,58849,48301,9298,64048,30367,16792,82170,199341,116560,81555,150758,84061,194874,2744,104059,49749,10789,63106,37737,95483,153278,65914,64070,49184,88893,8046,154664,97809,86176,129602,82208,161346,170122,18641,184981,57577,83710,162704,58942,160761,8097,183279,65921,34233,184397,50128,91421,55487,153028,123956,134729,130017,176944,104147,104104,124381,158476,34327,169520,43400,101440,31886,64487,134895,5442,47716,128869,17650,97350,90617,153943,30573,30052,157078,113573,107912,163676,97967,68266,152849,1343,154396,32277,100478,102266,173163,18313,19171,7676,7020,60770,199846,174368,69732,98945,187638,140621,150284,172250,44901,181514,114751,60711,3337,98821,130232,120059,86339,66726,75992,61461,161866,149749,33886,137867,186486,180007,106723,155075,18629,25476,6099,78347,52401,115622,634,16691,77787,51835,3986,187440,8019,124349,114318,37183,169734,60506,163750,193825,115492,83327,109687,144136,24200,133882,11220,169090,197129,177645,68737,7621,44650,100150,71806,109828,107422,29503,80706,86512,105099,43568,98533,133037,60003,188380,172512,22899,54099,53027,37842,117417,198178,62760,76464,21212,171994,17130,131750,74768,104385,10864,111513,171047,4714,171726,159365,58937,18538,18123,121322,84636,127871,167440,195412,5533,30516,88014,132977,104591,70727,39903,95161,187562,199333,6302,193060,126809,149938,151176,93975,52796,171614,51468,50697,14779,132562,131709,76896,67550,8958,57650,26239,73604,64212,181951,187264,11365,21871,82657,82889,118074,17280,199165,123888,52156,145339,150411,30019,99762,147834,62078,100971,44927,66007,112033,93923,54971,115972,34243,143184,157531,193588,97538,168619,160615,17883,104957,96956,89488,102575,61023,177995,2797,44591,154365,90981,168690,110483,407,8319,27301,74559,122105,188880,44809,193686,96311,18643,66667,189903,70049,20200,79655,175590,83702,183048,14229,143827,85691,50385,168742,122132,16601,51916,187166,80875,3952,164219,93114,49061,127481,152727,92892,125465,156686,63700,90887,158653,90477,155448,72810,41409,124020,28810,51924,13116,174514,145533,51439,96923,11036,149719,69886,134252,130453,12715,81737,141936,49288,43853,184983,31400,159427,91122,26667,16867,146037,124329,44348,23104,154727,11747,61703,145706,86494,10400,79000,178251,109462,152170,132767,164915,64960,78165,124184,113123,158357,106533,31877,134565,174139,172686,64392,104315,96250,14329,34679,38330,50869,173808,15578,8986,8919,176649,99750,23352,182896,182252,140194,121028,120942,198677,156212,126745,189074,109224,125184,73201,5875,128964,198130,123249,67512,77036,164735,27043,11935,114064,47571,17490,175151,98698,128056,41973,167563,8781,828,25686,150750,56290,182594,76352,90143,63325,150035,89422,86024,52477,56599,198828,105977,134342,148707,70851,75429,175515,141057,22344,177826,150807,123929,112774,98714,79387,107679,106823,79197,25278,36039,44048,98322,81971,80458,115916,123036,95945,193639,70068,76162,154654,104242,97412,125637,100399,179492,65722,176887,84634,81009,39598,73279,114278,192347,108764,174704,15960,176072,53315,136561,147195,195871,149024,142480,193034,1588,15939,193900,114513,128277,26885,172549,111247,154486,199588,79462,180046,165566,100172,103258,166780,65880,73610,89669,15257,166576,131692,111178,22467,23372,180410,74463,190088,175838,17598,4156,51974,24448,194659,72865,69895,55973,100429,186632,32772,43368,166210,139326,164753,77201,167209,191605,85367,130065,61876,186214,178956,125157,145288,123362,23222,170994,90560,139446,27768,140856,17616,117166,149647,190465,80403,67337,17318,65703,110298,69693,1439,162919,127352,63742,30217,31903,14628,37976,130205,118086,193071,38179,163281,185224,190766,17364,105457,111918,197921,78373,180411,165170,94455,186825,73158,36451,95101,14671,64994,102280,185077,7713,150906,157863,153547,73533,171839,25412,190119,80737,45276,184747,100839,107933,187849,67725,75603,45640,96421,11692,181680,141179,44028,29387,168397,188952,124209,167651,120531,193865,950,146409,63388,125131,129021,178921,170001,119499,67461,196938,75192,122874,62208,85536,132144,71792,31862,197590,77027,111473,113814,129647,68656,177708,176175,143958,102671,136260,157281,90277,178381,18834,9478,163215,167510,9872,161939,26161,1709,95832,58620,71430,56490,172544,107792,167366,86296,194618,130139,171246,2805,94493,117562,74802,196603,161960,46951,130835,33198,144600,196754,92595,127074,87258,137580,21906,183318,71491,167309,11665,195192,37642,199804,56374,80843,131229,105662,156113,123856,1290,14733,11930,8554,84199,95885,143275,112840,133371,148480,199012,162051,24291,163967,80488,27210,48861,43213,88348,46585,112086,65061,101458,187366,188442,100394,103464,198262,183441,39195,88775,87421,20853,82822,106239,90380,174022,130266,150802,73686,32105,161697,131085,138256,170733,151046,22959,118177,142518,64298,22229,72163,69438,30009,130360,168816,131381,7244,73749,43270,3472,111311,11959,68379,87950,23240,48018,10262,162926,23869,69360,185278,60818,97681,156575,81600,38532,186788,2995,175613,106645,91534,59105,150985,191576,21604,93067,101942,14660,107108,60905,22318,131542,186924,53458,122438,79686,172131,19026,160184,187945,99518,173587,131422,39914,87248,100980,171727,158304,32986,47309,49523,50869,34030,167777,11387,136673,89614,65536,143384,159553,115033,19091,63986,70888,192526,155750,188964,68502,20482,134581,82983,139765,162822,56680,102295,103755,111473,53775,163616,149654,9622,163281,61024,111819,53357,93777,67043,141783,40110,57367,185890,68290,15199,22691,181766,136860,155760,58311,162752,134376,10128,134312,112904,31037,98350,146140,127201,27045,178931,62352,13677,130176,27306,163223,186657,15270,199085,80557,94581,20414,134007,37412,46731,155162,47266,137975,176268,107150,91649,138209,184636,161106,185704,8827,87518,99493,125462,44632,147656,89186,50973,197238,165910,91793,2428,23053,44215,64195,162155,96464,115758,146546,88308,76232,138473,194670,61389,97023,67783,159379,2161,3009,150303,53527,181670,195362,35513,146163,167170,23410,70364,122163,20141,2653,41227,158254,26376,96372,196406,97965,178876,10804,193826,108739,162612,180539,71626,161627,199778,187656,144044,101462,129533,126936,117546,135760,89459,135259,32707,129963,34007,34317,102355,128217,70422,51102,6531,115098,114914,158465,42065,105496,190226,140712,142865,167970,74652,112306,197633,130499,182575,60344,82800,132954,48202,143662,92411,123745,84757,195547,6560,100780,8253,97590,157429,94913,95676,138906,79199,72563,149736,185705,53562,60523,78491,50370,104136,130899,124902,165780,173146,133391,117170,126523,169037,35933,37869,135541,87988,106631,66221,116655,66429,12022,130582,112094,29491,127678,180883,183466,165678,119050,146604,45244,120609,77335,74001,197394,43868,52284,179224,144746,105170,167909,24983,10406,88288,10339,15477,73350,171966,4850,18628,149162,170004,84543,115363,172806,61017,2116,150511,85334,113047,151654,69663,24334,118575,195303,28494,80893,34676,141383,193328,37172,124673,189691,187691,52702,75844,60030,43535,55733,13813,145685,102355,136095,194849,41764,75226,115494,15451,175417,17134,19956,150702,36662,94940,144507,78713,115279,126395,167111,165346,84326,155081,150433,122232,144681,139091,17825,17454,95149,154484,32344,151440,118596,2030,24040,65343,65877,110440,21165,112178,180355,32457,126282,2649,127985,53226,82240,123129,99221,78200,114850,164860,20089,1702,186241,76909,166997,39710,144753,51078,55042,96850,100093,32869,195578,46120,92200,68894,159930,44176,153149,96686,126325,197080,39186,95815,3371,151971,129803,121471,143092,10556,83307,96779,39449,104144,144616,109779,57556,89896,167569,84202,171339,92958,192168,185951,6174,163231,52217,29585,105054,37937,55301,106562,107771,63622,49015,195058,138682,86121,171029,192309,29708,70406,149364,162354,119779,3471,17076,27528,63209,137250,144334,179827,119607,182337,162828,187575,103118,164227,9469,7624,115440,161784,60440,135320,97002,158323,137200,128519,105548,78225,57278,187202,158365,89897,87529,91139,55935,22640,177750,73036,28285,155634,151036,82446,149068,59409,192996,12719,173657,6069,89359,139516,18203,128465,4093,64606,70188,150867,152382,19528,43203,161762,50976,32127,112152,149337,75342,72760,105162,88822,186452,101641,169798,61163,114215,100972,123894,71623,119332,40686,8633,5242,44613,172239,87246,12840,14581,130877,106493,98126,142006,49334,92580,108550,169312,124803,49703,165200,81093,141518,121009,81770,121052,165522,181683,16788,147278,161741,90060,160609,170554,175621,72452,89938,118862,25778,14613,162091,182018,184289,121781,19772,176958,92112,185875,157980,199251,125340,115812,81184,42019,11651,597,189723,20170,149683,119378,104441,198482,151767,41350,190826,67082,27816,57428,189537,136026,17352,21266,97658,183503,1973,165583,67876,162213,43683,179245,147067,119570,159350,56259,171403,79790,153051,198271,99057,134074,59206,138104,132769,55150,95059,32076,27915,183952,89382,165382,143124,95914,129256,69328,77535,115332,7490,79408,56431,178507,114944,189416,124297,188852,176877,191888,42165,97424,35544,124884,31897,178899,53325,74567,97970,132367,118678,142679,128289,195571,35944,43722,66770,28321,157847,171762,60912,145383,119238,35097,132110,1576,2730,189819,106091,147533,74148,89056,198691,126097,133996,124120,176516,162270,108834,151196,142398,72052,34409,110310,185214,140758,126887,171509,166168,27929,51078,142616,40560,87174,98370,21120,158674,106965,120571,7441,106464,106037,135645,93682,163619,33193,67253,180844,83938,55777,129225,64751,70677,111455,138369,141592,59823,77821,185191,149106,40638,88523,31072,95304,96377,122047,21646,109365,189818,138187,169383,123435,107151,107719,15595,69767,153936,191904,163491,1687,25850,326,171582,108444,114129,135764,95916,62755,110489,184695,144550,16401,124446,96925,191334,7178,24951,36927,92712,76089,159683,152212,160702,100847,14264,42047,162151,104630,15905,80849,129853,153017,77758,95028,126783,172527,56327,38240,101084,62399,103065,138396,175201,71197,188402,28749,1897,134123,80920,46047,5748,187183,92771,166956,136685,63430,52268,75121,157759,67578,38805,79492,43480,123606,26130,4695,25347,69618,196789,12612,129769,167149,91671,134998,52975,140692,41007,132020,64057,35688,130284,3248,13541,103760,46515,93222,76123,134188,82773,14647,33338,93226,59168,189682,50718,90734,116469,62860,54946,138701,194695,45254,187298,192720,197953,132481,160710,53540,41089,156750,7638,85824,168721,159322,128275,47509,147243,189876,181629,9162,144886,127621,199788,153681,90563,35041,30048,105740,47788,193918,125616,79882,88850,105594,167863,86767,7429,124843,182132,32687,195973,67298,110599,123558,156590,49430,133981,196787,127954,154480,157038,184584,190126,102329,63437,159111,83186,32913,11712,82350,103679,36585,82580,18727,96069,177041,57476,102241,133980,194593,5701,52346,153230,182357,65909,16419,85565,145982,194982,73262,113078,90618,182345,62054,10704,191026,33953,25064,153002,140763,4333,150087,121284,96993,7169,69306,127641,175362,171631,103994,124991,190235,26190,74043,12368,58678,32047,51772,143259,130951,132901,35028,11494,11153,22357,8595,112981,130031,112013,86032,130038,114105,184138,166380,31625,73436,43417,145758,133510,181831,111618,148398,99734,1994,64797,892,72756,19583,69686,122589,65056,144634,134902,7407,192668,139881,108605,62345,144120,13447,71926,171404,86383,198302,115046,195505,58337,195674,44142,56307,64178,10946,184374,56693,55456,6216,194806,70576,37892,34003,175779,112711,141687,121469,197954,29052,120523,8533,71258,92627,157259,165611,161418,97902,176035,181912,13438,17209,96813,24995,110845,96853,67525,132735,42868,153578,46173,62894,53708,21317,107383,59974,31230,114719,66541,106254,182143,412,195063,84439,88247,91032,89073,102616,76740,192286,155598,20708,107806,127893,43532,127910,36865,66151,69135,133624,62648,100395,14662,39220,158409,34721,25728,112676,101098,28994,52133,54287,126969,185385,179937,48218,50646,138114,153175,154953,103552,157027,177422,62277,136008,30101,197505,25530,189158,108337,57210,140998,93353,60297,111996,54673,4281,17562,88696,93531,72053,140799,11138,119705,176762,188995,67254,112908,132337,146933,9565,131791,155553,126882,105884,21268,15487,67687,73535,81134,38933,103474,85981,105667,127300,83705,1394,79859,59608,174462,62762,186946,188076,38063,152193,1523,73959,45596,75581,84187,71299,10419,105123,182657,66512,23198,193924,119045,4822,61583,74170,171937,198523,96747,150478,90605,94967,175253,31245,35366,151340,117337,111330,95312,177368,32514,194521,199755,77220,32895,184758,122885,128425,36536,182185,109930,167757,121643,137890,177887,194922,120819,191717,138125,191765,28300,146372,57249,141669,164019,912,120567,21026,146508,42559,30605,34947,147709,22800,71537,150054,70481,4286,70432,118425,114628,50379,151542,98585,132909,178987,70491,23823,87886,181233,158902,69109,140450,162334,20962,180547,174672,157938,188947,178835,193257,192569,67361,162611,50441,21691,93622,197643,122049,194763,185760,114962,81932,23588,23156,121845,88443,86336,160401,198570,42139,63689,15794,94366,147867,164480,23582,125259,46719,87247,87524,71357,45239,143330,10025,118514,196736,176433,173779,5498,121390,199411,33720,142594,150154,183498,31243,100061,158733,145223,17145,82325,158536,157319,4348,190449,118565,76286,48667,199162,181184,47392,186981,151082,113289,151927,26160,85628,199143,55869,3633,169679,199359,9010,198943,152229,58932,165463,108419,18229,48274,198821,164621,41434,66831,182340,198877,55364,159805,18795,84174,33024,59716,16588,129481,64100,95143,53901,22924,174682,112585,124517,192427,24989,98263,67191,67044,180955,120744,32464,1063,137150,138797,93799,161274,20949,90315,97757,12502,23714,167955,50443,160491,53673,140542,153637,104009,166410,46680,103137,32608,44128,158957,21745,23879,198930,186718,91931,189378,98029,167588,2810,127359,108544,2183,76853,74052,93491,40734,150480,179910,67826,114615,87967,3788,55696,165198,127514,100961,80358,85881,134116,95241,25907,175690,103495,173203,138095,86465,5260,26238,122258,130450,15316,37017,83696,17996,179873,133049,30814,109436,35275,10630,156445,117402,33224,83101,124458,170534,187744,110946,127474,187047,134958,193254,1722,10412,68585,15812,56095,11154,50343,35681,11077,18025,41969,164380,116022,20548,45552,47396,71083,37108,178777,109276,184030,185945,29047,189087,148537,199305,156047,28596,129719,184832,180922,59240,2708,174625,161243,82431,56609,187872,59698,138405,111354,154087,168087,133377,162033,76180,93244,159703,158759,74442,40864,110878,87175,116173,97865,29272,3682,171946,19918,163531,58958,169450,189618,193836,146747,106125,35664,79134,188439,125229,171024,119408,112760,117921,44323,69506,69827,132278,70893,2993,80744,145243,106631,8862,158484,88763,39358,19824,143747,109520,117264,189211,14559,18123,174090,99274,69872,80704,41682,107302,78986,115819,135314,5854,72233,134032,166701,132194,191621,151329,33762,139530,183416,75981,10484,15399,132365,91169,130585,159290,70419,167309,40718,198176,83740,65358,27795,79932,152194,18014,36201,2098,24725,150316,128005,155851,82781,104596,78122,69328,124833,30322,180469,105199,50184,78125,119334,187020,71825,141577,29289,93946,167487,60089,78214,146551,102642,84243,196779,195712,96805,105870,132429,155591,94717,182689,114720,78405,76906,153661,74851,42842,97176,67054,16275,160172,111944,40992,5591,3592,184089,28854,106951,147961,102694,103347,180543,135746,194288,38058,176692,126815,184504,64781,24361,90628,105718,79918,86837,19980,44785,158685,103437,15207,56726,79191,126093,11096,40311,63379,56323,164297,11954,147064,32941,75983,2654,136659,42528,49402,193917,46303,23750,83595,120273,174555,18564,191273,6148,113707,68732,194119,164778,43080,101120,185051,86440,142136,22116,75641,36108,15938,25162,199072,156204,170363,197320,178319,9361,151734,136786,184131,131295,54637,39210,50805,134492,163077,114262,103047,106332,142393,74229,164703,123448,13871,104763,34941,192654,38595,121844,121833,105466,167035,90321,46902,149475,186160,2555,116425,121384,128796,99298,152807,39657,180355,12979,15424,74572,175966,104795,103845,191172,106351,72062,55830,170415,32009,108630,110002,50839,14870,90015,132082,160265,198675,67586,71939,13396,115160,14994,178888,186518,11057,167578,174585,24470,74080,133080,57131,27028,55541,114092,58145,139266,155229,198015,198566,129227,81693,86997,64974,138099,136308,135444,139994,48335,31094,153061,10348,33502,148167,138112,154928,186328,161484,60418,67729,94264,38745,178864,11709,184119,105318,187624,134190,122031,194993,96961,127783,143728,34211,29856,125421,160892,126564,113007,6471,191426,129143,3523,21638,55330,14624,155825,184984,126677,198460,29052,114363,174313,198403,45371,96475,146653,161347,187114,106606,10276,24292,90334,33988,39712,5995,186082,141798,49424,151243,98832,24105,176013,25531,6736,126151,118931,78218,101022,39961,101419,73150,170003,91976,16375,6613,134994,131140,128704,8191,154927,119109,63889,9044,178780,135841,46913,164292,132323,140190,198724,66558,164366,112898,41629,165531,47540,94726,167879,118105,138262,13460,89300,146918,20812,182059,155435,47587,15875,184608,194259,21759,177069,31221,57535,118442,94193,71002,90304,178362,29863,104585,138132,150729,20365,3669,51437,85103,89540,89984,66979,128634,182734,99661,162709,33711,76614,129109,151533,98954,52068,186247,41890,47930,101702,100977,47665,79189,15566,50196,180181,198019,123067,193718,35254,49383,158521,164496,3400,63238,190573,161930,174104,145617,88887,17364,118158,96092,87029,181205,7714,158149,160799,17149,14750,199665,3444,81472,16555,137334,151355,158126,31653,25424,62344,106156,83507,6768,190981,188452,112856,184079,149295,166365,158229,110064,12613,177214,143449,134590,20734,181360,25468,152148,112786,31264,100724,40339,10246,157894,65402,70831,70448,94555,72695,15561,27169,140973,60396,1371,89393,74746,21909,143095,15764,37027,135156,49132,133688,153690,198556,46807,68941,52739,176633,129401,188996,116328,55096,109103,99780,16305,117513,79536,139365,190441,179762,24869,30325,190509,122255,33080,186850,138948,22047,89818,175446,43725,158968,123160,120290,6784,74750,42778,188322,175233,11615,68265,42641,67297,132911,73412,190103,26894,56191,199103,133623,60868,140014,43072,42681,4887,37833,62857,70681,101292,2178,78188,62314,128913,154229,5656,175372,33412,160062,192287,97730,111821,147450,72543,169294,9148,118159,197729,173979,40433,172485,16379,141258,29479,63609,103124,171188,12335,80674,107494,33608,15004,50758,124943,24415,59785,168414,64922,30434,85241,102759,194201,166425,115034,124845,176260,129349,75984,21685,10138,157773,89142,124600,117526,115564,20006,196928,15123,108831,165815,96331,67874,144548,108285,19256,145472,103133,146162,109915,91823,184616,173728,143702,86698,105618,36607,117827,96305,176040,165873,128053,61598,193275,194737,16080,39918,110430,69462,26042,100827,1329,177392,55925,38501,94776,177131,50928,26379,45431,153921,1351,61423,100255,135833,196742,147686,18178,96195,11660,81445,94120,37173,41557,101264,94562,37503,185522,100106,182598,57612,80190,49863,64086,125855,49788,131780,58709,96106,131666,67801,26812,185428,143340,189930,107924,56002,13486,79686,308,168156,72088,186888,196523,195943,72235,175264,30010,110929,50406,113395,161764,93772,188313,127572,181708,121638,24791,78314,161813,37957,29178,30823,89069,64013,152153,7846,13861,135845,196142,83609,28183,75739,95787,50666,12673,136284,170172,83599,115954,77876,8999,27878,2075,53106,196926,120072,10927,163027,53055,70351,123242,105006,86809,183421,37590,28119,184372,57245,91553,8445,39151,149686,52026,149823,88406,17796,101288,118130,106292,145204,87124,123833,192447,153638,80485,175781,137918,55403,62045,90531,48859,35963,167736,49460,4290,21186,163863,84506,73857,115843,120197,45731,26439,13658,165324,86824,14482,175455,108680,31142,142474,115623,23257,59794,38955,75182,21485,24655,95143,67351,3115,14845,47431,72247,144993,199136,131784,147050,37117,75478,24350,188197,68556,79684,52292,93363,7894,101327,149430,102071,103876,146179,127085,41533,188461,4454,135994,99509,158945,93840,71210,169311,25919,37366,97816,143231,163958,154015,14131,101428,80116,183742,197727,57069,16940,104244,88290,123955,114636,146757,63576,149158,164482,175701,21320,97207,114835,95787,101303,136269,39526,125223,19204,88576,166739,106994,97530,76014,153352,37776,199974,60488,12304,48931,36606,189496,150845,80220,7,127338,56268,153963,82982,167397,104798,39637,100910,90616,94704,156183,140072,199993,192445,5143,57548,183310,133430,96846,19752,22802,61164,81621,156046,2927,40715,112155,146584,32168,117426,104488,147851,157770,28374,180094,105626,167742,578,9208,51247,186402,186575,102201,78145,132333,57806,122010,110104,30812,90877,199018,92133,193421,36509,41947,47311,93415,79179,45888,118883,33306,193322,68512,179973,133017,156752,38547,132295,81999,163348,188415,43510,187634,136579,88491,49221,49763,12922,172776,60564,21539,29606,100175,41117,118469,105201,33247,196719,30196,35453,173125,62123,188938,4433,62535,192614,6820,183300,199319,11150,61824,104789,63342,85055,101562,28472,132148,34540,20730,51419,158797,194583,7819,168635,154910,198794,182202,25546,152695,167348,128446,126458,183424,184993,114647,60204,51037,114660,11627,178374,100181,3554,152962,198741,185178,153201,52982,85206,28799,195357,165617,59058,84362,116448,28012,15139,104033,194724,30784,182348,68887,40004,61177,154961,149582,63423,91567,176673,165390,145220,39471,154662,188024,150033,42835,64648,36274,195451,77228,43971,84503,93898,34623,143253,130437,163325,140611,21044,45799,121570,175012,69122,103110,191927,191500,76058,69595,149035,117272,198985,48007,71673,124296,43820,85415,136237,172636,1940,137272,44843,96037,156309,36354,143403,199713,8125,55216,111881,68345,134103,70104,196928,88491,37539,72032,189616,141055,169750,79182,140570,151520,130323,63372,122575,98081,134169,75540,96453,155561,93601,191643,158788,61124,30970,86284,97268,95246,167199,19032,38247,190773,52040,36389,46621,15984,109943,149039,44574,106966,152899,44895,138284,174088,170483,4932,34131,161598,8367,162286,50210,193162,29558,73133,14140,177845,136878,96594,178637,138963,77449,102173,135389,46629,122030,141223,148578,134993,51171,162515,44528,17813,81237,188632,96422,60356,104218,53748,120903,90385,139664,145450,10841,188410,178233,185320,73761,67122,70029,150091,186064,151169,118666,94831,196674,50970,122820,137037,72037,67594,5287,26811,78697,134450,36181,199140,150346,172711,150972,150624,162295,196611,111659,155803,85950,187028,153670,119787,149543,155034,27431,195135,73058,45643,174744,155630,133018,38979,109156,185774,74625,895,80198,170677,41374,102837,31638,155304,183667,49178,88913,131032,186704,190227,105183,122609,165377,194058,124941,184999,50966,19798,121538,133995,76235,183624,69120,127856,161008,55517,150284,198576,171678,103189,123676,139884,99195,106281,124456,129534,63832,40965,37606,155059,128309,116178,177708,142641,10738,80878,106042,62428,151324,66061,34917,24628,101944,16148,157019,18188,118967,126837,19647,120665,195328,187356,117662,124269,63814,103312,81315,165123,170639,149043,59229,46258,137388,99176,168629,44306,27453,189096,41075,169203,36901,73344,47673,44119,20861,191355,141419,79427,70461,18276,10681,37601,14936,63895,37035,139787,85484,30216,25043,7330,132169,128122,87724,159723,124464,167861,59404,99379,108385,58673,29035,140207,165025,33405,129668,164537,119651,192247,101656,24856,61211,162442,9268,10823,118116,77046,127390,33245,43436,14745,153006,1489,161948,195725,142207,7721,108376,108370,37890,40788,58614,43470,80775,115044,142068,9849,67568,48285,77853,174840,182951,125079,55802,82663,50423,46732,107244,116888,78961,45799,8699,75270,121325,72726,10446,118605,180828,111168,103498,52854,33871,140994,166523,163047,146671,101452,119756,9189,69215,89417,111880,172799,188217,139895,173115,140710,111884,75718,94156,180650,45596,18969,9222,93203,172645,126356,132672,190288,115413,101264,88630,185547,35105,45460,163336,53043,77409,100640,116758,50376,6917,82558,183155,182169,68038,181761,94678,113926,33354,49467,25633,47053,159156,188292,135920,940,161486,175688,139428,57501,101217,30274,27918,132382,116771,166396,119032,41478,98100,28626,105825,13254,175282,153811,193674,26181,173206,120505,6018,191537,90335,68192,196837,181494,27131,164328,185105,100885,76631,117660,30018,195577,169533,107571,199890,56396,159140,83411,149190,1629,119565,129134,41753,137501,102333,117164,28778,144594,13975,33338,67435,127288,103355,119109,120140,12125,178219,156417,192277,161902,67242,56060,160781,130293,83683,131130,87607,169741,149321,132862,134703,156861,135689,60314,189379,171268,53333,134341,57543,121303,68991,161661,87763,57025,58793,17223,75248,143358,197680,79466,192751,129479,147028,15265,8413,24712,131585,122391,191236,24459,178576,135922,46443,180035,74485,100185,17977,74058,160005,59754,181077,163923,46797,154680,48887,37244,88306,94733,122498,30180,9971,179884,11515,155837,187723,114131,88361,16011,198929,95412,116985,55859,102058,144759,180884,174555,131532,185633,113459,177163,169316,81162,156091,161167,32618,124508,161900,117569,24502,142108,146246,16245,87648,104537,153338,51666,79832,58558,164368,98555,120315,149855,61894,114479,194992,164016,191904,58402,56217,90624,65178,1036,70342,11240,36658,59817,134789,127069,166622,40535,132980,74450,47638,110346,150993,61990,13541,198576,182214,18830,36473,53554,184318,32310,133739,3913,103513,87507,139056,16467,67011,158961,39578,183528,162733,18790,102817,153848,12796,191012,159166,164837,173379,104436,13320,148658,191591,54284,120112,81053,2926,121929,57479,169271,44517,129602,80965,183730,124249,63053,19023,38453,61341,87198,94980,179844,171927,20609,18746,104147,199891,2120,25200,181621,35652,169636,6647,48864,197659,28104,185131,28861,44556,102631,75612,163849,23595,151527,98539,128892,92184,49536,27487,156934,84491,135218,115022,100869,165997,101727,37584,162382,48888,174000,132709,125062,67779,96377,168988,137477,163998,57834,162044,75977,95583,197254,4868,181480,13924,58785,90442,140465,4935,13575,18267,134260,172978,81344,199998,126923,155396,109990,39488,186052,174117,180120,149398,43368,109354,129518,28566,82687,183935,108560,164245,104157,142692,29602,33097,108054,56187,102180,193927,161573,28020,83974,44298,40876,139289,9697,59526,97240,186457,28610,197152,62256,10161,93819,67323,140695,29214,114522,193257,168383,53444,91741,127708,10600,76721,112154,42310,160749,168678,60968,188540,136996,5369,77695,121585,11498,29378,107397,9063,36997,194918,96953,88103,17206,188764,117384,18097,167564,180188,176216,170008,82412,170839,77024,198951,100914,139762,107692,86813,196425,197240,20986,196133,32874,39510,2610,38337,106888,171697,168291,30560,18911,88372,6245,16123,164990,27645,28016,188658,69571,88844,100132,46686,49148,149783,16649,15662,73876,80428,143602,136927,111800,56567,19404,95819,163808,62165,28994,30182,68155,15629,125620,17512,148662,133410,68933,195146,71748,150809,182165,16113,6297,13467,142489,173132,172413,47919,68228,27760,117678,112141,72740,184126,85008,54803,145745,99282,86140,67154,148817,198994,71688,28847,188013,91595,144332,85703,147640,188567,70357,153653,29478,196912,2958,30999,117868,199399,159427,23618,147624,43186,153221,133678,26330,121407,161760,68888,137877,128968,152905,117435,65345,22832,169046,12039,106190,51673,103878,142625,80078,60630,154199,156056,45085,176671,153343,91652,119485,185542,195060,168009,21721,9269,136353,76320,47872,132586,71607,194758,143052,187963,77857,72205,57552,41994,152583,167326,91344,41052,23500,18529,80324,182693,88615,146464,122735,31937,192525,66505,116079,199558,26509,78746,179750,119176]
    indices_val = [128126,151041,117144,161765,164474,21110,40787,176882,172079,143664,83351,11319,98211,167396,188454,53865,108105,173109,86793,34930,137520,5480,193193,119064,145283,89420,135656,154322,147650,175974,59531,139414,25850,156121,178049,60087,22345,37884,124689,35419,161759,57846,88829,3282,75725,161488,54250,5215,5200,61121,84226,7816,53143,31518,36093,10231,1571,47324,184889,102692,50458,143523,114072,51946,134630,165133,126183,186252,185665,198618,152750,166033,46579,192244,43362,110555,182820,113951,66166,165112,151745,170694,60075,64456,30572,66302,138454,135959,178510,176251,81053,35524,7580,25007,69832,196688,196221,12772,156955,156691,32492,94599,32484,43640,197692,102787,15854,115843,108599,142902,197267,20077,177375,163470,157288,110779,11886,163803,156076,106276,127221,6172,145877,148857,199527,39961,149358,90723,123858,176770,152863,180257,161375,88331,67677,43769,114107,158031,142243,121264,159327,117846,20038,123495,199352,62603,199207,396,129651,164235,129952,58858,153830,23659,130768,176200,136949,31957,146752,153624,91278,190774,88190,34259,59959,74832,159608,47148,15150,166102,69051,50015,25870,26191,921,138956,69251,115012,97108,53375,146068,160392,53151,147204,49645,43479,80102,38757,122703,42569,132586,133149,62251,35718,13075,182009,35395,84451,85254,128685,140481,112482,129820,116042,108343,89709,102506,10811,82078,39246,151810,36074,41804,102442,192668,96278,55732,42961,195559,116636,80550,59164,13780,57444,177973,179515,16309,1165,3069,133186,199234,96954,68657,199132,12709,30759,150101,81832,26837,30562,24189,173965,70940,179052,142274,24583,116520,102905,6162,178834,188804,144091,363,148872,33508,170373,180851,105932,126767,190573,133580,3793,76768,38341,178820,14690,71553,156656,13135,166010,1863,127955,128377,14092,115052,58539,35963,191885,3912,156005,85378,49601,102950,96746,87192,175310,126705,135864,152931,95060,69257,129067,72332,32673,49658,34041,148432,83409,128001,36905,192336,168676,195537,192844,127112,22797,154534,189009,163829,46327,148524,123622,90488,167958,186571,179625,121041,3199,119828,50866,38585,91223,47348,142740,17369,161937,59159,89825,162315,126276,160355,39976,165192,138510,50166,123089,5179,146738,112689,99175,71945,33530,35549,125516,1799,173042,197569,18444,180369,140208,74637,16208,157574,71440,134056,3703,135429,115246,29637,108712,171643,138320,39361,45809,96616,14532,131739,127486,167592,167187,2817,93119,106716,90631,19612,20038,131019,88916,128328,125876,21688,32308,53359,46631,33405,17394,151515,32612,65826,126300,164357,15706,133935,178162,145267,59509,13633,43374,133752,164499,32720,143157,192171,96908,152955,132194,42226,187401,181414,181574,169049,138969,2398,5146,163631,103406,64557,89966,44845,189763,158552,14651,88805,75675,73828,77975,136748,161028,176435,156382,90624,162023,97563,96383,62375,187428,37132,115066,33023,137589,22051,181662,66420,29011,108519,46237,116965,74998,12781,4325,118583,27254,104758,12071,38323,182231,137775,132140,63349,116217,183112,12202,160491,187866,138075,106337,93632,185936,42779,115430,158087,93848,121544,44450,140627,151425,175364,169917,80590,172729,188550,60500,111132,159638,155456,4185,45648,92642,147153,48470,158336,16949,87095,17749,192971,159370,192051,162469,38616,43961,39673,15338,50238,9199,36053,169830,4916,159597,98174,192343,147878,4776,109422,73937,16607,112642,95160,72835,172726,66052,144880,188615,114551,44009,1028,189270,169165,163654,165005,191931,143191,280,132029,6926,124788,36102,151122,114789,185397,65483,137129,104370,133884,176301,65374,10816,138565,143977,3347,156036,114068,25514,199270,109118,19665,103830,64747,57334,136207,169338,79613,176466,1255,56160,152200,67690,80715,125722,121240,1358,21261,196391,10073,163785,44271,72846,80637,113513,183213,59947,159414,152289,109063,13527,3310,98545,156056,90065,149701,164115,106491,71404,84719,10606,43755,118159,134985,141125,16851,87982,119854,113277,135724,120040,39844,16949,67182,173957,1752,11869,195450,132411,122,56124,149046,168587,117859,125034,28045,8042,149088,99711,96345,38477,174592,160089,134028,96314,150413,190199,102466,42605,45057,52703,95920,199629,40825,94727,139967,101812,195805,67364,5389,7,84584,197540,773,153231,171129,130685,44707,568,199293,11168,189609,9494,193,162080,76122,185345,191181,83756,141804,188531,16966,129995,126442,159023,186599,26205,51097,170045,76074,138554,77732,85984,75566,14354,52633,166304,128831,8581,87474,34138,159621,136208,72859,36641,58999,35510,78227,75099,78850,166382,52143,63329,163577,105095,172652,61239,86241,27699,171869,108649,105653,116212,195285,196378,151678,111711,199926,143233,134413,67100,80797,54558,183898,73465,65995,163817,175552,118478,19370,42572,196334,40848,179206,194316,198512,48002,60763,10655,30496,47271,59376,160666,50428,181268,80733,25812,104102,66822,106214,135127,175192,59288,175618,36346,131194,184440,96581,188090,102926,142600,81334,46504,110619,167738,78731,72712,17897,34174,144177,191759,198322,171794,157682,86855,164588,103201,75387,164895,83479,103855,94062,130325,51543,88040,106088,82972,136784,183808,103987,141821,50017,163367,126291,111223,8414,65368,65147,82128,57751,25098,15322,198651,144296,127970,186059,28696,28286,113019,129199,166884,172614,41279,156478,1413,119643,129284,28549,71733,10937,175999,59770,197578,96075,33497,105139,27704,76885,79450,3121,21981,68984,134623,26522,47929,109501,117271,91148,161391,195713,145134,81798,10448,184857,75455,143656,43524,143778,185103,345,40375,72082,189919,88284,8432,10529,142546,91652,95636,140184,941,101326,155939,84367,32392,150791,6048,174439,25435,183169,180416,75446,75922,119700,24285,174615,111692,48893,54624,131465,165930,110392,154470,113890,80101,149650,73780,159444,24947,138707,12842,146840,82076,121820,165198,66524,17810,16384,151920,149606,26890,114996,17959,111794,11563,115739,173628,165503,12129,16684,88827,72056,150891,22202,25718,78397,193300,73886,182936,135496,1704,131203,38227,19812,89883,26064,58912,162785,53955,189420,81233,90017,45887,85594,121847,23086,102931,131682,107898,151855,33507,177518,141305,146007,185386,128101,167787,142918,52870,86383,159714,20456,130574,155407,106627,152065,99114,85811,128161,49617,132778,129638,61460,181876,150624,65519,175516,3011,106397,110664,182926,21085,94211,156984,188999,75557,55125,196473,30734,53692,74565,102905,168342,66050,95175,127351,124764,85191,91916,123370,23277,174063,182900,10495,162449,22819,89417,120078,55459,81583,76047,174469,194017,117167,100931,29904,62556]
    indices = indices_train
    indices.extend(indices_val)
    indices.extend(indices_test)
    dataset = dataset[indices]
    print(len(dataset))
    mean = dataset.data.y.mean(dim=0, keepdim=True)
    std = dataset.data.y.std(dim=0, keepdim=True)
    dataset.data.y = (dataset.data.y - mean) / std
    device = torch.device(f"cuda:{cfg.general.device}")
    mean, std = mean.to(device), std.to(device)
    train_dataset = dataset[0:10000]
    val_dataset = dataset[10000:11000]
    test_dataset = dataset[11000:]
    dataloader = {
        "train": data.DataLoader(
            train_dataset,
            batch_size=cfg.data.bs,
            num_workers=cfg.data.num_workers,
            shuffle=True
        ),
        "val": data.DataLoader(
            val_dataset,
            batch_size=cfg.data.bs,
            num_workers=cfg.data.num_workers,
            shuffle=False
        ),
        "test": data.DataLoader(
            test_dataset,
            batch_size=cfg.data.bs,
            num_workers=cfg.data.num_workers,
            shuffle=False
        ),
    }
    num_elements_in_target = 12
    return dataloader, num_elements_in_target


def get_peptides_func_dataloader(cfg, transform_suffix, transforms):
    dataset = PeptidesFunctionalDataset(root = cfg.data.dir + "_using_pre_transform_" + transform_suffix, pre_transform=transforms)
    splits = dataset.get_idx_split()
    train_dataset = dataset[splits['train']]
    val_dataset = dataset[splits['val']]
    test_dataset = dataset[splits['test']]
    dataloader = {
        "train": data.DataLoader(train_dataset,
            batch_size=cfg.data.bs,
            num_workers=cfg.data.num_workers,
            shuffle=True),
        "val": data.DataLoader(val_dataset,
            batch_size=cfg.data.bs,
            num_workers=cfg.data.num_workers,
            shuffle=False),
        "test": data.DataLoader(test_dataset,
            batch_size=cfg.data.bs,
            num_workers=cfg.data.num_workers,
            shuffle=False),
    }
    num_elements_in_target = 10
    return dataloader, num_elements_in_target

def get_peptides_struct_dataloader(cfg, transform_suffix, transforms):
    dataset = PeptidesStructuralDataset(root = cfg.data.dir + "_using_pre_transform_" + transform_suffix, pre_transform=transforms)
    splits = dataset.get_idx_split()
    train_dataset = dataset[splits['train']]
    val_dataset = dataset[splits['val']]
    test_dataset = dataset[splits['test']]
    dataloader = {
        "train": data.DataLoader(train_dataset,
            batch_size=cfg.data.bs,
            num_workers=cfg.data.num_workers,
            shuffle=True),
        "val": data.DataLoader(val_dataset,
            batch_size=cfg.data.bs,
            num_workers=cfg.data.num_workers,
            shuffle=False),
        "test": data.DataLoader(test_dataset,
            batch_size=cfg.data.bs,
            num_workers=cfg.data.num_workers,
            shuffle=False),
    }
    num_elements_in_target = 11
    return dataloader, num_elements_in_target
        


# ----------------------------- datasets ----------------------------- #

class PeptidesFunctionalDataset(InMemoryDataset):
    def __init__(self, root='datasets', smiles2graph=smiles2graph,
                 transform=None, pre_transform=None):
        """
        PyG dataset of 15,535 peptides represented as their molecular graph
        (SMILES) with 10-way multi-task binary classification of their
        functional classes.

        The goal is use the molecular representation of peptides instead
        of amino acid sequence representation ('peptide_seq' field in the file,
        provided for possible baseline benchmarking but not used here) to test
        GNNs' representation capability.

        The 10 classes represent the following functional classes (in order):
            ['antifungal', 'cell_cell_communication', 'anticancer',
            'drug_delivery_vehicle', 'antimicrobial', 'antiviral',
            'antihypertensive', 'antibacterial', 'antiparasitic', 'toxic']

        Args:
            root (string): Root directory where the dataset should be saved.
            smiles2graph (callable): A callable function that converts a SMILES
                string into a graph object. We use the OGB featurization.
                * The default smiles2graph requires rdkit to be installed *
        """

        self.original_root = root
        self.smiles2graph = smiles2graph
        self.folder = osp.join(root, 'peptides-functional')

        self.url = 'https://www.dropbox.com/s/ol2v01usvaxbsr8/peptide_multi_class_dataset.csv.gz?dl=1'
        self.version = '701eb743e899f4d793f0e13c8fa5a1b4'  # MD5 hash of the intended dataset file
        self.url_stratified_split = 'https://www.dropbox.com/s/j4zcnx2eipuo0xz/splits_random_stratified_peptide.pickle?dl=1'
        self.md5sum_stratified_split = '5a0114bdadc80b94fc7ae974f13ef061'

        # Check version and update if necessary.
        release_tag = osp.join(self.folder, self.version)
        if osp.isdir(self.folder) and (not osp.exists(release_tag)):
            print(f"{self.__class__.__name__} has been updated.")
            if input("Will you update the dataset now? (y/N)\n").lower() == 'y':
                shutil.rmtree(self.folder)

        super().__init__(self.folder, transform, pre_transform)
        self.data, self.slices = torch.load(self.processed_paths[0])

    @property
    def raw_file_names(self):
        return 'peptide_multi_class_dataset.csv.gz'

    @property
    def processed_file_names(self):
        return 'geometric_data_processed.pt'

    def _md5sum(self, path):
        hash_md5 = hashlib.md5()
        with open(path, 'rb') as f:
            buffer = f.read()
            hash_md5.update(buffer)
        return hash_md5.hexdigest()

    def download(self):
        if decide_download(self.url):
            path = download_url(self.url, self.raw_dir)
            # Save to disk the MD5 hash of the downloaded file.
            hash = self._md5sum(path)
            if hash != self.version:
                raise ValueError("Unexpected MD5 hash of the downloaded file")
            open(osp.join(self.root, hash), 'w').close()
            # Download train/val/test splits.
            path_split1 = download_url(self.url_stratified_split, self.root)
            assert self._md5sum(path_split1) == self.md5sum_stratified_split
        else:
            print('Stop download.')
            exit(-1)

    def process(self):
        data_df = pd.read_csv(osp.join(self.raw_dir,
                                       'peptide_multi_class_dataset.csv.gz'))
        smiles_list = data_df['smiles']

        print('Converting SMILES strings into graphs...')
        data_list = []
        for i in tqdm(range(len(smiles_list))):
            data = Data()

            smiles = smiles_list[i]
            graph = self.smiles2graph(smiles)

            assert (len(graph['edge_feat']) == graph['edge_index'].shape[1])
            assert (len(graph['node_feat']) == graph['num_nodes'])

            data.__num_nodes__ = int(graph['num_nodes'])
            data.edge_index = torch.from_numpy(graph['edge_index']).to(
                torch.int64)
            data.edge_attr = torch.from_numpy(graph['edge_feat']).to(
                torch.int64)
            data.x = torch.from_numpy(graph['node_feat']).to(torch.int64)
            data.y = torch.Tensor([eval(data_df['labels'].iloc[i])])

            data_list.append(data)

        if self.pre_transform is not None:
            data_list = [self.pre_transform(data) for data in tqdm(data_list)]
            # data_list = [self.pre_transform(data) for data in data_list]

        data, slices = self.collate(data_list)

        print('Saving...')
        torch.save((data, slices), self.processed_paths[0])

    def get_idx_split(self):
        """ Get dataset splits.

        Returns:
            Dict with 'train', 'val', 'test', splits indices.
        """
        split_file = osp.join(self.root,
                              "splits_random_stratified_peptide.pickle")
        with open(split_file, 'rb') as f:
            splits = pickle.load(f)
        split_dict = replace_numpy_with_torchtensor(splits)
        return split_dict

class PeptidesStructuralDataset(InMemoryDataset):
    # TODO: change this smiles2graph=smiles2graph
    def __init__(self, root='datasets', smiles2graph=smiles2graph,
                 transform=None, pre_transform=None):
        """
        PyG dataset of 15,535 small peptides represented as their molecular
        graph (SMILES) with 11 regression targets derived from the peptide's
        3D structure.

        The original amino acid sequence representation is provided in
        'peptide_seq' and the distance between atoms in 'self_dist_matrix' field
        of the dataset file, but not used here as any part of the input.

        The 11 regression targets were precomputed from molecule XYZ:
            Inertia_mass_[a-c]: The principal component of the inertia of the
                mass, with some normalizations. Sorted
            Inertia_valence_[a-c]: The principal component of the inertia of the
                Hydrogen atoms. This is basically a measure of the 3D
                distribution of hydrogens. Sorted
            length_[a-c]: The length around the 3 main geometric axis of
                the 3D objects (without considering atom types). Sorted
            Spherocity: SpherocityIndex descriptor computed by
                rdkit.Chem.rdMolDescriptors.CalcSpherocityIndex
            Plane_best_fit: Plane of best fit (PBF) descriptor computed by
                rdkit.Chem.rdMolDescriptors.CalcPBF
        Args:
            root (string): Root directory where the dataset should be saved.
            smiles2graph (callable): A callable function that converts a SMILES
                string into a graph object. We use the OGB featurization.
                * The default smiles2graph requires rdkit to be installed *
        """

        self.original_root = root
        self.smiles2graph = smiles2graph
        self.folder = osp.join(root, 'peptides-structural')

        ## Unnormalized targets.
        # self.url = 'https://www.dropbox.com/s/464u3303eu2u4zp/peptide_structure_dataset.csv.gz?dl=1'
        # self.version = '9786061a34298a0684150f2e4ff13f47'

        ## Standardized targets to zero mean and unit variance.
        self.url = 'https://www.dropbox.com/s/0d4aalmq4b4e2nh/peptide_structure_normalized_dataset.csv.gz?dl=1'
        self.version = 'c240c1c15466b5c907c63e180fa8aa89'  # MD5 hash of the intended dataset file

        self.url_stratified_split = 'https://www.dropbox.com/s/9dfifzft1hqgow6/splits_random_stratified_peptide_structure.pickle?dl=1'
        self.md5sum_stratified_split = '5a0114bdadc80b94fc7ae974f13ef061'

        # Check version and update if necessary.
        release_tag = osp.join(self.folder, self.version)
        if osp.isdir(self.folder) and (not osp.exists(release_tag)):
            print(f"{self.__class__.__name__} has been updated.")
            if input("Will you update the dataset now? (y/N)\n").lower() == 'y':
                shutil.rmtree(self.folder)

        super().__init__(self.folder, transform, pre_transform)
        self.data, self.slices = torch.load(self.processed_paths[0])

    @property
    def raw_file_names(self):
        return 'peptide_structure_normalized_dataset.csv.gz'

    @property
    def processed_file_names(self):
        return 'geometric_data_processed.pt'

    def _md5sum(self, path):
        hash_md5 = hashlib.md5()
        with open(path, 'rb') as f:
            buffer = f.read()
            hash_md5.update(buffer)
        return hash_md5.hexdigest()

    def download(self):
        if decide_download(self.url):
            path = download_url(self.url, self.raw_dir)
            # Save to disk the MD5 hash of the downloaded file.
            hash = self._md5sum(path)
            if hash != self.version:
                raise ValueError("Unexpected MD5 hash of the downloaded file")
            open(osp.join(self.root, hash), 'w').close()
            # Download train/val/test splits.
            path_split1 = download_url(self.url_stratified_split, self.root)
            assert self._md5sum(path_split1) == self.md5sum_stratified_split
        else:
            print('Stop download.')
            exit(-1)

    def process(self):
        data_df = pd.read_csv(osp.join(self.raw_dir,
                                       'peptide_structure_normalized_dataset.csv.gz'))
        smiles_list = data_df['smiles']
        target_names = ['Inertia_mass_a', 'Inertia_mass_b', 'Inertia_mass_c',
                        'Inertia_valence_a', 'Inertia_valence_b',
                        'Inertia_valence_c', 'length_a', 'length_b', 'length_c',
                        'Spherocity', 'Plane_best_fit']
        # Assert zero mean and unit standard deviation.
        assert all(abs(data_df.loc[:, target_names].mean(axis=0)) < 1e-10)
        assert all(abs(data_df.loc[:, target_names].std(axis=0) - 1.) < 1e-10)

        print('Converting SMILES strings into graphs...')
        data_list = []
        for i in tqdm(range(len(smiles_list))):
            data = Data()

            smiles = smiles_list[i]
            y = data_df.iloc[i][target_names]
            graph = self.smiles2graph(smiles)

            assert (len(graph['edge_feat']) == graph['edge_index'].shape[1])
            assert (len(graph['node_feat']) == graph['num_nodes'])

            data.__num_nodes__ = int(graph['num_nodes'])
            data.edge_index = torch.from_numpy(graph['edge_index']).to(
                torch.int64)
            data.edge_attr = torch.from_numpy(graph['edge_feat']).to(
                torch.int64)
            data.x = torch.from_numpy(graph['node_feat']).to(torch.int64)
            data.y = torch.Tensor([y])

            data_list.append(data)

        if self.pre_transform is not None:
            data_list = [self.pre_transform(data) for data in tqdm(data_list)]
            # data_list = [self.pre_transform(data) for data in data_list]

        data, slices = self.collate(data_list)

        print('Saving...')
        torch.save((data, slices), self.processed_paths[0])

    def get_idx_split(self):
        """ Get dataset splits.

        Returns:
            Dict with 'train', 'val', 'test', splits indices.
        """
        split_file = osp.join(self.root,
                              "splits_random_stratified_peptide_structure.pickle")
        with open(split_file, 'rb') as f:
            splits = pickle.load(f)
        split_dict = replace_numpy_with_torchtensor(splits)
        return split_dict

