#!/usr/bin/env python3
import os
import subprocess
import argparse
import pathlib
import sys
import time
from datetime import datetime

def main():
    parser = argparse.ArgumentParser(description="Batch process multiple wolf folders with Qwen2.5 Vision Language Model")
    parser.add_argument("--audio_base_dir", default="../../../bourne_videos/", help="Base directory containing wolf folders")
    parser.add_argument("--output_base_dir", default="../../dataset/bourne_audios/", help="Base directory for output")
    parser.add_argument("-p", "--prompt-number", type=int, default=20, help="Prompt number to use")
    parser.add_argument("-b", "--batch-size", type=int, default=1, help="Batch size for video processing")
    parser.add_argument("--max-tokens", type=int, default=100, help="Maximum tokens to generate")
    parser.add_argument("--top-p", type=float, default=0.9, help="Top-p sampling for text generation")
    parser.add_argument("--top-k", type=int, default=2, help="Top-k sampling for text generation")
    parser.add_argument("--start", type=int, default=1, help="Starting folder number (e.g., 1 for wolf01)")
    parser.add_argument("--end", type=int, default=8, help="Ending folder number (e.g., 8 for wolf08)")
    parser.add_argument("--script", default="qwen2_vl_audio.py", help="Script to run (default: qwen2_vl_audio.py)")
    parser.add_argument("--log-dir", default="logs", help="Directory to store log files")
    args = parser.parse_args()

    # Create logs directory if it doesn't exist
    log_dir = pathlib.Path(args.log_dir)
    log_dir.mkdir(parents=True, exist_ok=True)

    # Generate the list of folders to process based on the range
    bourne_folders = []
    for i in range(args.start, args.end + 1):
        # Format with two digits (wolf01, wolf02, etc.)
        folder_name = f"bourne{i:02d}"
        bourne_folders.append(folder_name)
    
    print(f"Will process bourne folders: {bourne_folders}")
    
    # Process each folder
    for folder in bourne_folders:
        audio_path = os.path.join(args.audio_base_dir, folder)
        output_path = os.path.join(args.output_base_dir, folder)
        
        # Check if the folder exists
        if not os.path.exists(audio_path):
            print(f"Warning: Folder {audio_path} does not exist. Skipping.")
            continue
        
        # Create log file for this run
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        log_file = log_dir / f"{folder}_{timestamp}.log"
        
        print(f"\n{'='*50}")
        print(f"Processing folder: {folder}")
        print(f"Audio path: {audio_path}")
        print(f"Output path: {output_path}")
        print(f"Log file: {log_file}")
        print(f"{'='*50}\n")
        
        # Prepare the command with all the arguments from the script's interface
        cmd = [
            "python", args.script, 
            "-v", audio_path,
            "-b", str(args.batch_size),
            "-d", output_path,
            "-p", str(args.prompt_number),
            "--max-tokens", str(args.max_tokens),
            "--top-p", str(args.top_p),
            "--top-k", str(args.top_k)
        ]
        
        # Set environment variable if requested
        env = os.environ.copy()
        
        # Start time for this folder
        start_time = time.time()
        
        # Run the command
        try:
            # Open log file
            with open(log_file, 'w') as log:
                # Write header to log
                log.write(f"Processing folder: {folder}\n")
                log.write(f"Command: {' '.join(cmd)}\n")
                log.write(f"Started at: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n\n")
                log.flush()
                
                # Run the command and redirect output to log file
                process = subprocess.Popen(
                    cmd, 
                    env=env,
                    stdout=subprocess.PIPE,
                    stderr=subprocess.STDOUT,
                    text=True,
                    bufsize=1
                )
                
                # Stream output to both console and log file
                for line in process.stdout:
                    sys.stdout.write(line)
                    log.write(line)
                    log.flush()
                
                # Wait for process to complete
                return_code = process.wait()
                
                # Check return code
                if return_code != 0:
                    log.write(f"\nProcess failed with return code: {return_code}\n")
                    print(f"Error: Process failed with return code: {return_code}")
                else:
                    elapsed_time = time.time() - start_time
                    log.write(f"\nSuccessfully completed in {elapsed_time:.2f} seconds\n")
                    print(f"Successfully processed {folder} in {elapsed_time:.2f} seconds")
                
        except Exception as e:
            elapsed_time = time.time() - start_time
            print(f"Error processing {folder} after {elapsed_time:.2f} seconds: {e}")
            
            # Write to log file if possible
            try:
                with open(log_file, 'a') as log:
                    log.write(f"\nError after {elapsed_time:.2f} seconds: {e}\n")
            except:
                pass
            
            # Continue with the next folder
            continue
    
    print("\nAll processing complete!")

if __name__ == "__main__":
    main()
