""" This class allows importing the current logger from anywhere in the codebase. """

from contextvars import ContextVar
from typing import Optional

# Lazily typed to avoid circular import at module import time
_current_logger: ContextVar[Optional[object]] = ContextVar("current_logger", default=None)


def set_logger(logger) -> None:
    """Register the logger for this process/task context."""
    _current_logger.set(logger)


def get_logger():
    """Return the current logger or a NullLogger if none was set."""
    lg = _current_logger.get()
    if lg is None:
        # Local import to avoid requiring wandb in non-logging paths
        from src.utils.logger import NullLogger
        return NullLogger()
    return lg
