import torch.nn as nn
import torch.nn.functional as F
from torch_geometric.nn import GraphConv as PY_GraphConv
import torch

class GRAPHCONV(nn.Module):
    def __init__(self, args, input_dim, output_dim, hid_dim):
        super(GRAPHCONV, self).__init__()
        self.num_layers = args.K
        self.dropout = args.dropout

        self.activation = args.activation_fn
        if self.activation == 'relu':
            self.activation_fn = F.relu
        elif self.activation == 'leaky_relu':
            self.activation_fn = F.leaky_relu
        elif self.activation == 'tanh':
            self.activation_fn = F.tanh
        elif self.activation == 'sigmoid':
            self.activation_fn = F.sigmoid
        else:
            raise ValueError(f"Unsupported activation function: {self.activation}")




        self.layers = nn.ModuleList()
        if self.num_layers > 1:
            self.layers.append(PY_GraphConv(input_dim, hid_dim, aggr=args.aggregator))
            for _ in range(self.num_layers - 2):
                self.layers.append(PY_GraphConv(hid_dim, hid_dim, aggr=args.aggregator))
            self.layers.append(PY_GraphConv(hid_dim, hid_dim, aggr=args.aggregator))
        else:
            self.layers.append(PY_GraphConv(input_dim, hid_dim, aggr=args.aggregator))
        self.output = nn.Linear(hid_dim, output_dim)


        if args.rest_param:
            self.reset_parameter()


    def reset_parameter(self):
        nn.init.xavier_uniform_(self.output.weight.data)
        if self.output.bias is not None:
            self.output.bias.data.zero_()


    def forward(self, data):
        x, edge_index = data.x, data.edge_index

        if data.edge_weight is not None:
            edge_weight = data.edge_weight
        else:
            edge_weight = torch.ones(edge_index.size(1), device=edge_index.device)


        for i, layer in enumerate(self.layers):
            x = layer(x, edge_index, edge_weight)
            x = self.activation_fn(x)
            x = F.dropout(x, p=self.dropout, training=self.training)

        logits = self.output(x)

        return logits, x