import torch.nn as nn
import torch.nn.functional as F
from torch_geometric.nn import SGConv
import torch

class SGC(nn.Module):
    def __init__(self, args, input_dim, output_dim, hid_dim):
        super(SGC, self).__init__()
        self.num_layers = args.K
        self.dropout = args.dropout

        self.activation = args.activation_fn
        if self.activation == 'relu':
            self.activation_fn = F.relu
        elif self.activation == 'leaky_relu':
            self.activation_fn = F.leaky_relu
        elif self.activation == 'tanh':
            self.activation_fn = F.tanh
        elif self.activation == 'sigmoid':
            self.activation_fn = F.sigmoid
        else:
            raise ValueError(f"Unsupported activation function: {self.activation}")

        self.layers = SGConv(input_dim, hid_dim, K=self.num_layers)
        self.output = nn.Linear(hid_dim, output_dim)

        if args.rest_param:
            self.reset_parameter()

    def reset_parameter(self):
        # 初始化 SGConv 层的参数
        nn.init.xavier_uniform_(self.layers.lin.weight.data)
        if self.layers.lin.bias is not None:
            self.layers.lin.bias.data.zero_()

        # 初始化输出层的参数
        nn.init.xavier_uniform_(self.output.weight.data)
        if self.output.bias is not None:
            self.output.bias.data.zero_()


    def forward(self, data):
        x, edge_index = data.x, data.edge_index

        if data.edge_weight is not None:
            edge_weight = data.edge_weight
        else:
            edge_weight = torch.ones(edge_index.size(1), device=edge_index.device)


        x = self.layers(x, edge_index, edge_weight)
        x = self.activation_fn(x)
        x = F.dropout(x, p=self.dropout, training=self.training)

        logits = self.output(x)

        return logits, x