import torch.nn as nn
import torch.nn.functional as F
from torch_geometric.nn import SSGConv
import torch


class SSGC(nn.Module):
    def __init__(self, args, input_dim, output_dim, hid_dim):
        super(SSGC, self).__init__()
        self.num_layers = args.K
        self.dropout = args.dropout

        self.activation = args.activation_fn
        if self.activation == 'relu':
            self.activation_fn = F.relu
        elif self.activation == 'leaky_relu':
            self.activation_fn = F.leaky_relu
        elif self.activation == 'tanh':
            self.activation_fn = F.tanh
        elif self.activation == 'sigmoid':
            self.activation_fn = F.sigmoid
        else:
            raise ValueError(f"Unsupported activation function: {self.activation}")

        # SSGConv takes input_dim, output_dim, and K (number of propagation steps)
        self.conv = SSGConv(
            in_channels=input_dim,
            out_channels=hid_dim,
            K=self.num_layers,
            alpha=args.alpha if hasattr(args, 'alpha') else 0.1
        )

        self.output = nn.Linear(hid_dim, output_dim)

        if hasattr(args, 'reset_param') and args.reset_param:
            self.reset_parameter()
        elif hasattr(args, 'rest_param') and args.rest_param:
            self.reset_parameter()

    def reset_parameter(self):
        # Initialize SSGConv layer parameters
        nn.init.xavier_uniform_(self.conv.lin.weight.data)
        if self.conv.lin.bias is not None:
            self.conv.lin.bias.data.zero_()

        # Initialize output layer parameters
        nn.init.xavier_uniform_(self.output.weight.data)
        if self.output.bias is not None:
            self.output.bias.data.zero_()

    def forward(self, data):
        x, edge_index = data.x, data.edge_index

        if hasattr(data, 'edge_weight') and data.edge_weight is not None:
            edge_weight = data.edge_weight
        else:
            edge_weight = torch.ones(edge_index.size(1), device=edge_index.device)

        # Apply SSGConv
        x = self.conv(x, edge_index, edge_weight)
        x = self.activation_fn(x)
        x = F.dropout(x, p=self.dropout, training=self.training)

        # Apply output layer
        logits = self.output(x)

        return logits, x