import torch.nn as nn
import torch.nn.functional as F
from torch_geometric.nn import SAGEConv
import torch

class GRAPHSAGE(nn.Module):
    def __init__(self, args, input_dim, output_dim, hid_dim):
        super(GRAPHSAGE, self).__init__()
        self.num_layers = args.K
        self.args = args
        self.dropout = args.dropout
        self.threshold = args.threshold
        self.activation = args.activation_fn
        if self.activation == 'relu':
            self.activation_fn = F.relu
        elif self.activation == 'leaky_relu':
            self.activation_fn = F.leaky_relu
        elif self.activation == 'tanh':
            self.activation_fn = F.tanh
        elif self.activation == 'sigmoid':
            self.activation_fn = F.sigmoid
        else:
            raise ValueError(f"Unsupported activation function: {self.activation}")


        self.layers = nn.ModuleList()
        if self.num_layers > 1:
            self.layers.append(SAGEConv(input_dim, hid_dim, aggr="mean"))
            for _ in range(self.num_layers - 2):
                self.layers.append(SAGEConv(hid_dim, hid_dim, aggr="mean"))
            self.layers.append(SAGEConv(hid_dim, hid_dim, aggr="mean"))
        else:
            self.layers.append(SAGEConv(input_dim, hid_dim, aggr="mean"))
        self.output = nn.Linear(hid_dim, output_dim)


        if args.rest_param:
            self.reset_parameter()


    def reset_parameter(self):
        nn.init.xavier_uniform_(self.output.weight.data)
        if self.output.bias is not None:
            self.output.bias.data.zero_()


    def forward(self, data):
        if not self.args.original:
            data = self.filter_edges_by_threshold(data, self.threshold)
        x, edge_index = data.x, data.edge_index

        if data.edge_weight is not None:
            edge_weight = data.edge_weight
        else:
            edge_weight = torch.ones(edge_index.size(1), device=edge_index.device)


        for i, layer in enumerate(self.layers):
            x = layer(x, edge_index)
            x = self.activation_fn(x)
            x = F.dropout(x, p=self.dropout, training=self.training)

        logits = self.output(x)

        return logits, x
    def filter_edges_by_threshold(self,data, threshold):
        """
        Filter edges with weights below the given threshold.

        Args:
            data: The graph data object containing edge_index and edge_weight
            threshold: The weight threshold below which edges will be removed

        Returns:
            Updated data object with filtered edges
        """
        # Check if edge weights exist
        if not hasattr(data, 'edge_weight') or data.edge_weight is None:
            # If no edge weights, return the original data
            return data

        # Create mask for edges to keep (where weight >= threshold)
        mask = data.edge_weight >= threshold

        # Apply mask to filter both edge_index and edge_weight
        data.edge_index = data.edge_index[:, mask]
        data.edge_weight = data.edge_weight[mask]

        return data