#!python3

import argparse
import itertools
from pathlib import Path
from subprocess import Popen, PIPE

import jinja2
import yaml


# Iterator yielding all combinations for each option:
def all_combinations(d):
    k, v = zip(*d.items())
    k = list(k)
    v = list(v)

    for c in itertools.product(*v):
        yield dict(zip(k, c))


def main(args):
    with args.splitfile.open('r') as fp:
        splitfile = yaml.safe_load(fp)

    templateEnv = jinja2.Environment(loader=jinja2.FileSystemLoader(searchpath="./"))
    template = templateEnv.get_template(args.template)

    for cmb in all_combinations(splitfile):
        print(f"Submitting {cmb}", end="")
        rendered = template.render(cmb)
        if args.dry_run:
            print()
            print()
            print(rendered)
            print("#"*80)
            print()
            print()

        else:
            p = Popen(['sbatch'], stdout=PIPE, stdin=PIPE, stderr=PIPE)
            stdout_data = p.communicate(input=rendered.encode())[0]
            print(" ...Done!")


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Run multiple jobs using sbatch")
    parser.add_argument("--splitfile", type=Path, default="run_splits.yaml", help="Yaml file containing the relevant split parameters")
    parser.add_argument("--template", type=str, default="run_tpl.sh", help="Jinja2 template containing the template")
    parser.add_argument("--dry-run", action="store_true", help="Don't actually run anything")
    main(parser.parse_args())
