# 非均匀剪枝Mixtral模型加载指南

本目录包含用于加载非均匀剪枝Mixtral模型的工具和示例。所谓"非均匀剪枝"，指的是每一层保留不同数量的专家，而不是简单地对所有层使用相同的专家数量。

## 关键文件

- `modeling_pruned_mixtral.py`: 提供了对Transformers库中Mixtral模型类的修改，支持每层不同专家数量
- `load_model.py`: 基础模型加载示例

## 工作原理

非均匀剪枝模型加载的核心机制如下：

1. **模型类修改**: 通过`hack_pruned_mixtral()`函数修改Transformers库中的模型类
2. **配置文件更新**: 在模型的`config.json`中添加`layer_num_experts`字段，指定每层的专家数量
3. **标准加载接口**: 使用标准的`AutoModelForCausalLM.from_pretrained()`加载修改后的模型

## 使用方法

### 基本使用

```python
from transformers import AutoModelForCausalLM, AutoTokenizer 
from model import hack_pruned_mixtral_to_adapt
import torch
import json
import os.path as osp

# 步骤1: 调用hack_pruned_mixtral_to_adapt修改模型类
hack_pruned_mixtral_to_adapt()

# 步骤2: 准备专家数量配置
layer_experts = [4, 5, 6, 4, 5, 6, 4, 5, 6, 4, 5, 6, 4, 5, 6, 4, 5, 6, 4, 5, 6, 4, 5, 6, 4, 5, 6, 4, 5, 6, 4, 5]

# 步骤3: 更新模型配置
model_path = "/path/to/pruned/model"
config_path = osp.join(model_path, "config.json")
if osp.exists(config_path):
    with open(config_path, "r") as f:
        config = json.load(f)
    config["layer_num_experts"] = layer_experts
    with open(config_path, "w") as f:
        json.dump(config, f, indent=2)

# 步骤4: 加载模型
tokenizer = AutoTokenizer.from_pretrained(model_path)
model = AutoModelForCausalLM.from_pretrained(
    model_path,
    torch_dtype=torch.float16,
    device_map="auto",
)
```

### 从YAML配置加载

我们还提供了从YAML配置文件加载专家数量的功能，YAML文件格式如下：

```yaml
step1_expert:
  - 4  # 第0层
  - 5  # 第1层
  - 6  # 第2层
  # ...以此类推
```

使用YAML配置加载模型：

```python
import yaml
from transformers import AutoModelForCausalLM, AutoTokenizer
from model import hack_pruned_mixtral_to_adapt
import torch
import json
import os.path as osp

# 加载YAML配置
config_path = "/path/to/config.yaml"
with open(config_path, 'r') as f:
    config = yaml.safe_load(f)
    layer_experts = config["step1_expert"]

# 调用hack_pruned_mixtral
hack_pruned_mixtral_to_adapt()

# 更新模型配置
model_path = "/path/to/pruned/model"
config_json_path = osp.join(model_path, "config.json")
if osp.exists(config_json_path):
    with open(config_json_path, "r") as f:
        model_config = json.load(f)
    model_config["layer_num_experts"] = layer_experts
    with open(config_json_path, "w") as f:
        json.dump(model_config, f, indent=2)

# 加载模型
tokenizer = AutoTokenizer.from_pretrained(model_path)
model = AutoModelForCausalLM.from_pretrained(
    model_path,
    torch_dtype=torch.float16,
    device_map="auto",
)
```

## 模型兼容性

这种方法适用于使用标准剪枝方法处理过的Mixtral模型，包括以下剪枝策略:
- 逐层剪枝 (layerwise_pruning)
- 渐进式剪枝 (progressive_pruning)
- 专家重要性剪枝 (importance_pruning)

## 优势

与其他加载方法相比，使用`hack_pruned_mixtral`方式的优势:
1. **简单**: 使用标准HuggingFace接口加载模型
2. **灵活**: 支持每层不同数量的专家
3. **无侵入**: 不需要修改原始库代码
4. **高效**: 避免了额外的模型转换步骤 