import warnings
from typing import Optional, Tuple

import torch
import torch.nn as nn
import torch.nn.functional as F
from transformers.models.mixtral import modeling_mixtral
from transformers.models.mixtral.modeling_mixtral import (
    MixtralConfig,
    MixtralRMSNorm,
    ALL_ATTENTION_FUNCTIONS,
    MixtralDecoderLayer,
    MixtralSparseMoeBlock,
    MixtralBlockSparseTop2MLP,
)


class DynamicSkippingMixtralSparseMoeBlock(nn.Module):

    def __init__(self, config, beta):
        super().__init__()
        self.hidden_dim = config.hidden_size
        self.ffn_dim = config.intermediate_size
        self.num_experts = config.num_local_experts
        self.top_k = config.num_experts_per_tok

        
        self.gate = nn.Linear(self.hidden_dim, self.num_experts, bias=False)

        self.experts = nn.ModuleList([MixtralBlockSparseTop2MLP(config) for _ in range(self.num_experts)])
        
        self.beta = beta

    def forward(self, hidden_states: torch.Tensor) -> torch.Tensor:
        batch_size, sequence_length, hidden_dim = hidden_states.shape
        hidden_states = hidden_states.view(-1, hidden_dim)
        
        router_logits = self.gate(hidden_states)

        routing_weights = F.softmax(router_logits, dim=1, dtype=torch.float)
        routing_weights, selected_experts = torch.topk(routing_weights, self.top_k, dim=-1)
        
        onlytop1_mask = routing_weights[:, 1] < self.beta * routing_weights[:, 0]
        
        
        routing_weights[onlytop1_mask, 1] = 0
        routing_weights /= routing_weights.sum(dim=-1, keepdim=True)
        
        routing_weights = routing_weights.to(hidden_states.dtype)

        final_hidden_states = torch.zeros(
            (batch_size * sequence_length, hidden_dim), dtype=hidden_states.dtype, device=hidden_states.device
        )

        
        
        expert_mask = torch.nn.functional.one_hot(selected_experts, num_classes=self.num_experts)
        
        
        expert_mask[onlytop1_mask, 1, :] = 0
        expert_mask = expert_mask.permute(2, 1, 0)

        
        for expert_idx in range(self.num_experts):
            expert_layer = self.experts[expert_idx]
            idx, top_x = torch.where(expert_mask[expert_idx])

            if top_x.shape[0] == 0:
                continue

            
            top_x_list = top_x.tolist()
            idx_list = idx.tolist()

            
            
            
            current_state = hidden_states[None, top_x_list].reshape(-1, hidden_dim)
            current_hidden_states = expert_layer(current_state, routing_weights[top_x_list, idx_list, None])

            
            
            final_hidden_states.index_add_(0, top_x, current_hidden_states.to(hidden_states.dtype))
        final_hidden_states = final_hidden_states.reshape(batch_size, sequence_length, hidden_dim)
        return final_hidden_states, router_logits


class PrunedMixtralMoEBlock(MixtralSparseMoeBlock):
    def __init__(self, config, layer_idx):
        super().__init__(config)
        self.num_experts = config.layer_num_experts[layer_idx]
        self.gate = nn.Linear(self.hidden_dim, self.num_experts, bias=False)
        self.experts = nn.ModuleList([MixtralBlockSparseTop2MLP(config) for _ in range(self.num_experts)])

class PrunedMixtralDecoderLayer(MixtralDecoderLayer):
    def __init__(self, config, layer_idx):
        super().__init__(config, layer_idx)
        self.block_sparse_moe = PrunedMixtralMoEBlock(config, layer_idx)

def hack_pruned_mixtral_to_adapt():
    modeling_mixtral.MixtralPreTrainedModel._no_split_modules += ["PrunedMixtralDecoderLayer"]
    modeling_mixtral.MixtralDecoderLayer = PrunedMixtralDecoderLayer

def unhack_pruned_mixtral_to_adapt():
    modeling_mixtral.MixtralPreTrainedModel._no_split_modules.remove("PrunedMixtralDecoderLayer")
    modeling_mixtral.MixtralDecoderLayer = MixtralDecoderLayer
