"""
Clearance Optimization Configuration - High Cost Differential
Human experiments are 10x more expensive than rat
"""

module Config

# ============= DATASET CONFIGURATION =============
const DATASET_CONFIG = (
    data_file = joinpath(@__DIR__, "data.csv"),
    target_column = "CL_total_human_exp",  # Target to discover
    
    # Initial state: ONLY predictions
    feature_columns = ["CL_total_rat_pred", "CL_total_dog_pred", "CL_total_human_pred"],
    
    # Experimental columns are actions, NOT features
    id_column = "smiles",
    scenario_column = nothing
)

# ============= ROP CONFIGURATION =============
const ROP_CONFIG = (
    # NO progression criteria - pure POMDP
    progression_criteria = nothing,
    
    # Target: Human clearance > 1.0 (to be discovered)
    target_range = (
        min = 1.0,
        max = 10.0
    ),
    
    # Terminal condition
    is_terminal = (target_value) -> (target_value >= 1.0),
    
    # Optional: custom reward function
    reward_function = nothing
)

# ============= EXPERIMENTAL DESIGN =============
const EXPERIMENT_CONFIG = (
    assays = Dict(
        "CL_total_rat_exp" => [400.0, 1.0],     # Cheapest
        "CL_total_dog_exp" => [800.0, 2.0],     # 2x rat cost
        "CL_total_human_exp" => [4000.0, 3.0],  # 10x rat cost
    ),
    
    # Maximum parallel assays
    max_parallel_assays = 2,
    
    # Cost calculation method
    cost_calculation = "additive"
)

# ============= CEED ANALYSIS =============
const CEED_CONFIG = (
    # Belief thresholds (τ values)
    belief_thresholds = [0.6, 0.9],
    
    # Ensemble size for Monte Carlo analysis
    default_ensemble_size = 30,
    
    # Uncertainty configuration
    uncertainty_config = (
        distance_type = "quadratic",
        prior_type = "exponential",
        variance_type = "variance"
    ),
    
    # Solver configuration - Higher exploration to find MLASP
    solver_config = (
        max_iterations = 5000,
        exploration_constant = 5.0,
        depth = 10,
        tree_in_info = true,
        keep_tree = true,
        convergence_threshold = 1e-6,
        timeout_seconds = 300
    )
)

# ============= OUTPUT CONFIGURATION =============
const OUTPUT_CONFIG = (
    results_dir = "ADME_example/results",
    plot_config = (
        save_plots = true,
        plot_format = "png",
        dpi = 300,
        figure_size = (800, 600)
    ),
    report_config = (
        generate_summary = true,
        save_csv = true,
        save_json = false
    )
)

# ============= HELPER FUNCTIONS =============
function get_full_config()
    return (
        dataset = DATASET_CONFIG,
        rop = ROP_CONFIG,
        experiment = EXPERIMENT_CONFIG,
        ceed = CEED_CONFIG,
        scenarios = (scenarios = [], get_scenario = nothing, use_scenarios = false),
        output = OUTPUT_CONFIG
    )
end

function validate_config()
    println("✓ Configuration validated successfully")
    return true
end

end # module Config