#!/usr/bin/env julia

"""
CEEDesigns Analysis Runner
Flexible runner that accepts custom configuration files

Usage:
    julia run.jl [options]
    
Options:
    -c, --config <file>   Configuration file to use (default: config.jl)
    --data <file>         Data CSV file for automatic config generation (one-step mode)
    --name <name>         Experiment name (required with --data)
    -s, --sample <idx>    Run analysis for a specific sample index
    -e, --ensembles <n>   Number of ensemble runs (default: from config)
    -t, --tau <values>    Comma-separated tau values (default: from config)
    -o, --output <dir>    Output directory (overrides config)
    -h, --help            Show help message

Examples:
    julia run.jl                                  # Run with default config
    julia run.jl -c user_config.jl                # Use custom config
    julia run.jl -c materials_config.jl -e 50     # Custom config with 50 ensembles
    julia run.jl -s 1 -t 0.6,0.9                 # Run sample 1 with specific tau values
"""

# Setup Julia environment
using Pkg
Pkg.activate(@__DIR__)  # Use Template's own Project.toml

# Load required modules
include("Analysis.jl")
using .Analysis
using DataFrames
using Dates

# ============= ARGUMENT PARSING =============

function parse_arguments()
    # Check for one-step mode with --data
    data_file_arg = nothing
    experiment_name = nothing
    config_file = "ADME_example/config.jl"
    
    # First pass: check for --data (one-step mode)
    for i in 1:length(ARGS)-1
        if ARGS[i] == "--data" && i < length(ARGS)
            data_file_arg = ARGS[i+1]
        elseif ARGS[i] == "--name" && i < length(ARGS)
            experiment_name = ARGS[i+1]
        elseif ARGS[i] in ["-c", "--config"]
            config_file = ARGS[i+1]
        end
    end
    
    # If --data is provided, generate config automatically
    if !isnothing(data_file_arg)
        if isnothing(experiment_name)
            # Generate name from data file
            experiment_name = replace(basename(data_file_arg), ".csv" => "")
        end
        
        println("\n🚀 One-step mode: Auto-generating configuration...")
        
        # Generate config using domain_config_gen.jl
        include("user_domain/domain_config_gen.jl")
        
        # Run the automated config generation
        args = Dict{String, Any}("name" => experiment_name)
        config_data = Base.invokelatest(automated_config_generation, data_file_arg, args)
        example_dir, config_path = Base.invokelatest(create_example_folder, config_data)
        
        println("✅ Configuration generated: $config_path")
        config_file = config_path
    end
    
    # Load configuration
    if !isfile(config_file)
        error("Configuration file not found: $config_file")
    end
    
    # Load the config module
    config_module = Module()
    Base.include(config_module, abspath(config_file))
    
    # Get config values (use invokelatest to handle world age issues)
    cfg = Base.invokelatest(config_module.Config.get_full_config)
    
    # Initialize with config defaults
    sample_idx = nothing
    ensemble_size = cfg.ceed.default_ensemble_size
    tau_values = Float64[]
    data_file = nothing
    output_dir = nothing
    
    # Parse remaining arguments
    i = 1
    while i <= length(ARGS)
        arg = ARGS[i]
        
        if arg in ["-s", "--sample"]
            if i < length(ARGS)
                sample_idx = parse(Int, ARGS[i+1])
                i += 2
            else
                error("Missing value for $arg")
            end
            
        elseif arg in ["-e", "--ensembles"]
            if i < length(ARGS)
                ensemble_size = parse(Int, ARGS[i+1])
                i += 2
            else
                error("Missing value for $arg")
            end
            
        elseif arg in ["-t", "--tau"]
            if i < length(ARGS)
                tau_str = ARGS[i+1]
                try
                    tau_values = [parse(Float64, strip(t)) for t in split(tau_str, ",")]
                catch e
                    error("Failed to parse tau values: '$tau_str'. Expected format: '0.6' or '0.6,0.9'")
                end
                i += 2
            else
                error("Missing value for $arg")
            end
            
        elseif arg == "--data"
            # Already handled in first pass
            i += 2
            
        elseif arg == "--name"
            # Already handled in first pass
            i += 2
            
        elseif arg in ["-o", "--output"]
            if i < length(ARGS)
                output_dir = ARGS[i+1]
                i += 2
            else
                error("Missing value for $arg")
            end
            
        elseif arg in ["-c", "--config"]
            # Already handled
            i += 2
            
        elseif arg in ["-h", "--help"]
            println("""
            CEEDesigns Analysis Runner
            
            Usage:
                julia run.jl [options]
            
            Options:
                -c, --config <file>     Configuration file (default: config.jl)
                --data <file>           Data CSV for automatic config generation
                --name <name>           Experiment name (with --data)
                -s, --sample <idx>      Run single sample by index
                -e, --ensembles <n>     Number of ensemble runs (default: from config)
                -t, --tau <values>      Comma-separated tau values (default: from config)
                -o, --output <dir>      Output directory (overrides config)
                -h, --help              Show this help message
            
            Examples:
                julia run.jl                          # Run all samples with defaults
                julia run.jl -c user_config.jl        # Use custom configuration
                julia run.jl -s 1 -e 50              # Run sample 1 with 50 ensembles
                julia run.jl -t 0.6,0.9              # Run with tau values 0.6 and 0.9
            """)
            exit(0)
            
        else
            error("Unknown argument: $arg. Use --help for usage information.")
        end
    end
    
    # Use config defaults if not specified
    if isempty(tau_values)
        tau_values = cfg.ceed.belief_thresholds
    end
    
    return config_file, sample_idx, ensemble_size, tau_values, data_file, output_dir
end

# ============= MAIN FUNCTION =============

function main(config_file; sample_idx=nothing, ensemble_size=10, tau_values=[0.9], 
              data_file=nothing, output_dir=nothing)
    
    # Load configuration module
    if !isfile(config_file)
        error("Configuration file not found: $config_file")
    end
    
    println("\n🧬 CEEDesigns Analysis")
    println("=" ^ 80)
    println("📁 Configuration: $config_file")

    # Load the config module properly
    include(abspath(config_file))
    
    # Get configuration
    config = Base.invokelatest(Main.Config.get_full_config)
    
    # Use provided data file or default from config
    data_path = isnothing(data_file) ? config.dataset.data_file : data_file
    
    # Check if data file exists
    if !isfile(data_path)
        error("Data file not found: $data_path")
    end
    
    # Setup analysis (will use Main.Config internally)
    setup = setup_analysis(data_path)
    
    # Determine output directory
    if !isnothing(output_dir)
        # User specified output directory
        final_output_dir = output_dir
    elseif occursin("examples", config_file)
        # Config is in examples folder, save results there
        config_dir = dirname(abspath(config_file))
        final_output_dir = joinpath(config_dir, "results")
        println("📂 Auto-detected example folder, results will be saved to: $final_output_dir")
    else
        # Use default from config
        final_output_dir = setup.results_base
    end
    
    # Create new setup with determined output directory
    if final_output_dir != setup.results_base
        setup = AnalysisSetup(
            setup.data,
            setup.data_for_sampler,
            setup.sampler,
            setup.uncertainty,
            setup.weights,
            setup.solver,
            setup.features,
            setup.target,
            setup.assays,
            setup.config,
            final_output_dir
        )
        mkpath(final_output_dir)
    end
    
    println("🎯 Using tau values: $tau_values")
    
    # Run analysis
    if isnothing(sample_idx)
        # Run all samples
        println("📌 Running all samples with $ensemble_size ensembles, tau = $tau_values")
        run_all_samples(
            setup;
            ensemble_size = ensemble_size,
            save_plots = true,
            tau_values = tau_values
        )
    else
        # Run specific sample
        println("📌 Running sample $sample_idx with $ensemble_size ensembles, tau = $tau_values")
        
        # Get the sample
        if sample_idx > nrow(setup.data)
            error("Sample index $sample_idx exceeds data size ($(nrow(setup.data)))")
        end
        
        sample = setup.data[sample_idx, :]
        sample_name = if hasproperty(sample, Symbol(setup.config.dataset.id_column))
            sample[Symbol(setup.config.dataset.id_column)]
        else
            "Sample_$sample_idx"
        end
        
        results = Base.invokelatest(
            Main.Analysis.run_single_sample,
            setup,
            sample,
            string(sample_name);
            ensemble_size = ensemble_size,
            save_plots = true,
            tau_values = tau_values
        )
    end
    
    println("\n✨ Analysis complete!")
    println("📁 Results saved in: $(setup.results_base)")
end

# ============= EXECUTION =============

if abspath(PROGRAM_FILE) == @__FILE__
    config_file, sample_idx, ensemble_size, tau_values, data_file, output_dir = parse_arguments()
    
    main(config_file; 
         sample_idx=sample_idx, 
         ensemble_size=ensemble_size, 
         tau_values=tau_values,
         data_file=data_file, 
         output_dir=output_dir)
end