#!/usr/bin/env julia

"""
IBMDP algorithm

Usage:
    julia run.jl                        # Run all molecules with 10 ensembles, tau=0.9
    julia run.jl --molecule 2           # Run single molecule 2 with 10 ensembles
    julia run.jl --molecule 3 -e 20     # Run molecule 3 with 20 ensembles
    julia run.jl --ensembles 50         # Run all molecules with 50 ensembles
    julia run.jl --tau 0.6,0.9          # Run with tau values 0.6 and 0.9
    julia run.jl -m 1 -e 50 -t 0.6,0.9  # Run molecule 1, 50 ensembles, tau 0.6 and 0.9
"""

using Pkg
Pkg.activate(@__DIR__)
Pkg.instantiate()

include("config.jl")
using .CNS

# ============= MAIN EXECUTION =============
function main(; molecule::Union{Int,Nothing} = nothing, ensemble_size::Int = 10, tau_values::Vector{Float64} = [0.9])
    println("🧬 CNS Conditional CEED Analysis")
    println("=" ^ 80)
    
    # Setup analysis
    data_path = joinpath(@__DIR__, "data", "multitier.csv")
    setup = CNS.setup_cns_analysis(data_path)
    
    println("🎯 Using tau values: $(tau_values)")
    
    # Run analysis
    if !isnothing(molecule)
        # Single molecule
        println("📌 Running molecule $molecule with $ensemble_size ensembles, tau = $(tau_values)")
        
        # Validate molecule index
        if molecule < 1 || molecule > length(CNS.CNS_CONFIG.scenario_names)
            error("Invalid molecule index. Must be between 1 and $(length(CNS.CNS_CONFIG.scenario_names))")
        end
        
        scenario_name = CNS.CNS_CONFIG.scenario_names[molecule]
        all_results = Dict()
        scenario_results = CNS.run_single_molecule(setup, molecule, scenario_name; 
                                                     ensemble_size = ensemble_size, 
                                                     save_plots = true,
                                                     tau_values = tau_values)
        if !isempty(scenario_results)
            all_results[scenario_name] = scenario_results
        end
    else
        # All molecules
        println("📌 Running all molecules with $ensemble_size ensembles, tau = $(tau_values)")
        all_results = CNS.run_all_molecules(setup; ensemble_size = ensemble_size, save_plots = true, tau_values = tau_values)
    end
    
    # Create comparison plots if multiple scenarios
    if length(all_results) > 1
        CNS.create_comparison_plots(all_results, setup)
    end
    
    # Generate final report
    CNS.generate_final_report(all_results, setup; ensemble_size = ensemble_size)
    
    return all_results, setup
end

# Parse arguments and run
let
    molecule = nothing
    ensemble_size = 10
    tau_values = Float64[0.9]  # Default tau value
    
    i = 1
    while i <= length(ARGS)
        if ARGS[i] in ["--molecule", "-m"]
            molecule = parse(Int, ARGS[i + 1])
            i += 2
        elseif ARGS[i] in ["--ensembles", "-e"]
            ensemble_size = parse(Int, ARGS[i + 1])
            i += 2
        elseif ARGS[i] in ["--tau", "-t"]
            # Parse comma-separated tau values
            tau_str = ARGS[i + 1]
            try
                if contains(tau_str, ",")
                    # Split by comma, strip whitespace, and filter out empty strings
                    tau_parts = [strip(s) for s in split(tau_str, ",")]
                    tau_parts = filter(!isempty, tau_parts)  # Remove empty strings
                    
                    if isempty(tau_parts)
                        error("No valid tau values found in: '$tau_str'")
                    end
                    
                    tau_values = [parse(Float64, s) for s in tau_parts]
                else
                    stripped = strip(tau_str)
                    if isempty(stripped)
                        error("Empty tau value provided")
                    end
                    tau_values = [parse(Float64, stripped)]
                end
                
                # Validate tau values are in reasonable range
                for tau in tau_values
                    if tau < 0 || tau > 1
                        error("Tau values must be between 0 and 1, got: $tau")
                    end
                end
                
            catch e
                if isa(e, ArgumentError)
                    error("Invalid tau value format: '$tau_str'. Use numbers between 0-1, e.g., '0.6' or '0.6,0.9'")
                else
                    rethrow(e)
                end
            end
            i += 2
        elseif ARGS[i] in ["--help", "-h"]
            println("Usage: julia run.jl [OPTIONS]")
            println("  --molecule, -m <N>       Run only molecule N (1-4)")
            println("  --ensembles, -e <N>      Number of ensemble runs (default: 10)")
            println("  --tau, -t <values>       Comma-separated tau values (default: 0.9)")
            println("                           Examples: --tau 0.9 or --tau 0.6,0.9")
            exit(0)
        else
            error("Unknown option: $(ARGS[i]). Use --help for usage.")
        end
    end
    
    # Warning for large runs
    if isnothing(molecule) && ensemble_size >= 50
        println("⚠️  Running all molecules with $ensemble_size ensembles (may take hours)")
        sleep(3)
    end
    
    # Run analysis
    results, setup = main(molecule = molecule, ensemble_size = ensemble_size, tau_values = tau_values)
    println("\n✨ Complete! Results in: $(setup.results_base)")
end