from operator import matmul
import numpy as np
import math
from scipy.optimize import fsolve
import numpy.matlib
import scipy.stats
import scipy.linalg as scilin
from scipy.stats import multivariate_normal
from scipy.special import logsumexp
import matplotlib.pyplot as plt
import plotly.graph_objs as go
import time
import random
import util
from sklearn import datasets
from matplotlib.patches import Ellipse
from sklearn import mixture
import matplotlib.transforms as transforms
from functools import partial
import tensorflow as tf
from tensorflow.python.ops.numpy_ops import np_config
from scipy import optimize
np_config.enable_numpy_behavior()
# import os
# os.environ['TF_CPP_MIN_LOG_LEVEL'] = '2'
###################################################################################
###########################Outline#################################################
# 1) Generate a Gaussian Mixture Model Parameters
# 2) Numerically Integrate MI
#       a) Break into seperate Entropy terms
# 3) Create General Variational approx. computations
#       a) Marginal and Conditional
# 4) Compute Moment Matching
# 5) Compute Gradient Ascent
# 6) Compute Barber and Agakov
# 8) Plot Samples and approximation
###################################################################################

def GaussianMixtureParams(M,Dx):
    ###############################################################################
    # Outline: Randomly Generates Parameters for GMM
    #
    # Inputs:
    #       M - Number of components
    #       Dx - Number of dimensions for Latent Variable, X
    #       Dy - Number of dimensions for Observation Variable, Y
    #
    # Outputs:
    #       w - weights of components
    #       mu - means of components
    #       Sigma - Variance of components
    ###############################################################################
    D = Dx
    w = np.random.dirichlet(np.ones(M))
    mu = []
    sigma = []
    for d in range(M):
        mu.append(np.random.uniform(-5,5,(D,1)))
        A  = np.random.uniform(-2,2,(D, D))
        sigma.append(A.T@A/2+.01*np.eye(D))
        # sigma.append(datasets.make_spd_matrix(D))
    return w,mu,sigma

def SampleGMM(N,w,mu,sigma):
    ###############################################################################
    # Outline: Samples Points from a GMM
    #
    # Inputs:
    #       N - Number of points to sample
    #       w - weights of GMM components
    #       mu - means of GMM components
    #       Sigma - Variance of GMM components
    #
    # Outputs:
    #       samples - coordniates of sampled points
    ###############################################################################
    samples = np.zeros((N,len(mu[0])))
    for j in range(N):
        acc_pis = [np.sum(w[:i]) for i in range(1, len(w)+1)]
        r = np.random.uniform(0, 1)
        k = 0
        for i, threshold in enumerate(acc_pis):
            if r < threshold:
                k = i
                break
        x = np.random.multivariate_normal(mu[k].T.tolist()[0],sigma[k].tolist())
        samples[j,:] = x
    return samples

def MargEntGMM(N,L,Dx,w,mu,Sigma):
    ###############################################################################
    # Outline: Numerically Calculates Marginal Entropy
    #
    # Inputs:
    #       samples - List of full sample set
    #       Dx - Dimension of Latenat Variable, X
    #       w - weights of components
    #       mu - means of components
    #       Sigma - Variance of components
    #
    # Outputs:
    #       MargEnt - Marginal Entropy
    ###############################################################################
    M = len(w)
    x = np.linspace(-L,L,N)
    if Dx == 1:
        X=x 
    else:
        X1, X2 = np.meshgrid(x,x)
        X = np.vstack((X1.flatten(),X2.flatten()))

    MargEntPart = np.zeros((M,len(X.T)))
    for d in range(M):
        MargEntPart[d,:] = multivariate_normal.logpdf(X.T,mu[d][0:Dx].T.tolist()[0],Sigma[d][0:Dx,0:Dx])+np.log(w[d])
    if Dx == 1:
        MargEnt = -1*sum(np.sum(np.exp(MargEntPart),axis=0)*logsumexp(MargEntPart,axis=0))*2*L/N
    else:
        MargEnt = -1*sum(np.sum(np.exp(MargEntPart),axis=0)*logsumexp(MargEntPart,axis=0))*(2*L/N)**2
    return MargEnt

def MargEntGMMSample(N,Dx,w,mu,Sigma):
    ###############################################################################
    # Outline: Numerically Calculates Marginal Entropy
    #
    # Inputs:
    #       samples - List of full sample set
    #       Dx - Dimension of Latenat Variable, X
    #       w - weights of components
    #       mu - means of components
    #       Sigma - Variance of components
    #
    # Outputs:
    #       MargEnt - Marginal Entropy
    ###############################################################################
    M = len(w)
    sample = SampleGMM(N,w,mu,Sigma)
    MargEntPart = np.zeros((M,N))
    for d in range(M):
        MargEntPart[d,:] = multivariate_normal.logpdf(sample,mu[d][0:Dx].T.tolist()[0],Sigma[d][0:Dx,0:Dx])+np.log(w[d])
    MargEntPart = logsumexp(MargEntPart,axis=0)
    MargEnt = (-1/N)*np.sum(MargEntPart)
    return MargEnt

def MargEntGMMLimit(N,Dx,w,mu,Sigma):
    ###############################################################################
    # Outline: Numerically Calculates Marginal Entropy
    #
    # Inputs:
    #       samples - List of full sample set
    #       Dx - Dimension of Latenat Variable, X
    #       w - weights of components
    #       mu - means of components
    #       Sigma - Variance of components
    #
    # Outputs:
    #       MargEnt - Marginal Entropy
    ###############################################################################
    M = len(w)
    TaylorEnt = np.zeros((2,N+1))
    TaylorLimit = np.zeros((2,1))
    Scale = np.zeros((M,1))
    Sigma_inv = []
    
    t0 = time.time()
    for i in range(M):
        Scale[i] = w[i]*np.linalg.det(2*np.pi*Sigma[i])**(-1/2)
        Sigma_inv.append(np.linalg.inv(Sigma[i]))
    MaxConst = np.sum(Scale)
    TaylorEnt[0,0] += -w[i]*np.log(MaxConst)
    t1 = time.time()
    TaylorEnt[1,0] = t1-t0+.0005
    for i in range(M):
        outer = 0
        for n in range(1,N+1):
            s0 = time.time()
            middle = 0
            for k in range(n+1):
                Nmatrix, NCoef = multinomial_expand(n-k,M)#
                # numtermscheck = math.comb(n-k+M-1,M-1)
                inner = 0
                for t in range(len(NCoef)):
                    SumSigmaInv = np.matmul(Sigma_inv[i],np.eye(len(Sigma_inv[i])))
                    SumMu = np.matmul(Sigma_inv[i],mu[i])
                    SumInd = np.matmul(mu[i].T,np.matmul(Sigma_inv[i],mu[i]))
                    for j in range(M):
                        SumSigmaInv += Nmatrix[t,j]*Sigma_inv[j]
                        SumMu += Nmatrix[t,j]*np.matmul(Sigma_inv[j],mu[j])
                        SumInd += Nmatrix[t,j]*np.matmul(mu[j].T,np.matmul(Sigma_inv[j],mu[j]))
                    SumSigma = np.linalg.inv(SumSigmaInv)
                    SumMu = np.matmul(SumSigma,SumMu)
                    expTerm = np.exp(-.5*(-1*np.matmul(SumMu.T,np.matmul(SumSigmaInv,SumMu))+SumInd))
                    inner += NCoef[t]*np.prod((Scale.T)**Nmatrix[t])*expTerm*(np.linalg.det(2*np.pi*SumSigma)**(1/2))*(Scale[i]/w[i])
                combcoef = math.comb(n,k)
                middle += combcoef*inner*(-MaxConst)**k
            outer += (-1)**(n-1)/(n*MaxConst**n)*middle
            TaylorEnt[0,n] += -w[i]*(np.log(MaxConst)+outer)
            s1 = time.time()
            TaylorEnt[1,n] += s1-s0
    TaylorEnt[1,:] = np.cumsum(TaylorEnt[1,:])
    r0 = time.time()        
    TaylorLimit[0,0] = TaylorEnt[0,-3]-(TaylorEnt[0,-2]-TaylorEnt[0,-3])**2/(TaylorEnt[0,-1]-2*TaylorEnt[0,-2]+TaylorEnt[0,-3])
    r1 = time.time()
    TaylorLimit[1,0] = r1-r0+TaylorEnt[1,-1]
    return TaylorEnt, TaylorLimit

def multinomial_expand(pow,dim):
    ############ https://www.mathworks.com/matlabcentral/fileexchange/48215-multinomial-expansion
    NMatrix = multinomial_powers_recursive(pow,dim)
    powvec = np.matlib.repmat(pow,np.shape(NMatrix)[0],1)
    NCoef = np.floor(np.exp(scipy.special.gammaln(powvec+1).flatten() - np.sum(scipy.special.gammaln(NMatrix+1),1))+0.5)
    return NMatrix, NCoef

def multinomial_powers_recursive(pow,dim):
    if dim == 1:
        Nmatrix = np.array([[pow]])
    else:
        Nmatrix = []
        for pow_on_x1 in range(pow+1):
            newsubterms = multinomial_powers_recursive(pow-pow_on_x1,dim-1)
            new = np.hstack((pow_on_x1*np.ones((np.shape(newsubterms)[0],1)),newsubterms))
            if len(Nmatrix)==0:#Nmatrix == []:
                Nmatrix = new
            else:
                Nmatrix =np.vstack((Nmatrix, new))
            # Nmatrix = [Nmatrix; [pow_on_x1*ones(np.shape(newsubterms,1),1) , newsubterms] ]
    return Nmatrix

def HuberTaylor0(w,mu,Sigma):
    ###############################################################################
    # Outline: Numerically Calculates Marginal Entropy
    #
    # Inputs:
    #       samples - List of full sample set
    #       Dx - Dimension of Latenat Variable, X
    #       w - weights of components
    #       mu - means of components
    #       Sigma - Variance of components
    #
    # Outputs:
    #       MargEnt - Marginal Entropy
    ###############################################################################
    M = len(w)
    TaylorEnt = 0
    logNorms = np.zeros((M,1))
    for i in range(M):
        logNorms = np.zeros((M,1))
        for j in range(M):
            logNorms[j] = multivariate_normal.logpdf(mu[i].flatten(),mu[j].flatten(),Sigma[j])+np.log(w[j])
        TaylorEnt += -1*w[i]*logsumexp(logNorms)
    return TaylorEnt

def HuberTaylor2(w,mu,Sigma):
    ###############################################################################
    # Outline: Numerically Calculates Marginal Entropy
    #
    # Inputs:
    #       samples - List of full sample set
    #       Dx - Dimension of Latenat Variable, X
    #       w - weights of components
    #       mu - means of components
    #       Sigma - Variance of components
    #
    # Outputs:
    #       MargEnt - Marginal Entropy
    ###############################################################################
    M = len(w)
    TaylorEnt = HuberTaylor0(w,mu,Sigma)
    logNorms = np.zeros((M,1))
    for i in range(M):
        logNorms = np.zeros((M,1))
        F=0
        # f = w[i]*multivariate_normal.pdf(mu[i].flatten(),mu[i].flatten(),Sigma[i])
        # df = np.matmul(np.linalg.inv(Sigma[i]),(mu[i]-mu[i]))*w[i]*multivariate_normal.pdf(mu[i].flatten(),mu[i].flatten(),Sigma[i])
        f=0
        df = 0
        for j in range(M):
            f += w[j]*multivariate_normal.pdf(mu[i].flatten(),mu[j].flatten(),Sigma[j])
            df += -np.matmul(np.linalg.inv(Sigma[j]),(mu[i]-mu[j]))*w[j]*multivariate_normal.pdf(mu[i].flatten(),mu[j].flatten(),Sigma[j])
        for k in range(M):
            F += w[k]*np.matmul(np.linalg.inv(Sigma[k]),((1/f)*np.matmul((mu[i]-mu[k]),df.T)+(mu[i]-mu[k])*np.matmul(np.linalg.inv(Sigma[k]),(mu[i]-mu[k])).T-np.eye(len(Sigma[i]))))*multivariate_normal.pdf(mu[i].flatten(),mu[k].flatten(),Sigma[k])
        F = (1/f)*F
        TaylorEnt += -1*(w[i]/2)*np.sum((F*Sigma[i]))
    return TaylorEnt

def HuberTaylorN(N,w,mu,Sigma):
    M = len(w)
    # HuberTaylorEnt = HuberTaylor1(w,mu,Sigma)
    # HuberTaylorEnt = 0
    HuberTaylorEnt = np.zeros((1,N))
    # derivatives = np.zeros((10,1))
    for i in range(M):
        derivatives = logGMMautoGrad10(mu[i],w,mu,Sigma)
        for n in range(N):
            if n==0:
                HuberTaylorEnt[0,n] += -w[i]*derivatives[n]
            if n%2==0 and not n==0:
                # HuberTaylorEnt += -w[i]*(doublefactorial(n-1)/np.math.factorial(n))*np.sum(np.matmul(derivatives[n].T,Sigma[i]**(n/2)))
                HuberTaylorEnt[0,n] += -w[i]*(doublefactorial(n-1)/np.math.factorial(n))*np.sum(derivatives[n]*Sigma[i]**(n/2))
    return HuberTaylorEnt

def doublefactorial(n):
     if n <= 0:
         return 1
     else:
         return n * doublefactorial(n-2)

def logGMMautoGrad10(center,w,mu,Sigma):
    M =len(w)
    x = tf.Variable(center, dtype='float32')
    w = tf.convert_to_tensor(w, dtype=tf.float32)
    mu = tf.convert_to_tensor(mu, dtype=tf.float32)
    Sigma = tf.convert_to_tensor(Sigma, dtype=tf.float32)
    pi = tf.constant(np.pi)
    # with tf.GradientTape(persistent=True) as t16:
    #     with tf.GradientTape(persistent=True) as t15:
    #         with tf.GradientTape(persistent=True) as t14:
    #             with tf.GradientTape(persistent=True) as t13:
    #                 with tf.GradientTape(persistent=True) as t12:
    #                     with tf.GradientTape(persistent=True) as t11:
    #                         with tf.GradientTape(persistent=True) as t10:
    #                             with tf.GradientTape(persistent=True) as t9:
    #                                 with tf.GradientTape(persistent=True) as t8:
    #                                     with tf.GradientTape(persistent=True) as t7:
    #                                         with tf.GradientTape(persistent=True) as t6:
    with tf.GradientTape(persistent=True) as t5:
        with tf.GradientTape(persistent=True) as t4:
            with tf.GradientTape(persistent=True) as t3:
                with tf.GradientTape(persistent=True) as t2:
                    with tf.GradientTape(persistent=True) as t1:
                        with tf.GradientTape(persistent=True) as t0:
                            y=0
                            for j in range(M):
                                # y += -.5*tf.math.log(tf.linalg.det(2*pi*Sigma[j]))-.5*tf.linalg.matmul((x-mu[j]),tf.linalg.matmul(Sigma[j],(x-mu[j])),transpose_a=True)+tf.math.log(w[j])
                                y += w[j]*tf.linalg.det(2*pi*Sigma[j])**(-.5)*tf.math.exp(-.5*tf.linalg.matmul((x-mu[j]),tf.linalg.matmul(tf.linalg.inv(Sigma[j]),(x-mu[j])),transpose_a=True))
                            y = tf.math.log(y)
                        dy = t0.gradient(y,x)
                    dy2 = t1.gradient(dy,x)
                dy3 = t2.gradient(dy2,x)
            dy4 = t3.gradient(dy3,x)
        dy5 = t4.gradient(dy4,x)
    dy6 = t5.gradient(dy5,x)
    #                                         dy7 = t6.gradient(dy6,x)
    #                                     dy8 = t7.gradient(dy7,x)
    #                                 dy9 = t8.gradient(dy8,x)
    #                             dy10 = t9.gradient(dy9,x)
    #                         dy11 = t10.gradient(dy9,x)
    #                     dy12 = t11.gradient(dy9,x)
    #                 dy13 = t12.gradient(dy9,x)
    #             dy14 = t13.gradient(dy9,x)
    #         dy15 = t14.gradient(dy9,x)
    #     dy16 = t15.gradient(dy9,x)
    # dy17 = t16.gradient(dy9,x)
    return y.numpy(), dy.numpy(), dy2.numpy(), dy3.numpy(), dy4.numpy(), dy5.numpy(), dy6.numpy()#, dy7.numpy(), dy8.numpy(), dy9.numpy()#, dy10.numpy(), dy11.numpy(), dy12.numpy(), dy13.numpy(), dy14.numpy(), dy15.numpy(), dy16.numpy(), dy17.numpy()

def HuberExample(K,Ns):
    ###############################################################################
    # Outline: Numerically Calculates Marginal Entropy
    #
    # Inputs:
    #       samples - List of full sample set
    #       Dx - Dimension of Latenat Variable, X
    #       Dy - Dimension of Obsevation Variable, Y
    #       w - weights of components
    #       mu - means of components
    #       Sigma - Variance of components
    #
    # Outputs:
    #       CondEnt - Conditional Entropy
    ###############################################################################
    Dx = 2
    M = 5
    TrueEnt = np.zeros((K,1))
    TaylorEnt = np.zeros((K,len(Ns)))
    TaylorLimit = np.zeros((K,1))
    UpperBound = np.zeros((K,1))
    HuberEnt = np.zeros((K,len(Ns)))
    i=0
    for c in np.linspace(-3,3,K):    
            
        #Huber Example
        ws = np.array([0.2, 0.2, 0.2, 0.2 , 0.2])
        mus = [np.array([[0],[0]]), np.array([[3],[2]]), np.array([[1],[-.5]]), np.array([[2.5],[1.5]]),np.array([[c],[c]])]
        sigmas = [np.diag((.16,1)),np.diag((1,.16)),np.diag((.5,.5)),np.diag((.5,.5)),np.diag((.5,.5))]
        
        # c1 = 1/(6*np.pi)
        # ws = np.array([0.05, 0.05, 0.05, 0.05 , 0.8])
        # mus = [np.array([[0],[0]]), np.array([[3],[2]]), n p.array([[1],[-.5]]), np.array([[2.5],[1.5]]),np.array([[c],[c]])]
        # sigmas = [np.diag((.16,1)),np.diag((1,.16)),np.diag((.5,.5)),np.diag((.5,.5)),np.diag((c1,c1))]
        
        HuberEnt[i,0] = HuberTaylor0(ws,mus,sigmas)
        HuberEnt[i,1] = HuberEnt[i,0]
        HuberEnt[i,2] = HuberTaylor2(ws,mus,sigmas)
        TaylorEnt[i,:], TaylorLimit[i], UpperBound[i] = MargEntGMMLimit(Ns[-1],Dx,ws,mus,sigmas)
        TrueEnt[i] = MargEntGMM(1000,50,Dx,ws,mus,sigmas)
        i+=1
    return TrueEnt, TaylorEnt, TaylorLimit, HuberEnt #UpperBound, EntTaylor

def LargeGMMExample(Dx,Ns,Ms,ws,mus,sigmas,K):
    ###############################################################################
    # Outline: Numerically Calculates Marginal Entropy
    #
    # Inputs:
    #       samples - List of full sample set
    #       Dx - Dimension of Latenat Variable, X
    #       Dy - Dimension of Obsevation Variable, Y
    #       w - weights of components
    #       mu - means of components
    #       Sigma - Variance of components
    #
    # Outputs:
    #       CondEnt - Conditional Entropy
    ###############################################################################
    TrueEnt = np.zeros((2,K,len(Ms)))
    TaylorEnt = np.zeros((2,len(Ns)))
    TaylorLimit = np.zeros((2,1))
    HuberEnt = np.zeros((2,len(Ns)))
           
    # HuberEnt[0,0] = HuberTaylor0(ws,mus,sigmas)
    # HuberEnt[0,1] = HuberEnt[0,0]
    # HuberEnt[0,2] = HuberTaylor2(ws,mus,sigmas)
    
    # TaylorEnt, TaylorLimit = MargEntGMMLimit(Ns[-1],Dx,ws,mus,sigmas)
    for i in range(len(Ms)):
        for k in range(K):     
            t0 =time.time()
            TrueEnt[0,k,i] = MargEntGMMSample(Ms[i],Dx,ws,mus,sigmas)
            t1 = time.time()
            TrueEnt[1,k,i] = t1-t0
    return TrueEnt, TaylorEnt, TaylorLimit, HuberEnt

def DimensionExample(Ds,K):
    ###############################################################################
    # Outline: Numerically Calculates Marginal Entropy
    #
    # Inputs:
    #       samples - List of full sample set
    #       Dx - Dimension of Latenat Variable, X
    #       Dy - Dimension of Obsevation Variable, Y
    #       w - weights of components
    #       mu - means of components
    #       Sigma - Variance of components
    #
    # Outputs:
    #       CondEnt - Conditional Entropy
    ###############################################################################
    N = 2#3
    TrueEnt = np.zeros((2,K,len(Ds)))
    TaylorEnt = np.zeros((2,K,len(Ds)))
    LegendreEnt = np.zeros((2,K,len(Ds)))
    TaylorLimit = np.zeros((2,K,len(Ds)))
    HuberEnt = np.zeros((2,K,len(Ds)))
    M = 2
    i=0
    for Dx in Ds:
        print(Dx)
        for k in range(K): 
            ws, mus, Sigmas = GaussianMixtureParams(M,Dx)
            
            t0 =time.time()
            HuberEnt[0,k,i] = HuberTaylor2(ws,mus,Sigmas)
            t1 =time.time()
            HuberEnt[1,k,i] = t1-t0+.0005
            
            # HuberEnt[:,3,i] = HuberEnt[:,2,i]
            
            t0 = time.time()
            M = len(ws)
            MaxBound = 0
            for m in range(M):
                MaxBound += ws[m]*np.linalg.det(2*np.pi*Sigmas[m])**(-1/2)
            Epp = util.gmm_power_expected_value_parallel(N, ws, mus, Sigmas, ws, mus, Sigmas)
            t1 = time.time()
            sampleTime = t1-t0+.0005
            
            t0 = time.time()
            TaylorEnt[0,k,i] = -1*util.log_Taylor_series(N, MaxBound, Epp)
            t1 = time.time()
            TaylorEnt[1,k,i] = t1-t0 + sampleTime
            
            t0 = time.time()
            TaylorLimit[0,k,i] = -1*util.log_Taylor_limit(N, MaxBound, Epp)
            t1 = time.time()
            TaylorLimit[1,k,i] = t1-t0 + sampleTime
            
            t0 = time.time()
            LegendreEnt[0,k,i] = -1*util.log_Legendre_series(N, MaxBound, Epp)
            t1 = time.time()
            LegendreEnt[1,k,i] = t1-t0 + sampleTime
            
            t0 =time.time()
            TrueEnt[0,k,i] = MargEntGMMSample(1000,Dx,ws,mus,Sigmas)#Dx*2
            t1 = time.time()
            TrueEnt[1,k,i] = t1-t0+.0005
        i+=1
    return TrueEnt, TaylorEnt, TaylorLimit, HuberEnt, LegendreEnt

def ComponentExample(Ms,K):
    ###############################################################################
    # Outline: Numerically Calculates Marginal Entropy
    #
    # Inputs:
    #       samples - List of full sample set
    #       Dx - Dimension of Latenat Variable, X
    #       Dy - Dimension of Obsevation Variable, Y
    #       w - weights of components
    #       mu - means of components
    #       Sigma - Variance of components
    #
    # Outputs:
    #       CondEnt - Conditional Entropy
    ###############################################################################
    N = 2
    TrueEnt = np.zeros((2,K,len(Ms)))
    TaylorEnt = np.zeros((2,K,len(Ms)))
    LegendreEnt = np.zeros((2,K,len(Ms)))
    TaylorLimit = np.zeros((2,K,len(Ms)))
    HuberEnt = np.zeros((2,K,len(Ms)))
    Dx = 2 
    i=0
    for M in Ms:
        print(M)
        for k in range(K): 
            ws, mus, Sigmas = GaussianMixtureParams(M,Dx)
            
            t0 =time.time()
            HuberEnt[0,k,i] = HuberTaylor2(ws,mus,Sigmas)
            t1 =time.time()
            HuberEnt[1,k,i] = t1-t0+.0005
            
            # HuberEnt[:,3,i] = HuberEnt[:,2,i]
            
            t0 = time.time()
            M = len(ws)
            MaxBound = 0
            for m in range(M):
                MaxBound += ws[m]*np.linalg.det(2*np.pi*Sigmas[m])**(-1/2)
            Epp = util.gmm_power_expected_value_parallel(N, ws, mus, Sigmas, ws, mus, Sigmas)
            t1 = time.time()
            sampleTime = t1-t0+.0005
            
            t0 = time.time()
            TaylorEnt[0,k,i] = -1*util.log_Taylor_series(N, MaxBound, Epp)
            t1 = time.time()
            TaylorEnt[1,k,i] = t1-t0 + sampleTime
            
            t0 = time.time()
            TaylorLimit[0,k,i] = -1*util.log_Taylor_limit(N, MaxBound, Epp)
            t1 = time.time()
            TaylorLimit[1,k,i] = t1-t0 + sampleTime
            
            t0 = time.time()
            LegendreEnt[0,k,i] = -1*util.log_Legendre_series(N, MaxBound, Epp)
            t1 = time.time()
            LegendreEnt[1,k,i] = t1-t0 + sampleTime
            
            t0 =time.time()
            TrueEnt[0,k,i] = MargEntGMMSample(1000,Dx,ws,mus,Sigmas)#Dx*2
            t1 = time.time()
            TrueEnt[1,k,i] = t1-t0+.0005
            # i+=1
        i+=1
    return TrueEnt, TaylorEnt, TaylorLimit, HuberEnt, LegendreEnt


def OrderExample(Ns,K):
    ###############################################################################
    # Outline: Numerically Calculates Marginal Entropy
    #
    # Inputs:
    #       samples - List of full sample set
    #       Dx - Dimension of Latenat Variable, X
    #       Dy - Dimension of Obsevation Variable, Y
    #       w - weights of components
    #       mu - means of components
    #       Sigma - Variance of components
    #
    # Outputs:
    #       CondEnt - Conditional Entropy
    ###############################################################################
    M=2
    TrueEnt = np.zeros((2,K,len(Ns)))
    TaylorEnt = np.zeros((2,K,len(Ns)))
    LegendreEnt = np.zeros((2,K,len(Ns)))
    TaylorLimit = np.zeros((2,K,len(Ns)))
    HuberEnt = np.zeros((2,K,len(Ns)))
    Dx = 3
    # i=0
    # for N in Ns:
    #     print(N)
    #     for k in range(K): 
    for k in range(K):
        print(k)
        ws, mus, Sigmas = GaussianMixtureParams(M,Dx)
        i=0
        for N in Ns:
            # print(N)
            if N == 0 or N==1:
                t0 =time.time()
                HuberEnt[0,k,i] = HuberTaylor0(ws,mus,Sigmas)
                t1 =time.time()
                HuberEnt[1,k,i] = t1-t0+.0005
            
            if N==2 or N==3:    
                t0 =time.time()
                HuberEnt[0,k,i] = HuberTaylor2(ws,mus,Sigmas)
                t1 =time.time()
                HuberEnt[1,k,i] = t1-t0+.0005
            
            # HuberEnt[:,3,i] = HuberEnt[:,2,i]
            
            t0 = time.time()
            M = len(ws)
            MaxBound = 0
            for m in range(M):
                MaxBound += ws[m]*np.linalg.det(2*np.pi*Sigmas[m])**(-1/2)
            Epp = util.gmm_power_expected_value_parallel(N, ws, mus, Sigmas, ws, mus, Sigmas)
            t1 = time.time()
            sampleTime = t1-t0+.0005
            
            t0 = time.time()
            TaylorEnt[0,k,i] = -1*util.log_Taylor_series(N, MaxBound, Epp)
            t1 = time.time()
            TaylorEnt[1,k,i] = t1-t0 + sampleTime
            
            if N>=2:
                t0 = time.time()
                TaylorLimit[0,k,i] = -1*util.log_Taylor_limit(N, MaxBound, Epp)
                t1 = time.time()
                TaylorLimit[1,k,i] = t1-t0 + sampleTime
            
            t0 = time.time()
            LegendreEnt[0,k,i] = -1*util.log_Legendre_series(N, MaxBound, Epp)
            t1 = time.time()
            LegendreEnt[1,k,i] = t1-t0 + sampleTime
            
            t0 =time.time()
            TrueEnt[0,k,i] = MargEntGMMSample(1000,Dx,ws,mus,Sigmas)#Dx*2
            t1 = time.time()
            TrueEnt[1,k,i] = t1-t0+.0005
            i+=1
        # i+=1
    return TrueEnt, TaylorEnt, TaylorLimit, HuberEnt, LegendreEnt



# def OrderExample(N,K):
#     ###############################################################################
#     # Outline: Numerically Calculates Marginal Entropy
#     #
#     # Inputs:
#     #       samples - List of full sample set
#     #       Dx - Dimension of Latenat Variable, X
#     #       Dy - Dimension of Obsevation Variable, Y
#     #       w - weights of components
#     #       mu - means of components
#     #       Sigma - Variance of components
#     #
#     # Outputs:
#     #       CondEnt - Conditional Entropy
#     ###############################################################################
#     M=5
#     TrueEnt = np.zeros((2,K))
#     TaylorEnt = np.zeros((2,N+1))
#     TaylorLimit = np.zeros((2,1))
#     HuberEnt = np.zeros((2,3))
#     Dx = 50
#     ws, mus, Sigmas = GaussianMixtureParams(M,Dx,0)
    
#     t0 =time.time()    
#     HuberEnt[0,0] = HuberTaylor0(ws,mus,Sigmas)
#     t1 =time.time()
#     HuberEnt[1,0] = t1-t0+.0005
    
#     HuberEnt[:,1] = HuberEnt[:,0]
    
#     t0 =time.time()
#     HuberEnt[0,2] = HuberTaylor2(ws,mus,Sigmas)
#     t1 =time.time()
#     HuberEnt[1,2] = t1-t0+.0005
    
#     # HuberEnt[:,3,i] = HuberEnt[:,2,i]
    
#     TaylorEnt, TaylorLimit = MargEntGMMLimit(N,Dx,ws,mus,Sigmas)
#     for k in range(K):     
#         t0 =time.time()
#         TrueEnt[0,k] = MargEntGMMSample(500,Dx,ws,mus,Sigmas)
#         t1 = time.time()
#         TrueEnt[1,k] = t1-t0+.0005
#     return TrueEnt, TaylorEnt, TaylorLimit, HuberEnt

def plot_entropies(SampleEnt, TaylorEnt, TaylorLimit, HuberEnt, LegendreEnt, Ds, xaxis_title):
    mean_list = [np.mean(SampleEnt, axis=1), np.mean(TaylorEnt, axis=1), np.mean(TaylorLimit, axis=1),
                 np.mean(HuberEnt, axis=1), np.mean(LegendreEnt, axis=1)]
    std_list = [np.std(SampleEnt, axis=1), np.std(TaylorEnt, axis=1), np.std(TaylorLimit, axis=1),
                np.std(HuberEnt, axis=1), np.std(LegendreEnt, axis=1)]
    names_list = ['SampleEnt', 'TaylorEnt', 'TaylorLimit', 'HuberEnt', 'LegendreEnt']
    colors = ['rgba(0, 0, 0, 1)', 'rgba(0,255,255, 1)', 'rgba(0,0,255, 1)', 'rgba(255,0,0, 1)', 'rgba(0,255,0, 1)']
    colorsfill = ['rgba(0, 0, 0, 0.2)', 'rgba(0,255,255, 0.2)', 'rgba(0,0,255, 0.2)', 'rgba(255,0,0, 0.2)', 'rgba(0,255,0, 0.2)']

    # Plot 1: Mean Values of Entropies
    fig1 = go.Figure()
    for i in range(len(mean_list)):
        upper_bound = mean_list[i][0] + std_list[i][0]
        lower_bound = mean_list[i][0] - std_list[i][0]
        fig1.add_trace(go.Scatter(x=Ds, y=mean_list[i][0], name=names_list[i], line=dict(color=colors[i]), mode='lines'))
        fig1.add_trace(go.Scatter(x=Ds + Ds[::-1], y=np.concatenate([upper_bound, lower_bound[::-1]]), fill='toself', fillcolor=colorsfill[i], line=dict(color='rgba(255,255,255,0)'), showlegend=False))
    fig1.update_xaxes(title_text=xaxis_title, type="log", tickvals=Ds)#, dtick = "D2"
    if xaxis_title == "Dimension":
        fig1.update_yaxes(title_text="Entropy", type="log", tickvals = [.1,1,10,100])#, dtick = 1
    else:
        fig1.update_yaxes(title_text="Entropy", type="log",tickvals = [1,2,5,10])
    fig1.update_layout(font=dict(size=25),showlegend=False)#,legend=dict(xanchor="left",x=0.05,yanchor="top", y=0.95),showlegend=False
    fig1.update_layout(plot_bgcolor='white')
    fig1.update_xaxes(
        range=[np.log10(min(Ds)), np.log10(max(Ds))],
        mirror=True,
        ticks='outside',
        showline=True,
        linecolor='black',
        gridcolor='lightgrey'
    )
    fig1.update_yaxes(
        # range = [1,3.5],
        mirror=True,
        ticks='outside',
        showline=True,
        linecolor='black',
        gridcolor='lightgrey'
    )
    fig1.show()
    time.sleep(2)
    if xaxis_title == "Dimension":
        fig1.write_image("KLSyntheticDimensionEdge.pdf")
    else:
        fig1.write_image("KLSyntheticComponentEdge.pdf")
    

    # Plot 2: Mean Time with Standard Deviations
    fig2 = go.Figure()
    for i in range(len(mean_list)):
        upper_bound = mean_list[i][1] + std_list[i][1]
        lower_bound = mean_list[i][1] - std_list[i][1]
        fig2.add_trace(go.Scatter(x=Ds, y=mean_list[i][1], name=names_list[i], line=dict(color=colors[i]), mode='lines'))
        fig2.add_trace(go.Scatter(x=Ds + Ds[::-1], y=np.concatenate([upper_bound, lower_bound[::-1]]), fill='toself', fillcolor=colorsfill[i], line=dict(color='rgba(255,255,255,0)'), showlegend=False))
    # fig2.update_layout(title=title_list[1], xaxis_title='Dimension (D)', yaxis_title='Time (Seconds)')
    # fig2.show()
    fig2.update_xaxes(title_text=xaxis_title, type="log", tickvals=Ds)#, dtick = "D2"
    fig2.update_yaxes(title_text="Time", type="log", dtick = 1)#, dtick = 1
    #fig1.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')
    fig2.update_layout(font=dict(size=25),showlegend=False)#,legend=dict(xanchor="right",x=0.999,yanchor="bottom", y=0.001),showlegend=False
    fig2.update_layout(plot_bgcolor='white')
    fig2.update_xaxes(
        range=[np.log10(min(Ds)), np.log10(max(Ds))],
        mirror=True,
        ticks='outside',
        showline=True,
        linecolor='black',
        gridcolor='lightgrey'
    )
    fig2.update_yaxes(
        # range = [1,3.5],
        mirror=True,
        ticks='outside',
        showline=True,
        linecolor='black',
        gridcolor='lightgrey'
    )
    fig2.show()
    time.sleep(2)
    if xaxis_title == "Dimension":
        fig2.write_image("TimeSyntheticDimensionEdge.pdf")
    else:
        fig2.write_image("TimeSyntheticComponentEdge.pdf")
    
def plot_entropies_order(SampleEnt, TaylorEnt, TaylorLimit, HuberEnt, LegendreEnt, Ds):
    mean_list = [np.mean(SampleEnt, axis=1), np.mean(TaylorEnt, axis=1), np.mean(TaylorLimit, axis=1),
                 np.mean(HuberEnt, axis=1), np.mean(LegendreEnt, axis=1)]
    std_list = [np.std(SampleEnt, axis=1), np.std(TaylorEnt, axis=1), np.std(TaylorLimit, axis=1),
                np.std(HuberEnt, axis=1), np.std(LegendreEnt, axis=1)]
    names_list = ['Monte Carlo', 'Our Taylor', 'Taylor Limit', 'Huber et al.', 'Our Legendre']
    colors = ['rgba(0, 0, 0, 1)', 'rgba(0,255,255, 1)', 'rgba(0,0,255, 1)', 'rgba(255,0,0, 1)', 'rgba(0,255,0, 1)']
    colorsfill = ['rgba(0, 0, 0, 0.2)', 'rgba(0,255,255, 0.2)', 'rgba(0,0,255, 0.2)', 'rgba(255,0,0, 0.2)', 'rgba(0,255,0, 0.2)']

    # Plot 1: Mean Values of Entropies
    fig1 = go.Figure()
    for i in [0,3,1,2,4]:#range(len(mean_list)): #
        upper_bound = mean_list[i][0] + std_list[i][0]
        lower_bound = mean_list[i][0] - std_list[i][0]
        if i == 2:
            fig1.add_trace(go.Scatter(x=Ds[2:], y=mean_list[i][0][2:], name=names_list[i], line=dict(color=colors[i]), mode='lines'))
            fig1.add_trace(go.Scatter(x=Ds[2:] + Ds[2:][::-1], y=np.concatenate([upper_bound[2:], lower_bound[2:][::-1]]), fill='toself', fillcolor=colorsfill[i], line=dict(color='rgba(255,255,255,0)'), showlegend=False))
        elif i == 3:
            fig1.add_trace(go.Scatter(x=Ds[:4], y=mean_list[i][0][:4], name=names_list[i], line=dict(color=colors[i]), mode='lines'))
            fig1.add_trace(go.Scatter(x=Ds[:4] + Ds[3::-1], y=np.concatenate([upper_bound[:4], lower_bound[3::-1]]), fill='toself', fillcolor=colorsfill[i], line=dict(color='rgba(255,255,255,0)'), showlegend=False))
        else:
            fig1.add_trace(go.Scatter(x=Ds, y=mean_list[i][0], name=names_list[i], line=dict(color=colors[i]), mode='lines'))
            fig1.add_trace(go.Scatter(x=Ds + Ds[::-1], y=np.concatenate([upper_bound, lower_bound[::-1]]), fill='toself', fillcolor=colorsfill[i], line=dict(color='rgba(255,255,255,0)'), showlegend=False))
            
    fig1.update_xaxes(title_text="Order", tickvals=Ds)#, dtick = "D2"
    fig1.update_yaxes(title_text="Entropy")#, type="log",tickvals = [1,2,5,10]
    fig1.update_layout(font=dict(size=25),showlegend=False)#,legend=dict(xanchor="left",x=0.05,yanchor="top", y=0.95),showlegend=False
    fig1.update_layout(plot_bgcolor='white')
    fig1.update_xaxes(
        range=[np.log10(min(Ds)), np.log10(max(Ds))],
        mirror=True,
        ticks='outside',
        showline=True,
        linecolor='black',
        gridcolor='lightgrey'
    )
    fig1.update_yaxes(
        # range = [1,3.5],
        mirror=True,
        ticks='outside',
        showline=True,
        linecolor='black',
        gridcolor='lightgrey'
    )
    fig1.show()
    time.sleep(2)
    fig1.write_image("KLSyntheticOrderEdge.pdf")
    

    # Plot 2: Mean Time with Standard Deviations
    fig2 = go.Figure()
    for i in range(len(mean_list)):
        upper_bound = mean_list[i][1] + std_list[i][1]
        lower_bound = mean_list[i][1] - std_list[i][1]
        if i == 2:
            fig2.add_trace(go.Scatter(x=Ds[2:], y=mean_list[i][1][2:], name=names_list[i], line=dict(color=colors[i]), mode='lines'))
            fig2.add_trace(go.Scatter(x=Ds[2:] + Ds[2:][::-1], y=np.concatenate([upper_bound[2:], lower_bound[2:][::-1]]), fill='toself', fillcolor=colorsfill[i], line=dict(color='rgba(255,255,255,0)'), showlegend=False))
        elif i == 3:
            fig2.add_trace(go.Scatter(x=Ds[:4], y=mean_list[i][1][:4], name=names_list[i], line=dict(color=colors[i]), mode='lines'))
            fig2.add_trace(go.Scatter(x=Ds[:4] + Ds[3::-1], y=np.concatenate([upper_bound[:4], lower_bound[3::-1]]), fill='toself', fillcolor=colorsfill[i], line=dict(color='rgba(255,255,255,0)'), showlegend=False))
        else:
            fig2.add_trace(go.Scatter(x=Ds, y=mean_list[i][1], name=names_list[i], line=dict(color=colors[i]), mode='lines'))
            fig2.add_trace(go.Scatter(x=Ds + Ds[::-1], y=np.concatenate([upper_bound, lower_bound[::-1]]), fill='toself', fillcolor=colorsfill[i], line=dict(color='rgba(255,255,255,0)'), showlegend=False))
    # fig2.update_layout(title=title_list[1], xaxis_title='Dimension (D)', yaxis_title='Time (Seconds)')
    # fig2.show()
    fig2.update_xaxes(title_text="Order",tickvals=Ds)#, dtick = "D2"
    fig2.update_yaxes(title_text="Time", type="log", dtick = 1)#, dtick = 1
    #fig1.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')
    fig2.update_layout(font=dict(size=25),legend=dict(xanchor="right",x=.999,yanchor="bottom", y=.001,orientation='h'))#,showlegend=False
    fig2.update_layout(plot_bgcolor='white')
    fig2.update_xaxes(
        range=[np.log10(min(Ds)), np.log10(max(Ds))],
        mirror=True,
        ticks='outside',
        showline=True,
        linecolor='black',
        gridcolor='lightgrey'
    )
    fig2.update_yaxes(
        range = [-4.5, -.5],
        mirror=True,
        ticks='outside',
        showline=True,
        linecolor='black',
        gridcolor='lightgrey'
    )
    fig2.show()
    time.sleep(2)
    fig2.write_image("TimeSyntheticOrderEdge.pdf")



# M = 2
# Dx = 1000
# Dy = 0
# Ns = [0,1,2,3]
# Ms = [1,10,20]#[200,500,1000,1500]
# K=2
# ws, mus, Sigmas = GaussianMixtureParams(M,Dx,Dy)
# SampleEnt, TaylorEnt, TaylorLimit, HuberEnt = LargeGMMExample(Dx,Ns,Ms,ws,mus,Sigmas,K)

# Ds=Ms
############################## Dimension Example ######################################
random.seed(10)
np.random.seed(10)

Ds = [1,2,5,10,20,50]#,100,200,500,1000
K=30

SampleEnt, TaylorEnt, TaylorLimit, HuberEnt, LegendreEnt = DimensionExample(Ds,K)
plot_entropies(SampleEnt, TaylorEnt, TaylorLimit, HuberEnt, LegendreEnt, Ds,"Dimension")

# ################################ COMPONENT EXAMPLE ###########################################
Ms = [1,2,5,10,20]#
K=30
SampleEnt, TaylorEnt, TaylorLimit, HuberEnt, LegendreEnt = ComponentExample(Ms,K)
plot_entropies(SampleEnt, TaylorEnt, TaylorLimit, HuberEnt, LegendreEnt, Ms, "Components")

# ################################ ORDER EXAMPLE ###########################################
Ns = [0,1,2,3,4,5,6,7,8,9,10]#
K=30
SampleEnt, TaylorEnt, TaylorLimit, HuberEnt, LegendreEnt = OrderExample(Ns,K)
plot_entropies_order(SampleEnt, TaylorEnt, TaylorLimit, HuberEnt, LegendreEnt, Ns)

# SampleEnt, TaylorEnt, TaylorLimit, HuberEnt = OrderExample(Ns,K)
# meanSample = np.mean(SampleEnt,axis=1)
# stdSample = np.std(SampleEnt,axis=1)

# fig = go.Figure([
#         go.Scatter(
#             x=Ds,
#             y=meanSample[0,:].flatten(),
#             line=dict(color='rgb(255,0,0)', width=3),
#             mode='lines',
#             name='Monte Carlo'
#         ),
#         go.Scatter(
#             x=Ds,
#             y=(TaylorLimit[0,:]).flatten(),
#             line=dict(color='rgb(255,0,255)', width=3),
#             mode='lines',
#             name='Ent Limit'
#         )])
# for i in range(4):
#     C = 'rgb(0,0,%d)'%(i*155/(4-1)+100)
#     D = 'rgb(0,%d,0)'%(i*155/(4-1)+100)
#     # if i == Ns[-1]-2:
#     fig.add_trace(
#             go.Scatter(
#                 x=Ds,
#                 y=(TaylorEnt[0,i]).flatten(),
#                 line=dict(color=D, width=3),
#                 mode='lines',
#                 name='Our Taylor'))
#     fig.add_trace(
#             go.Scatter(
#                 x=Ds,
#                 y=(HuberEnt[0,i]).flatten(),
#                 line=dict(color=C, width=3),
#                 mode='lines',
#                 name='Huber Taylor'))
#     # else:
#     #     fig.add_trace(
#     #             go.Scatter(
#     #                 x=Ms,
#     #                 y=(TaylorEnt[0,i]*np.ones((1,len(Ms)))).flatten(),
#     #                 line=dict(color=D, width=3),
#     #                 mode='lines',
#     #                 showlegend=False))
#     #     fig.add_trace(
#     #             go.Scatter(
#     #                 x=Ms,
#     #                 y=(HuberEnt[0,i]*np.ones((1,len(Ms)))).flatten(),
#     #                 line=dict(color=C, width=3),
#     #                 mode='lines',
#     #                 showlegend=False))
# fig.update_xaxes(title_text="5th Gaussian Component Mean")#, type="log", dtick = "D2"
# fig.update_yaxes(title_text="H(x)")#, type="log", dtick = 1
# #fig1.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')
# fig.update_layout(font=dict(size=25))#,legend=dict(orientation="h",xanchor="center",x=0.5,yanchor="bottom", y=0.02),showlegend=False
# fig.update_layout(plot_bgcolor='white')
# fig.update_xaxes(
#     mirror=True,
#     ticks='outside',
#     showline=True,
#     linecolor='black',
#     gridcolor='lightgrey'
# )
# fig.update_yaxes(
#     # range = [1,3.5],
#     mirror=True,
#     ticks='outside',
#     showline=True,
#     linecolor='black',
#     gridcolor='lightgrey'
# )
# # fig.write_image("HuberExample.pdf")
# fig.show()

# fig1 = go.Figure([
#         go.Scatter(
#             x=Ds,
#             y=meanSample[1,:].flatten(),
#             line=dict(color='rgb(255,0,0)', width=3),
#             mode='lines',
#             name='Monte Carlo' 
#         ),
#         go.Scatter(
#             x=Ds,
#             y=(HuberEnt[1,-1]).flatten(),
#             line=dict(color='rgb(0,0,255)', width=3),
#             mode='lines',
#             name='Huber et al.'
#         ),      
#         go.Scatter(
#             x=Ds,
#             y=(TaylorEnt[1,-1]).flatten(),
#             line=dict(color='rgb(0,255,0)', width=3),
#             mode='lines',
#             name='Our Method'
#         ),
#         go.Scatter(
#             x=Ds,
#             y=(TaylorLimit[1,:]).flatten(),
#             line=dict(color='rgb(255,0,255)', width=3, dash='dash'),
#             mode='lines',
#             name='Approx. Limit'

#         )])
# fig1.update_xaxes(title_text="GMM Dimension", type="log", dtick = 1)#, dtick = "D2"
# fig1.update_yaxes(title_text="Computation Time", type="log", dtick = 1)#, dtick = 1
# #fig1.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')
# fig1.update_layout(font=dict(size=25),legend=dict(xanchor="left",x=0.05,yanchor="top", y=0.95))#,showlegend=False
# fig1.update_layout(plot_bgcolor='white')
# fig1.update_xaxes(
#     mirror=True,
#     ticks='outside',
#     showline=True,
#     linecolor='black',
#     gridcolor='lightgrey'
# )
# fig1.update_yaxes(
#     # range = [1,3.5],
#     mirror=True,
#     ticks='outside',
#     showline=True,
#     linecolor='black',
#     gridcolor='lightgrey'
# )
# # fig1.write_image("DimensionExample.pdf")
# fig1.show()


# ################################ COMPONENT EXAMPLE ###########################################
# Ms = [1,2,5,10,20,50]#,100,200,500,1000
# K=2

# SampleEnt, TaylorEnt, TaylorLimit, HuberEnt = ComponentExample(Ms,K)
# meanSample = np.mean(SampleEnt,axis=1)
# stdSample = np.std(SampleEnt,axis=1)

# # fig2 = go.Figure([
# #         go.Scatter(
# #             x=Ms,
# #             y=meanSample[0,:].flatten(),
# #             line=dict(color='rgb(255,0,0)', width=3),
# #             mode='lines',
# #             name='True Ent'
# #         ),
# #         go.Scatter(
# #             x=Ms,
# #             y=(TaylorLimit[0,:]).flatten(),
# #             line=dict(color='rgb(255,0,255)', width=3),
# #             mode='lines',
# #             name='Ent Limit'
# #         )])
# # for i in range(4):
# #     C = 'rgb(0,0,%d)'%(i*155/(4-1)+100)
# #     D = 'rgb(0,%d,0)'%(i*155/(4-1)+100)
# #     # if i == Ns[-1]-2:
# #     fig2.add_trace(
# #             go.Scatter(
# #                 x=Ms,
# #                 y=(TaylorEnt[0,i]).flatten(),
# #                 line=dict(color=D, width=3),
# #                 mode='lines',
# #                 name='Our Taylor'))
# #     fig2.add_trace(
# #             go.Scatter(
# #                 x=Ms,
# #                 y=(HuberEnt[0,i]).flatten(),
# #                 line=dict(color=C, width=3),
# #                 mode='lines',
# #                 name='Huber Taylor'))
# #     # else:
# #     #     fig.add_trace(
# #     #             go.Scatter(
# #     #                 x=Ms,
# #     #                 y=(TaylorEnt[0,i]*np.ones((1,len(Ms)))).flatten(),
# #     #                 line=dict(color=D, width=3),
# #     #                 mode='lines',
# #     #                 showlegend=False))
# #     #     fig.add_trace(
# #     #             go.Scatter(
# #     #                 x=Ms,
# #     #                 y=(HuberEnt[0,i]*np.ones((1,len(Ms)))).flatten(),
# #     #                 line=dict(color=C, width=3),
# #     #                 mode='lines',
# #     #                 showlegend=False))
# # fig2.update_xaxes(title_text="5th Gaussian Component Mean")#, type="log", dtick = "D2"
# # fig2.update_yaxes(title_text="H(x)")#, type="log", dtick = 1
# # #fig1.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')
# # fig2.update_layout(font=dict(size=25))#,legend=dict(orientation="h",xanchor="center",x=0.5,yanchor="bottom", y=0.02),showlegend=False
# # fig2.update_layout(plot_bgcolor='white')
# # fig2.update_xaxes(
# #     mirror=True,
# #     ticks='outside',
# #     showline=True,
# #     linecolor='black',
# #     gridcolor='lightgrey'
# # )
# # fig2.update_yaxes(
# #     # range = [1,3.5],
# #     mirror=True,
# #     ticks='outside',
# #     showline=True,
# #     linecolor='black',
# #     gridcolor='lightgrey'
# # )
# # # fig.write_image("HuberExample.pdf")
# # fig2.show()

# fig3 = go.Figure([
#         go.Scatter(
#             x=Ms,
#             y=meanSample[1,:].flatten(),
#             line=dict(color='rgb(255,0,0)', width=3),
#             mode='lines',
#             name='Monte Carlo' 
#         ),
#         go.Scatter(
#             x=Ms,
#             y=(HuberEnt[1,-1]).flatten(),
#             line=dict(color='rgb(0,0,255)', width=3),
#             mode='lines',
#             name='Huber et al.'
#         ),      
#         go.Scatter(
#             x=Ms,
#             y=(TaylorEnt[1,-1]).flatten(),
#             line=dict(color='rgb(0,255,0)', width=3),
#             mode='lines',
#             name='Our Method'
#         ),
#         go.Scatter(
#             x=Ms,
#             y=(TaylorLimit[1,:]).flatten(),
#             line=dict(color='rgb(255,0,255)', width=3, dash='dash'),
#             mode='lines',
#             name='Approx. Limit'

#         )])
# fig3.update_xaxes(title_text="GMM Components", type="log", dtick = 1)#, dtick = "D2"
# fig3.update_yaxes(title_text="Computation Time", type="log", dtick = 1)#, dtick = 1
# #fig1.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')
# fig3.update_layout(font=dict(size=25),legend=dict(xanchor="left",x=0.05,yanchor="top", y=0.95),)#showlegend=False
# fig3.update_layout(plot_bgcolor='white')
# fig3.update_xaxes(
#     mirror=True,
#     ticks='outside',
#     showline=True,
#     linecolor='black',
#     gridcolor='lightgrey'
# )
# fig3.update_yaxes(
#     # range = [1,3.5],
#     mirror=True,
#     ticks='outside',
#     showline=True,
#     linecolor='black',
#     gridcolor='lightgrey'
# )
# fig3.write_image("ComponentExample.pdf")
# fig3.show()

# ################################ ORDER EXAMPLE ###########################################
# N = 6
# K=2

# SampleEnt, TaylorEnt, TaylorLimit, HuberEnt = OrderExample(N,K)
# meanSample = np.mean(SampleEnt,axis=1)
# stdSample = np.std(SampleEnt,axis=1)


# fig4 = go.Figure(data=[
#     go.Bar(x=['Monte Carlo'], y=[meanSample[1]],marker_color='rgb(255,0,0)',showlegend=False),
#     go.Bar(name='Huber et al.',x=['Zeroth','First','Second','Third'], y=[HuberEnt[1,0],0,HuberEnt[1,2],0],marker_color='rgb(0,0,255)'),
#     go.Bar(name='Our Method',x=['Zeroth','First','Second','Third','Fourth','Fifth'], y=[TaylorEnt[1,0],TaylorEnt[1,1],TaylorEnt[1,2],TaylorEnt[1,3],TaylorEnt[1,4],TaylorEnt[1,5]],marker_color='rgb(0,255,0)')#,'Sixth',TaylorEnt[1,6]
#     ])
# fig4.update_yaxes(title_text="Computation Time")#, type="log", dtick = 1, dtick = 1
# fig4.update_layout(font=dict(size=25),legend=dict(xanchor="left",x=0.05,yanchor="top", y=0.95),)#showlegend=False
# fig4.update_layout(plot_bgcolor='white')
# fig4.update_xaxes(
#     mirror=True,
#     ticks='outside',
#     showline=True,
#     linecolor='black',
#     gridcolor='lightgrey'
# )
# fig4.update_yaxes(
#     # range = [1,3.5],
#     mirror=True,
#     ticks='outside',
#     showline=True,
#     linecolor='black',
#     gridcolor='lightgrey'
# )
# fig4.write_image("OrderExampl.pdf")
# fig4.show()

# # fig2 = go.Figure([
# #         go.Scatter(
# #             x=Ms,
# #             y=meanSample[0,:].flatten(),
# #             line=dict(color='rgb(255,0,0)', width=3),
# #             mode='lines',
# #             name='True Ent'
# #         ),
# #         go.Scatter(
# #             x=Ms,
# #             y=(TaylorLimit[0,:]).flatten(),
# #             line=dict(color='rgb(255,0,255)', width=3),
# #             mode='lines',
# #             name='Ent Limit'
# #         )])
# # for i in range(4):
# #     C = 'rgb(0,0,%d)'%(i*155/(4-1)+100)
# #     D = 'rgb(0,%d,0)'%(i*155/(4-1)+100)
# #     # if i == Ns[-1]-2:
# #     fig2.add_trace(
# #             go.Scatter(
# #                 x=Ms,
# #                 y=(TaylorEnt[0,i]).flatten(),
# #                 line=dict(color=D, width=3),
# #                 mode='lines',
# #                 name='Our Taylor'))
# #     fig2.add_trace(
# #             go.Scatter(
# #                 x=Ms,
# #                 y=(HuberEnt[0,i]).flatten(),
# #                 line=dict(color=C, width=3),
# #                 mode='lines',
# #                 name='Huber Taylor'))
# #     # else:
# #     #     fig.add_trace(
# #     #             go.Scatter(
# #     #                 x=Ms,
# #     #                 y=(TaylorEnt[0,i]*np.ones((1,len(Ms)))).flatten(),
# #     #                 line=dict(color=D, width=3),
# #     #                 mode='lines',
# #     #                 showlegend=False))
# #     #     fig.add_trace(
# #     #             go.Scatter(
# #     #                 x=Ms,
# #     #                 y=(HuberEnt[0,i]*np.ones((1,len(Ms)))).flatten(),
# #     #                 line=dict(color=C, width=3),
# #     #                 mode='lines',
# #     #                 showlegend=False))
# # fig2.update_xaxes(title_text="5th Gaussian Component Mean")#, type="log", dtick = "D2"
# # fig2.update_yaxes(title_text="H(x)")#, type="log", dtick = 1
# # #fig1.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')
# # fig2.update_layout(font=dict(size=25))#,legend=dict(orientation="h",xanchor="center",x=0.5,yanchor="bottom", y=0.02),showlegend=False
# # fig2.update_layout(plot_bgcolor='white')
# # fig2.update_xaxes(
# #     mirror=True,
# #     ticks='outside',
# #     showline=True,
# #     linecolor='black',
# #     gridcolor='lightgrey'
# # )
# # fig2.update_yaxes(
# #     # range = [1,3.5],
# #     mirror=True,
# #     ticks='outside',
# #     showline=True,
# #     linecolor='black',
# #     gridcolor='lightgrey'
# # )
# # # fig.write_image("HuberExample.pdf")
# # fig2.show()

# fig3 = go.Figure([
#         go.Scatter(
#             x=Ms,
#             y=meanSample[1,:].flatten(),
#             line=dict(color='rgb(255,0,0)', width=3),
#             mode='lines',
#             name='Monte Carlo' 
#         ),
#         go.Scatter(
#             x=Ms,
#             y=(HuberEnt[1,-1]).flatten(),
#             line=dict(color='rgb(0,0,255)', width=3),
#             mode='lines',
#             name='Huber et al.'
#         ),      
#         go.Scatter(
#             x=Ms,
#             y=(TaylorEnt[1,-1]).flatten(),
#             line=dict(color='rgb(0,255,0)', width=3),
#             mode='lines',
#             name='Our Method'
#         ),
#         go.Scatter(
#             x=Ms,
#             y=(TaylorLimit[1,:]).flatten(),
#             line=dict(color='rgb(255,0,255)', width=3, dash='dash'),
#             mode='lines',
#             name='Approx. Limit'

#         )])
# fig3.update_xaxes(title_text="GMM Components", type="log", dtick = 1)#, dtick = "D2"
# fig3.update_yaxes(title_text="Computation Time", type="log", dtick = 1)#, dtick = 1
# #fig1.update_layout(paper_bgcolor='rgba(0,0,0,0)',plot_bgcolor='rgba(0,0,0,0)')
# fig3.update_layout(font=dict(size=25),legend=dict(xanchor="left",x=0.05,yanchor="top", y=0.95),)#showlegend=False
# fig3.update_layout(plot_bgcolor='white')
# fig3.update_xaxes(
#     mirror=True,
#     ticks='outside',
#     showline=True,
#     linecolor='black',
#     gridcolor='lightgrey'
# )
# fig3.update_yaxes(
#     # range = [1,3.5],
#     mirror=True,
#     ticks='outside',
#     showline=True,
#     linecolor='black',
#     gridcolor='lightgrey'
# )
# fig.write_image("HuberExample.pdf")
# fig3.show()