import cv2
import imageio
import numpy as np
import wandb
from PIL import Image, ImageDraw, ImageFont


class VideoRecorder:
    def __init__(self, root_dir, render_size=256, fps=20, camera_id=0, use_wandb=False):
        if root_dir is not None:
            self.save_dir = root_dir / "eval_video"
            self.save_dir.mkdir(exist_ok=True)
        else:
            self.save_dir = None

        self.render_size = render_size
        self.fps = fps
        self.frames = []
        self.camera_id = camera_id
        self.use_wandb = use_wandb

    def init(self, env, enabled=True):
        self.frames = []
        self.enabled = self.save_dir is not None and enabled
        self.record(env)

    def record(self, env):
        if self.enabled:
            if hasattr(env, "physics"):
                frame = env.physics.render(
                    height=self.render_size,
                    width=self.render_size,
                    camera_id=self.camera_id,
                )
            else:
                assert False
                frame = env.render()
            self.frames.append(frame)

    def save(self, file_name, masks=None):
        if self.enabled:
            path = self.save_dir / file_name
            if masks is not None:
                for i in masks:
                    if i >= len(self.frames):
                        continue

                    image = Image.fromarray(
                        np.zeros_like(self.frames[0], dtype=np.uint8)
                    )
                    id = ImageDraw.Draw(image)
                    font = ImageFont.truetype("arial.ttf", 48)
                    id.text((240, 300), "Masked", fill=(255, 255, 255), font=font)
                    self.frames[i] = np.asarray(image)
            imageio.mimsave(str(path), self.frames, fps=self.fps)
