import collections

from dm_control import mujoco
from dm_control.rl import control
from dm_control.suite import base
from dm_control.suite import common
from dm_control.suite.utils import randomizers
from dm_control.utils import containers
from dm_control.utils import rewards
from dm_control.utils import io as resources
from dm_env import specs
import numpy as np
import os

_DEFAULT_TIME_LIMIT = 38
SUITE = containers.TaggedTasks()


def make(task, task_kwargs=None, environment_kwargs=None, visualize_reward=False):
    task_kwargs = task_kwargs or {}
    if environment_kwargs is not None:
        task_kwargs = task_kwargs.copy()
        task_kwargs["environment_kwargs"] = environment_kwargs
    env = SUITE[task](**task_kwargs)
    env.task.visualize_reward = visualize_reward
    return env


@SUITE.add("benchmarking")
def dummy(time_limit=_DEFAULT_TIME_LIMIT, random=None, environment_kwargs=None):
    """Returns the Run task."""
    root_dir = os.path.dirname(os.path.dirname(__file__))
    xml = os.path.join(
        root_dir, "custom_dmc_tasks", "universal_robots_ur5e", "ur5e.xml"
    )
    physics = Physics.from_xml_path(xml)
    task = EmptyTask(random=random)
    environment_kwargs = environment_kwargs or {}
    return control.Environment(
        physics,
        task,
        time_limit=time_limit,
        control_timestep=0.05,
        **{k:v for k, v in environment_kwargs.items() if k != 'targets'},
    )


class Physics(mujoco.Physics):
    """physics for the point_mass domain."""


class EmptyTask(base.Task):
    """A point_mass `Task` to reach target with smooth reward."""

    def __init__(self, random=None):
        """Initialize an instance of `PointMassMaze`.

        Args:
          randomize_gains: A `bool`, whether to randomize the actuator gains.
          random: Optional, either a `numpy.random.RandomState` instance, an
            integer seed for creating a new `RandomState`, or None to select a seed
            automatically (default).
        """
        super().__init__(random=random)

    def initialize_episode(self, physics):
        """Sets the state of the environment at the start of each episode.

           If _randomize_gains is True, the relationship between the controls and
           the joints is randomized, so that each control actuates a random linear
           combination of joints.

        Args:
          physics: An instance of `mujoco.Physics`.
        """
        randomizers.randomize_limited_and_rotational_joints(physics, self.random)
        # physics.data.qpos[0] = 0.0
        # physics.data.qpos[1] = 0.0
        # physics.data.qpos = np.array([1.4, 1.6, 0.0, -1.7, 6.3, -5.9, 0.0])
        physics.data.qpos = 0.0
        physics.data.qvel = 0.0
        # physics.data.qvel = 0.0

        super().initialize_episode(physics)

    def get_observation(self, physics):
        """Returns an observation of the state."""
        obs = collections.OrderedDict()
        obs["position"] = physics.data.qpos
        obs["velocity"] = physics.data.qvel
        return obs

    def get_reward_spec(self):
        return specs.Array(shape=(1,), dtype=np.float32, name="reward")

    def get_reward(self, physics):
        """Returns a reward to the agent."""

        return 0.1
