# Copyright 2017 The dm_control Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or  implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ============================================================================
"""Point-mass domain."""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import collections

from dm_control import mujoco
from dm_control.rl import control
from dm_control.suite import base
from dm_control.suite import common
from dm_control.suite.utils import randomizers
from dm_control.utils import containers
from dm_control.utils import rewards
from dm_control.utils import io as resources
from dm_env import specs
import numpy as np
import os

_DEFAULT_TIME_LIMIT = 27
SUITE = containers.TaggedTasks()


TASKS = [
    ("reach_top_left", np.array([-0.15, 0.15, 0.01])),
    ("reach_top_right", np.array([0.15, 0.15, 0.01])),
    ("reach_bottom_left", np.array([-0.15, -0.15, 0.01])),
    ("reach_bottom_right", np.array([0.15, -0.15, 0.01])),
]


def get_xml(targets):
    return f"""<mujoco model="planar point mass">
  <include file="./common/skybox.xml"/>
  <include file="./common/visual.xml"/>
  <include file="./common/materials.xml"/>
  <visual>
    <global offheight="640"/>
  </visual>

  <option timestep="0.025">
  </option>
  <asset>  
      <material name="self2" rgba=".1 .2 .3 1"/>
      <material name="target_site" rgba=".8 .8 .8 .2" specular="0" shininess="0"/> 
    </asset>

  <default>
    <joint type="hinge" axis="0 0 1" limited="true" range="-100 100" damping="1"/>
    <motor gear=".1" ctrlrange="-1 1" ctrllimited="true"/>
    <default class="wall">
      <geom type="box" material="site"/>
    </default>
  </default>

  <worldbody>
    <light name="light" pos="0 0 1"/>
    <!-- <camera name="view_stay" mode="targetbody" pos="0 -0.2 0.25" quat="1 0 0 0" target="origin"/> -->
    {f'<camera name="view_stay" mode="targetbody" pos="{targets[0]} {targets[1]} {targets[2]}" quat="1 0 0 0" target="origin"/>'}
    <geom name="ground" type="plane" pos="0 0 0" size="30 30 .1" material="grid"/>

    <!--{f'<body name="meter" pos="{targets[5]} -0.025 0">'}
        <site name="meter_geom" pos="0 0 0" rgba="1 0 0 0.5" size="{f'{targets[5]}'} 0.005 0.001" type="box"/>
    </body>-->
    <body name="target" >
      {os.linesep.join([f'<site name="pointmass_{i}" type="sphere" size=".008" material="target_site" pos="{t}"/>' for i,t in enumerate(targets[6:])])}
    </body>
    {f'<body name="origin" pos="{targets[3]} {targets[4]} 0.01"/>'}
    <body name="pointmass" pos="0 0 .01">
      <camera name="cam0" pos="0 -0.3 0.3" xyaxes="1 0 0 0 0.7 0.7"/>
      <joint name="root_x" type="slide"  pos="0 0 0" axis="1 0 0" />
      <joint name="root_y" type="slide"  pos="0 0 0" axis="0 1 0" />
      <geom name="pointmass" type="sphere" size=".01" material="self" mass=".3"/>
    </body>

  </worldbody>

  <tendon>
    <fixed name="t1">
      <joint joint="root_x" coef="1"/>
      <joint joint="root_y" coef="0"/>
    </fixed>
    <fixed name="t2">
      <joint joint="root_x" coef="0"/>
      <joint joint="root_y" coef="1"/>
    </fixed>
  </tendon>

  <actuator>
    <motor name="t1" tendon="t1" gear="2" />
    <motor name="t2" tendon="t2" gear="2" />
  </actuator>
</mujoco>
"""


def make(task, task_kwargs=None, environment_kwargs=None, visualize_reward=False):
    task_kwargs = task_kwargs or {}
    if environment_kwargs is not None:
        task_kwargs = task_kwargs.copy()
        task_kwargs["environment_kwargs"] = environment_kwargs
    env = SUITE[task](**task_kwargs)
    env.task.visualize_reward = visualize_reward
    return env


def get_model_and_assets(targets):
    """Returns a tuple containing the model XML string and a dict of assets."""
    root_dir = os.path.dirname(os.path.dirname(__file__))
    # xml = resources.GetResource(
    # os.path.join(root_dir, 'custom_dmc_tasks', f'point_mass_maze_{task}.xml'))
    return get_xml(targets), common.ASSETS


@SUITE.add("benchmarking")
def reach_bottom_left(
    time_limit=_DEFAULT_TIME_LIMIT, random=None, environment_kwargs=None
):
    """Returns the Run task."""
    physics = Physics.from_xml_string(
        *get_model_and_assets(environment_kwargs and environment_kwargs["targets"])
    )
    task = MultiTaskPointMassMaze(target_id=2, random=random)
    environment_kwargs = environment_kwargs or {}
    return control.Environment(
        physics,
        task,
        time_limit=time_limit,
        control_timestep=0.05,
        **{k: v for k, v in environment_kwargs.items() if k != "targets"},
    )


class Physics(mujoco.Physics):
    """physics for the point_mass domain."""

    def mass_to_target_dist(self, target):
        """Returns the distance from mass to the target."""
        d = target - self.named.data.geom_xpos["pointmass"]
        return np.linalg.norm(d)


class MultiTaskPointMassMaze(base.Task):
    """A point_mass `Task` to reach target with smooth reward."""

    def __init__(self, target_id, random=None):
        """Initialize an instance of `PointMassMaze`.

        Args:
          randomize_gains: A `bool`, whether to randomize the actuator gains.
          random: Optional, either a `numpy.random.RandomState` instance, an
            integer seed for creating a new `RandomState`, or None to select a seed
            automatically (default).
        """
        self._target = TASKS[target_id][1]
        super().__init__(random=random)

    def initialize_episode(self, physics):
        """Sets the state of the environment at the start of each episode.

           If _randomize_gains is True, the relationship between the controls and
           the joints is randomized, so that each control actuates a random linear
           combination of joints.

        Args:
          physics: An instance of `mujoco.Physics`.
        """
        randomizers.randomize_limited_and_rotational_joints(physics, self.random)
        physics.data.qpos[0] = 0.0
        physics.data.qpos[1] = 0.0
        # import ipdb; ipdb.set_trace()
        # physics.named.data.geom_xpos['target'][:] = self._target

        super().initialize_episode(physics)

    def get_observation(self, physics):
        """Returns an observation of the state."""
        obs = collections.OrderedDict()
        obs["position"] = physics.position()
        obs["velocity"] = physics.velocity()
        return obs

    def get_reward_spec(self):
        return specs.Array(shape=(1,), dtype=np.float32, name="reward")

    def get_reward(self, physics):
        """Returns a reward to the agent."""
        target_size = 0.015
        control_reward = rewards.tolerance(
            physics.control(), margin=1, value_at_margin=0, sigmoid="quadratic"
        ).mean()
        small_control = (control_reward + 4) / 5
        near_target = rewards.tolerance(
            physics.mass_to_target_dist(self._target),
            bounds=(0, target_size),
            margin=target_size,
        )
        reward = near_target * small_control
        return reward
