import matplotlib.pyplot as plt
import numpy as np
from sklearn import metrics


class ROC:
    def __init__(self, y_prediction, y_true):
        self.y = np.array(y_prediction)
        self.n = len(self.y)
        self.y_t = np.array(y_true).astype(bool)
        self.n_plus = np.sum(self.y_t)
        self.n_minus = np.sum(~self.y_t)
        self.level_sets = -np.sort(-np.unique(self.y))
        self.c_alpha, self.c_beta, self.t = metrics.roc_curve(self.y_t, self.y)
        self.auc = metrics.roc_auc_score(self.y_t, self.y)

        range_ = np.linspace(0, 1, 500)
        self.c_beta_interp = np.interp(range_, self.c_alpha, self.c_beta)

    # PLOTS
    def plot_roc_curve(self, *args, **kwargs):
        plt.plot(self.c_alpha, self.c_beta, *args, **kwargs)

    @staticmethod
    def plot_back_ground():
        plt.plot([0, 1], [0, 1], color="k", ls='--', alpha=0.5,
                 label='Random estimator')
        plt.xlim((-0.1, 1.1))
        plt.ylim((-0.1, 1.1))
        plt.xlabel("False positive rate")
        plt.ylabel("True positive rate")


def infinity_norm(roc1: ROC, roc2: ROC):
    return max(np.abs(roc1.c_beta_interp - roc2.c_beta_interp))


def norm_one(roc1: ROC, roc2: ROC):
    d_alpha = 1 / len(roc1.c_beta_interp)
    return np.sum((np.abs(roc1.c_beta_interp - roc2.c_beta_interp)) * d_alpha)


def test():
    roc = ROC(
        [0, 0, 0.1, 0.3, 0.3, 0.33],
        [0, 0, 0, 0, 1, 1], )

    self = roc

    def ground_truth(x): return np.array(x * 10).astype(int) / 10

    X = np.random.uniform(size=1000000)
    true = ground_truth(X) > np.random.uniform(size=X.shape[0])
    prediction = ground_truth(X)

    roc_true = ROC(prediction, true)

    plt.figure(figsize=(6, 6))
    roc.plot_roc_curve(color='b', marker='o',
                       label=f"Empirical ROC curve (AUC = {np.round(roc.auc, 2)})")
    roc_true.plot_roc_curve(color='r', marker='o',
                            label=f"True ROC curve  (AUC = {np.round(roc_true.auc, 2)})")
    roc.plot_back_ground()
    plt.legend()

    infty_norm = infinity_norm(roc, roc_true)
    one_norm = norm_one(roc, roc_true)

    assert int(1000 * one_norm) == int(1000 * (roc.auc - roc_true.auc))


if __name__ == "__main__":
    test()
