#    Copyright 2023 Haotian Liu
#
#    Licensed under the Apache License, Version 2.0 (the "License");
#    you may not use this file except in compliance with the License.
#    You may obtain a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS,
#    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#    See the License for the specific language governing permissions and
#    limitations under the License.


from abc import ABC, abstractmethod

import torch
import torch.nn as nn

from .multimodal_encoder.builder import build_vision_tower, build_protein_tower
from .multimodal_projector.builder import build_vision_projector, build_protein_projector

from llava.constants import IGNORE_INDEX, PROTEIN_TOKEN_INDEX, DEFAULT_PROTEIN_PATCH_TOKEN, DEFAULT_PROT_START_TOKEN, DEFAULT_PROT_END_TOKEN


class LlavaMetaModel:

    def __init__(self, config, residual_dropout=0.2):
        super(LlavaMetaModel, self).__init__(config)

        if hasattr(config, "mm_protein_tower"):
            self.protein_tower = build_protein_tower(config, delay_load=True)
            self.mm_projector = build_protein_projector(config)
            
            self.smiles_projector = build_protein_projector(config, 768)

    def get_protein_tower(self):
        protein_tower = getattr(self, "protein_tower", None)
        if type(protein_tower) is list:
            protein_tower = protein_tower[0]
        return protein_tower

    # def get_vision_tower(self):
    #     vision_tower = getattr(self, 'vision_tower', None)
    #     if type(vision_tower) is list:
    #         vision_tower = vision_tower[0]
    #     return vision_tower

    def initialize_protein_modules(self, model_args, fsdp=None):
        protein_tower = model_args.protein_tower
        # mm_vision_select_layer = model_args.mm_vision_select_layer
        # mm_vision_select_feature = model_args.mm_vision_select_feature
        pretrain_mm_mlp_adapter = model_args.pretrain_mm_mlp_adapter

        self.config.mm_protein_tower = protein_tower

        if self.get_protein_tower() is None:
            protein_tower = build_protein_tower(model_args)

            if fsdp is not None and len(fsdp) > 0:
                self.protein_tower = [protein_tower]
            else:
                self.protein_tower = protein_tower
        else:
            if fsdp is not None and len(fsdp) > 0:
                protein_tower = self.protein_tower[0]
            else:
                protein_tower = self.protein_tower
            protein_tower.load_model()

        self.config.use_mm_proj = True
        self.config.mm_projector_type = getattr(model_args, 'mm_projector_type', 'linear')
        self.config.mm_hidden_size = protein_tower.hidden_size
        
        if getattr(self, 'mm_projector', None) is None:
            self.mm_projector = build_protein_projector(self.config)
            self.smiles_projector = build_protein_projector(self.config, 768)
        else:
            # In case it is frozen by LoRA
            for p in self.mm_projector.parameters():
                p.requires_grad = True
            for p in self.smiles_projector.parameters():
                p.requires_grad = True

        if pretrain_mm_mlp_adapter is not None:
            mm_projector_weights = torch.load(pretrain_mm_mlp_adapter, map_location='cpu')
            def get_w(weights, keyword):
                return {k.split(keyword + '.')[1]: v for k, v in weights.items() if keyword in k}

            self.mm_projector.load_state_dict(get_w(mm_projector_weights, 'mm_projector'))
            self.smiles_projector.load_state_dict(get_w(mm_projector_weights, 'smiles_projector'))

class LlavaMetaForCausalLM(ABC):

    @abstractmethod
    def get_model(self):
        pass

    def get_protein_tower(self):
        return self.get_model().get_protein_tower()

    def encode_protein(self, proteins, smiles, spc_inds):
        if type(proteins[0]) is list:
            new_proteins = []
            for pps in proteins:
                for pp in pps:
                    new_proteins.append(pp)
        else:
            new_proteins = proteins
        proteins_features = self.get_model().get_protein_tower()(new_proteins)
        
        proteins_features = self.get_model().mm_projector(proteins_features)
        # align smiles length
        if smiles is not None:
            smiles_feature = self.get_model().smiles_projector(torch.cat(smiles, dim=0))
            max_len = smiles_feature.shape[1]
            smiles_c = 0
        else:
            max_len = 0

        all_feauture = []
        protein_c = 0
        for sub in spc_inds:
            for ind in sub:
                if ind == 0:
                    if max_len-proteins_features[protein_c].shape[1] <= 0:
                        all_feauture.append(proteins_features[protein_c])
                    else:
                        all_feauture.append(torch.nn.functional.pad(proteins_features[protein_c], (0, 0, 0, max_len-proteins_features[protein_c].shape[1])))
                    protein_c += 1
                else:
                    all_feauture.append(smiles_feature[smiles_c])
                    smiles_c += 1
        all_feauture = torch.stack(all_feauture)
        
        return all_feauture

    def prepare_inputs_labels_for_multimodal(
        self, input_ids, position_ids, attention_mask, past_key_values, labels, proteins, smiles, spc_inds
    ):
        protein_tower = self.get_protein_tower()
        # print(proteins, protein_tower)
        if protein_tower is None or proteins is None or input_ids.shape[1] == 1:
            if past_key_values is not None and protein_tower is not None and proteins is not None and input_ids.shape[1] == 1:
                target_shape = past_key_values[-1][-1].shape[-2] + 1
                attention_mask = torch.cat((attention_mask, torch.ones(
                    (attention_mask.shape[0], target_shape - attention_mask.shape[1]),
                    dtype=attention_mask.dtype,
                    device=attention_mask.device
                )), dim=1)
                position_ids = torch.sum(attention_mask, dim=1).unsqueeze(-1) - 1
            return input_ids, position_ids, attention_mask, past_key_values, None, labels

        # if type(proteins) is list or proteins.ndim == 5:
        #     concat_images = torch.cat([image for image in images], dim=0)
        #     image_features = self.encode_images(concat_images)
        #     split_sizes = [image.shape[0] for image in images]
        #     image_features = torch.split(image_features, split_sizes, dim=0)
        #     image_features = [x.flatten(0, 1).to(self.device) for x in image_features]
        # else:
        # print(input_ids.shape)
        protein_features = self.encode_protein(proteins, smiles, spc_inds).to(self.device)
        
        # TODO: image start / end is not implemented here to support pretraining.
        if getattr(self.config, 'tune_mm_mlp_adapter', False) and getattr(self.config, 'mm_use_im_start_end', False):
            raise NotImplementedError

        # Let's just add dummy tensors if they do not exist,
        # it is a headache to deal with None all the time.
        # But it is not ideal, and if you have a better idea,
        # please open an issue / submit a PR, thanks.
        _labels = labels
        _position_ids = position_ids
        _attention_mask = attention_mask
        if attention_mask is None:
            attention_mask = torch.ones_like(input_ids, dtype=torch.bool)
        else:
            attention_mask = attention_mask.bool()
        if position_ids is None:
            position_ids = torch.arange(0, input_ids.shape[1], dtype=torch.long, device=input_ids.device)
        if labels is None:
            labels = torch.full_like(input_ids, IGNORE_INDEX)

        # remove the padding using attention_mask -- TODO: double check]
        # print(input_ids.shape)
        input_ids = [cur_input_ids[cur_attention_mask] for cur_input_ids, cur_attention_mask in zip(input_ids, attention_mask)]
        labels = [cur_labels[cur_attention_mask] for cur_labels, cur_attention_mask in zip(labels, attention_mask)]

        new_input_embeds = []
        new_labels = []
        cur_protein_idx = 0
        for batch_idx, cur_input_ids in enumerate(input_ids):
            num_proteins = (cur_input_ids == PROTEIN_TOKEN_INDEX).sum()
            # print(cur_input_ids, num_proteins)
            if num_proteins == 0:
                cur_protein_features = protein_features[cur_protein_idx]
                cur_input_embeds_1 = self.get_model().embed_tokens(cur_input_ids)
                cur_input_embeds = torch.cat([cur_input_embeds_1, cur_protein_features[0:0]], dim=0)
                new_input_embeds.append(cur_input_embeds)
                new_labels.append(labels[batch_idx])
                cur_protein_idx += 1
                continue

            protein_token_indices = [-1] + torch.where(cur_input_ids == PROTEIN_TOKEN_INDEX)[0].tolist() + [cur_input_ids.shape[0]]
            cur_input_ids_noim = []
            cur_labels = labels[batch_idx]
            cur_labels_noim = []
            for i in range(len(protein_token_indices) - 1):
                cur_input_ids_noim.append(cur_input_ids[protein_token_indices[i]+1:protein_token_indices[i+1]])
                cur_labels_noim.append(cur_labels[protein_token_indices[i]+1:protein_token_indices[i+1]])
            split_sizes = [x.shape[0] for x in cur_labels_noim]
            cur_input_embeds = self.get_model().embed_tokens(torch.cat(cur_input_ids_noim))
            cur_input_embeds_no_im = torch.split(cur_input_embeds, split_sizes, dim=0)
            cur_new_input_embeds = []
            cur_new_labels = []

            for i in range(num_proteins + 1):
                cur_new_input_embeds.append(cur_input_embeds_no_im[i])
                cur_new_labels.append(cur_labels_noim[i])
                if i < num_proteins:
                    cur_protein_features = protein_features[cur_protein_idx]
                    cur_protein_idx += 1
                    cur_new_input_embeds.append(cur_protein_features)
                    cur_new_labels.append(torch.full((cur_protein_features.shape[0],), IGNORE_INDEX, device=cur_labels.device, dtype=cur_labels.dtype))

            cur_new_input_embeds = torch.cat(cur_new_input_embeds)
            cur_new_labels = torch.cat(cur_new_labels)

            new_input_embeds.append(cur_new_input_embeds)
            new_labels.append(cur_new_labels)

        # Truncate sequences to max length as image embeddings can make the sequence longer
        tokenizer_model_max_length = getattr(self.config, 'tokenizer_model_max_length', None)
        if tokenizer_model_max_length is not None:
            new_input_embeds = [x[:tokenizer_model_max_length] for x in new_input_embeds]
            new_labels = [x[:tokenizer_model_max_length] for x in new_labels]

        # Combine them
        max_len = max(x.shape[0] for x in new_input_embeds)
        batch_size = len(new_input_embeds)

        new_input_embeds_padded = []
        new_labels_padded = torch.full((batch_size, max_len), IGNORE_INDEX, dtype=new_labels[0].dtype, device=new_labels[0].device)
        attention_mask = torch.zeros((batch_size, max_len), dtype=attention_mask.dtype, device=attention_mask.device)
        position_ids = torch.zeros((batch_size, max_len), dtype=position_ids.dtype, device=position_ids.device)

        for i, (cur_new_embed, cur_new_labels) in enumerate(zip(new_input_embeds, new_labels)):
            cur_len = cur_new_embed.shape[0]
            if getattr(self.config, 'tokenizer_padding_side', 'right') == "left":
                new_input_embeds_padded.append(torch.cat((
                    torch.zeros((max_len - cur_len, cur_new_embed.shape[1]), dtype=cur_new_embed.dtype, device=cur_new_embed.device),
                    cur_new_embed
                ), dim=0))
                if cur_len > 0:
                    new_labels_padded[i, -cur_len:] = cur_new_labels
                    attention_mask[i, -cur_len:] = True
                    position_ids[i, -cur_len:] = torch.arange(0, cur_len, dtype=position_ids.dtype, device=position_ids.device)
            else:
                new_input_embeds_padded.append(torch.cat((
                    cur_new_embed,
                    torch.zeros((max_len - cur_len, cur_new_embed.shape[1]), dtype=cur_new_embed.dtype, device=cur_new_embed.device)
                ), dim=0))
                if cur_len > 0:
                    new_labels_padded[i, :cur_len] = cur_new_labels
                    attention_mask[i, :cur_len] = True
                    position_ids[i, :cur_len] = torch.arange(0, cur_len, dtype=position_ids.dtype, device=position_ids.device)

        new_input_embeds = torch.stack(new_input_embeds_padded, dim=0)

        if _labels is None:
            new_labels = None
        else:
            new_labels = new_labels_padded

        if _attention_mask is None:
            attention_mask = None
        else:
            attention_mask = attention_mask.to(dtype=_attention_mask.dtype)

        if _position_ids is None:
            position_ids = None

        return None, position_ids, attention_mask, past_key_values, new_input_embeds, new_labels

    def initialize_protein_tokenizer(self, model_args, tokenizer):
        if model_args.mm_use_prot_patch_token:
            tokenizer.add_tokens([DEFAULT_PROTEIN_PATCH_TOKEN], special_tokens=True)
            self.resize_token_embeddings(len(tokenizer))

        if model_args.mm_use_prot_start_end:
            num_new_tokens = tokenizer.add_tokens([DEFAULT_PROT_START_TOKEN, DEFAULT_PROT_END_TOKEN], special_tokens=True)
            self.resize_token_embeddings(len(tokenizer))

            if num_new_tokens > 0:
                input_embeddings = self.get_input_embeddings().weight.data
                output_embeddings = self.get_output_embeddings().weight.data

                input_embeddings_avg = input_embeddings[:-num_new_tokens].mean(
                    dim=0, keepdim=True)
                output_embeddings_avg = output_embeddings[:-num_new_tokens].mean(
                    dim=0, keepdim=True)

                input_embeddings[-num_new_tokens:] = input_embeddings_avg
                output_embeddings[-num_new_tokens:] = output_embeddings_avg

            if model_args.tune_mm_mlp_adapter:
                for p in self.get_input_embeddings().parameters():
                    p.requires_grad = True
                for p in self.get_output_embeddings().parameters():
                    p.requires_grad = False

            if model_args.pretrain_mm_mlp_adapter:
                mm_projector_weights = torch.load(model_args.pretrain_mm_mlp_adapter, map_location='cpu')
                embed_tokens_weight = mm_projector_weights['model.embed_tokens.weight']
                assert num_new_tokens == 2
                if input_embeddings.shape == embed_tokens_weight.shape:
                    input_embeddings[-num_new_tokens:] = embed_tokens_weight[-num_new_tokens:]
                elif embed_tokens_weight.shape[0] == num_new_tokens:
                    input_embeddings[-num_new_tokens:] = embed_tokens_weight
                else:
                    raise ValueError(f"Unexpected embed_tokens_weight shape. Pretrained: {embed_tokens_weight.shape}. Current: {input_embeddings.shape}. Numer of new tokens: {num_new_tokens}.")
        elif model_args.mm_use_prot_patch_token:
            if model_args.tune_mm_mlp_adapter:
                for p in self.get_input_embeddings().parameters():
                    p.requires_grad = False
                for p in self.get_output_embeddings().parameters():
                    p.requires_grad = False
