import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
import torchvision.models as models
from gym import spaces
from habitat import logger
from habitat_baselines.rl.ddppo.policy import resnet
from habitat_baselines.rl.ddppo.policy.resnet_policy import ResNetEncoder
import torchvision
import clip
from typing import Tuple


class ConvAdapter(nn.Module):
    """
    Convolutional adapter for depth features with bottleneck architecture.
    Uses 1x1 convolutions with residual connection and starts as identity.
    
    Args:
        in_ch: Number of input channels
        mid_ch: Number of middle/bottleneck channels  
        p_drop: Dropout probability
    """
    def __init__(self, in_ch: int, mid_ch: int = 64, p_drop: float = 0.1):
        super().__init__()
        self.norm = nn.GroupNorm(1, in_ch)  # Layer normalization via GroupNorm
        self.down = nn.Conv2d(in_ch, mid_ch, 1, bias=False)  # Bottleneck down
        self.act = nn.GELU()
        self.up = nn.Conv2d(mid_ch, in_ch, 1, bias=False)    # Bottleneck up
        self.drop = nn.Dropout2d(p_drop)
        
        # Initialize final conv weights to zero for identity startup
        nn.init.zeros_(self.up.weight)
        
    def forward(self, x):
        # Residual connection: x + drop(up(gelu(down(norm(x)))))
        residual = self.up(self.act(self.down(self.norm(x))))
        return x + self.drop(residual)


class VlnResnetDepthEncoder(nn.Module):
    def __init__(
        self,
        observation_space,
        output_size=128,
        checkpoint="NONE",
        backbone="resnet50",
        resnet_baseplanes=32,
        normalize_visual_inputs=False,
        trainable=False,
        spatial_output: bool = False,
        # Adapter parameters
        use_depth_adapters: bool = False,
        depth_adapter_width: int = 64,
        depth_adapter_dropout: float = 0.1,
        depth_adapter_stages: Tuple[int, ...] = (2, 3),
        train_ln_with_adapters: bool = True,
    ):
        super().__init__()
        self.use_depth_adapters = use_depth_adapters
        self.train_ln_with_adapters = train_ln_with_adapters
        
        self.visual_encoder = ResNetEncoder(
            spaces.Dict({"depth": observation_space.spaces["depth"]}),
            baseplanes=resnet_baseplanes,
            ngroups=resnet_baseplanes // 2,
            make_backbone=getattr(resnet, backbone),
            normalize_visual_inputs=normalize_visual_inputs,
        )

        # Store original trainable setting for compatibility
        self.original_trainable = trainable
        for param in self.visual_encoder.parameters():
            param.requires_grad_(trainable)
            
        # Add adapters if enabled
        if use_depth_adapters:
            self._add_adapters(depth_adapter_stages, depth_adapter_width, depth_adapter_dropout)
            self._register_adapter_hooks()
            logger.info(f"Added depth adapters to stages {depth_adapter_stages} with width {depth_adapter_width}")

        if checkpoint != "NONE":
            ddppo_weights = torch.load(checkpoint)

            weights_dict = {}
            for k, v in ddppo_weights["state_dict"].items():
                split_layer_name = k.split(".")[2:]
                if split_layer_name[0] != "visual_encoder":
                    continue

                layer_name = ".".join(split_layer_name[1:])
                weights_dict[layer_name] = v

            del ddppo_weights
            # Use strict=False when adapters are enabled, since checkpoint won't have adapter parameters
            strict_loading = not self.use_depth_adapters
            self.visual_encoder.load_state_dict(weights_dict, strict=strict_loading)

        self.spatial_output = spatial_output

        if not self.spatial_output:
            self.output_shape = (output_size,)
            # self.visual_fc = nn.Sequential(
            #     nn.Flatten(),
            #     nn.Linear(
            #         np.prod(self.visual_encoder.output_shape), output_size
            #     ),
            #     nn.ReLU(True),
            # )
            None
        else:
            self.spatial_embeddings = nn.Embedding(
                self.visual_encoder.output_shape[1]
                * self.visual_encoder.output_shape[2],
                64,
            )

            self.output_shape = list(self.visual_encoder.output_shape)
            self.output_shape[0] += self.spatial_embeddings.embedding_dim
            self.output_shape = tuple(self.output_shape)

    def _add_adapters(self, stages: Tuple[int, ...], width: int, dropout: float):
        """Add ConvAdapter modules to specified ResNet stages."""
        # Get the ResNet backbone from the visual encoder
        backbone = self.visual_encoder.backbone
        
        # Map stage numbers to layer names
        stage_layers = {
            1: backbone.layer1,
            2: backbone.layer2, 
            3: backbone.layer3,
            4: backbone.layer4
        }
        
        for stage_idx in stages:
            if stage_idx not in stage_layers:
                logger.warning(f"Invalid adapter stage {stage_idx}, skipping")
                continue
                
            layer = stage_layers[stage_idx]
            
            # Add adapter to each bottleneck block in this stage
            for i, block in enumerate(layer):
                # Try different ways to get output channels
                out_channels = None
                
                # Method 1: Check for conv3 (standard PyTorch Bottleneck blocks)
                if hasattr(block, 'conv3') and hasattr(block.conv3, 'out_channels'):
                    out_channels = block.conv3.out_channels
                # Method 2: Check for conv2 (BasicBlock)
                elif hasattr(block, 'conv2') and hasattr(block.conv2, 'out_channels'):
                    out_channels = block.conv2.out_channels
                # Method 3: Check habitat_baselines custom Bottleneck with 'convs' attribute
                elif hasattr(block, 'convs') and isinstance(block.convs, nn.Sequential):
                    # Get the last conv layer from the sequential
                    last_conv = None
                    for layer_module in reversed(block.convs):
                        if isinstance(layer_module, nn.Conv2d):
                            last_conv = layer_module
                            break
                    if last_conv is not None:
                        out_channels = last_conv.out_channels
                # Method 4: Use known ResNet50 channel dimensions (based on debug output)
                elif stage_idx == 1:
                    out_channels = 64   # ResNet50 layer1 output (actually 64*4=256 after bottleneck)
                elif stage_idx == 2:
                    out_channels = 256  # ResNet50 layer2 output (from debug: torch.Size([1, 256, 16, 16]))
                elif stage_idx == 3:
                    out_channels = 512  # ResNet50 layer3 output
                elif stage_idx == 4:
                    out_channels = 1024 # ResNet50 layer4 output
                
                if out_channels is None:
                    logger.warning(f"Could not determine output channels for block {i} in stage {stage_idx}")
                    continue
                    
                # Add adapter as an attribute to the block
                adapter = ConvAdapter(out_channels, width, dropout)
                setattr(block, 'depth_adapter', adapter)
                logger.debug(f"Added adapter to stage {stage_idx}, block {i} with {out_channels} channels")

    def _register_adapter_hooks(self):
        """Register forward hooks to apply adapters after block outputs."""
        if not self.use_depth_adapters:
            return
            
        backbone = self.visual_encoder.backbone
        stage_layers = [backbone.layer1, backbone.layer2, backbone.layer3, backbone.layer4]
        
        def create_adapter_hook(block):
            def adapter_hook(module, input, output):
                if hasattr(block, 'depth_adapter'):
                    return block.depth_adapter(output)
                return output
            return adapter_hook
        
        # Register hooks for blocks that have adapters
        for layer in stage_layers:
            for block in layer:
                if hasattr(block, 'depth_adapter'):
                    block.register_forward_hook(create_adapter_hook(block))

    def _apply_adapters_to_block_output(self, x, block):
        """Apply adapter to block output if it exists."""
        if hasattr(block, 'depth_adapter') and self.use_depth_adapters:
            return block.depth_adapter(x)
        return x

    def freeze_base_parameters(self):
        """Freeze all base ResNet parameters, keep only adapters trainable."""
        # Freeze all visual encoder parameters
        for param in self.visual_encoder.parameters():
            param.requires_grad_(False)
            
        # Unfreeze adapter parameters
        if self.use_depth_adapters:
            backbone = self.visual_encoder.backbone
            stage_layers = [backbone.layer1, backbone.layer2, backbone.layer3, backbone.layer4]
            
            for layer in stage_layers:
                for block in layer:
                    if hasattr(block, 'depth_adapter'):
                        for param in block.depth_adapter.parameters():
                            param.requires_grad_(True)
                            
                        # Optionally unfreeze normalization parameters in adapters
                        if self.train_ln_with_adapters and hasattr(block.depth_adapter, 'norm'):
                            for param in block.depth_adapter.norm.parameters():
                                param.requires_grad_(True)
                                
        logger.info("Froze base depth encoder parameters, kept adapters trainable")

    def get_adapter_parameters(self):
        """Get all adapter parameters for optimizer."""
        adapter_params = []
        if self.use_depth_adapters:
            backbone = self.visual_encoder.backbone
            stage_layers = [backbone.layer1, backbone.layer2, backbone.layer3, backbone.layer4]
            
            for layer in stage_layers:
                for block in layer:
                    if hasattr(block, 'depth_adapter'):
                        adapter_params.extend(block.depth_adapter.parameters())
                        
        return adapter_params

    def count_adapter_parameters(self):
        """Count trainable adapter parameters."""
        if not self.use_depth_adapters:
            return 0
            
        adapter_params = self.get_adapter_parameters()
        return sum(p.numel() for p in adapter_params if p.requires_grad)

    def forward(self, observations):
        """
        Args:
            observations: [BATCH, HEIGHT, WIDTH, CHANNEL]
        Returns:
            [BATCH, OUTPUT_SIZE]
        """
        if "depth_features" in observations:
            x = observations["depth_features"]
        else:
            x = self.visual_encoder(observations)

        if self.spatial_output:
            b, c, h, w = x.size()

            spatial_features = (
                self.spatial_embeddings(
                    torch.arange(
                        0,
                        self.spatial_embeddings.num_embeddings,
                        device=x.device,
                        dtype=torch.long,
                    )
                )
                .view(1, -1, h, w)
                .expand(b, self.spatial_embeddings.embedding_dim, h, w)
            )

            return torch.cat([x, spatial_features], dim=1)
        else:
            # return self.visual_fc(x)
            return x


class TorchVisionResNet50(nn.Module):
    r"""
    Takes in observations and produces an embedding of the rgb component.

    Args:
        observation_space: The observation_space of the agent
        output_size: The size of the embedding vector
        device: torch.device
    """

    def __init__(
        self,
        observation_space,
        output_size,
        device,
        spatial_output: bool = False,
    ):
        super().__init__()
        self.device = device
        self.resnet_layer_size = 2048
        linear_layer_input_size = 0
        if "rgb" in observation_space.spaces:
            self._n_input_rgb = observation_space.spaces["rgb"].shape[2]
            obs_size_0 = observation_space.spaces["rgb"].shape[0]
            obs_size_1 = observation_space.spaces["rgb"].shape[1]
            if obs_size_0 != 224 or obs_size_1 != 224:
                logger.warn(
                    "TorchVisionResNet50: observation size is not conformant to expected ResNet input size [3x224x224]"
                )
            linear_layer_input_size += self.resnet_layer_size
        else:
            self._n_input_rgb = 0

        if self.is_blind:
            self.cnn = nn.Sequential()
            return

        rgb_resnet = models.resnet50(pretrained=True)
        rgb_modules = list(rgb_resnet.children())[:-2]
        self.cnn = torch.nn.Sequential(*rgb_modules)

        # disable gradients for resnet, params frozen
        for param in self.cnn.parameters():
            param.requires_grad_(False)
        self.cnn.eval()

        self.spatial_output = spatial_output

        if not self.spatial_output:
            self.output_shape = (output_size,)
            # self.fc = nn.Linear(linear_layer_input_size, output_size)
            # self.activation = nn.ReLU()
            None
        else:
            class SpatialAvgPool(nn.Module):
                def forward(self, x):
                    x = F.adaptive_avg_pool2d(x, (4, 4))

                    return x
            self.cnn.avgpool = SpatialAvgPool()
            self.cnn.fc = nn.Sequential()
            self.spatial_embeddings = nn.Embedding(4 * 4, 64)
            self.output_shape = (
                self.resnet_layer_size + self.spatial_embeddings.embedding_dim,
                4,
                4,
            )

        # self.layer_extract = self.cnn._modules.get("avgpool")

        from torchvision import transforms
        self.rgb_transform = torch.nn.Sequential(
            transforms.ConvertImageDtype(torch.float),
            transforms.Normalize([0.485, 0.456, 0.406], [0.229, 0.224, 0.225]),
            )

    @property
    def is_blind(self):
        return self._n_input_rgb == 0

    def forward(self, observations):
        r"""Sends RGB observation through the TorchVision ResNet50 pre-trained
        on ImageNet. Sends through fully connected layer, activates, and
        returns final embedding.
        """

        def resnet_forward(observation):
            # resnet_output = torch.zeros(
            #     1, dtype=torch.float32, device=observation.device
            # )
            # def hook(m, i, o):
            #     resnet_output.set_(o)

            # output: [BATCH x RESNET_DIM]
            # h = self.layer_extract.register_forward_hook(hook)
            resnet_output = self.cnn(observation)
            # h.remove()
            return resnet_output

        if "rgb_features" in observations:
            resnet_output = observations["rgb_features"]
        else:
            # permute tensor to dimension [BATCH x CHANNEL x HEIGHT x WIDTH]
            rgb_observations = observations["rgb"].permute(0, 3, 1, 2)

            rgb_observations = self.rgb_transform(rgb_observations)
            # rgb_observations = rgb_observations / 255.0  # normalize RGB

            resnet_output = resnet_forward(rgb_observations.contiguous())

        if self.spatial_output:
            b, c, h, w = resnet_output.size()

            spatial_features = (
                self.spatial_embeddings(
                    torch.arange(
                        0,
                        self.spatial_embeddings.num_embeddings,
                        device=resnet_output.device,
                        dtype=torch.long,
                    )
                )
                .view(1, -1, h, w)
                .expand(b, self.spatial_embeddings.embedding_dim, h, w)
            )

            return torch.cat([resnet_output, spatial_features], dim=1)#.to(self.device)
        else:
            # return self.activation(
            #     self.fc(torch.flatten(resnet_output, 1))
            # )  # [BATCH x OUTPUT_DIM]
            return resnet_output


class CLIPEncoder(nn.Module):
    r"""
    Takes in observations and produces an embedding of the rgb component.

    Args:
        observation_space: The observation_space of the agent
        output_size: The size of the embedding vector
        device: torch.device
    """

    def __init__(
        self, device,
    ):
        super().__init__()
        self.model, _ = clip.load("ViT-B/32", device=device)
        for param in self.model.parameters():
            param.requires_grad_(False)
        self.model.eval()

        from torchvision import transforms
        self.rgb_transform = torch.nn.Sequential(
            transforms.ConvertImageDtype(torch.float),
            transforms.Normalize([0.48145466, 0.4578275, 0.40821073], [0.26862954, 0.26130258, 0.27577711]),
            )

    def forward(self, observations):
        r"""Sends RGB observation through the TorchVision ResNet50 pre-trained
        on ImageNet. Sends through fully connected layer, activates, and
        returns final embedding.
        """
        rgb_observations = observations["rgb"].permute(0, 3, 1, 2)
        rgb_observations = self.rgb_transform(rgb_observations)
        output = self.model.encode_image(rgb_observations.contiguous())

        return output.float() # to fp32