import cv2
import numpy as np

# 1. Apply Gaussian noise
def apply_gaussian_noise(normalized_image, intensity=0.6):
    sigma = intensity * 0.1  # standard deviation of the noise
    noise = np.random.normal(0, sigma, normalized_image.shape)
    noisy_image = normalized_image + noise
    noisy_image = np.clip(noisy_image, 0, 1)
    return noisy_image

# 2. Simulate missing data
def apply_missing_data(normalized_image, intensity=0.6):
    mask = np.random.random(normalized_image.shape) < (intensity * 0.2)
    corrupted_image = normalized_image.copy()
    corrupted_image[mask] = 0  # 0 is the missing data
    return corrupted_image

# 3. Apply multipath interference
def apply_multipath_interference(depth_image, intensity=0.6):
    # Detect edges to focus interference on complex geometry
    sobel_x = cv2.Sobel(depth_image, cv2.CV_64F, 1, 0, ksize=3)
    sobel_y = cv2.Sobel(depth_image, cv2.CV_64F, 0, 1, ksize=3)
    edge_magnitude = np.sqrt(sobel_x**2 + sobel_y**2)
    edge_mask = edge_magnitude > np.percentile(edge_magnitude, 90)  # Top 10% edges

    # Base weight for interference
    base_weight = intensity * 0.5
    # Spatially varying weight (stronger near edges)
    weight = np.ones_like(depth_image) * base_weight * 0.7
    weight[edge_mask] = base_weight  # Stronger interference at edges

    # Generate secondary depth with random offset
    secondary_depth = depth_image + np.random.uniform(0, 1, depth_image.shape)
    secondary_depth = np.clip(secondary_depth, 0, 1)

    # Blend depths
    corrupted_image = (1 - weight) * depth_image + weight * secondary_depth
    return np.clip(corrupted_image, 0, 1)


# 4. Apply depth quantization
def apply_depth_quantization(normalized_image, intensity=0.6):
    delta = intensity * 0.05  # Quantization step size
    if delta == 0:
        return normalized_image.copy()
    quantized_image = np.round(normalized_image / delta) * delta
    quantized_image = np.clip(quantized_image, 0, 1)
    return quantized_image


