import os
import tqdm
import numpy as np
from PIL import Image
import func as fc
import uuid
from scipy.ndimage import gaussian_filter, uniform_filter

### Manipulation functions ###

# Apply defocus blur effect to an image
def defocus_blur(image, intensity=0.6):
    if intensity == 0:
        return image.copy()
    # adjust the sigma value of the gaussian blur based on intensity, the maximum sigma is 20
    sigma = 20 * intensity
    blurred = np.zeros_like(image, dtype=np.float32)
    for channel in range(3):
        blurred[:, :, channel] = gaussian_filter(image[:, :, channel], sigma=sigma)
    # linear interpolation between the original image and the blurred image using intensity
    result = (1 - intensity) * image.astype(np.float32) + intensity * blurred
    return np.clip(result, 0, 255).astype(np.uint8)

def spatter(image, intensity=0.6):
    if intensity == 0:
        return image.copy()
    # add speckle noise to the image
    spattered = image.copy()
    height, width, _ = image.shape
    # adjust the number of noise points based on intensity, the maximum is 2% of the pixels
    num_spots = int(height * width * 0.02 * intensity)
    for _ in range(num_spots):
        x = np.random.randint(0, width)
        y = np.random.randint(0, height)
        # scale the size of the noise points based on intensity
        size = np.random.randint(1, max(2, int(5 * intensity)))
        spattered[max(0, y-size):min(height, y+size), max(0, x-size):min(width, x+size)] = np.random.randint(0, 256, (3,))
    return spattered

# Draw a black circle or a blurred circular region at the center of the image
def foreign_object(image, radius_height_ratio=0.2, blur=False):
    # the radius of the circle is relative to the image's height
    output_image = image.copy()
    height, width, channels = image.shape
    center_x = width // 2
    center_y = height // 2

    # Calculate actual radius in pixels based on the height ratio
    actual_radius_pixels = radius_height_ratio * height

    Y, X = np.ogrid[:height, :width]
    dist_from_center = np.sqrt((X - center_x)**2 + (Y - center_y)**2)

    mask = dist_from_center <= actual_radius_pixels

    if blur:
        # Apply Gaussian blur to the circular region
        # Sigma for blur is proportional to the radius for consistent effect
        # Ensure sigma is at least 1 to have a visible blur effect
        sigma = max(1.0, actual_radius_pixels / 5.0) 
        blurred_image = np.zeros_like(image, dtype=image.dtype)
        for c in range(channels):
            blurred_image[:,:,c] = gaussian_filter(image[:,:,c], sigma=sigma)
        
        # Apply the blur only within the masked (circular) area
        # The mask needs to be broadcastable to the image shape for np.where
        mask_3d = np.stack([mask]*channels, axis=-1)
        output_image = np.where(mask_3d, blurred_image, output_image)
    else:
        output_image[mask] = 0  # Black color

    return output_image

# Motion Blur
def motion_blur(image, intensity=0.6):
    # simulate motion blur, intensity controls the blurring degree
    if intensity == 0:
        return image.copy()
    # adjust the size of the blur kernel based on intensity, the maximum is 60
    kernel_size = int(60 * intensity)
    if kernel_size < 1:
        return image.copy()
    blurred = np.zeros_like(image, dtype=np.float32)
    for channel in range(3):
        blurred[:, :, channel] = uniform_filter(image[:, :, channel], size=(1, kernel_size))
    # linear interpolation between the original image and the blurred image using intensity
    result = (1 - intensity) * image.astype(np.float32) + intensity * blurred
    return np.clip(result, 0, 255).astype(np.uint8)


# Add enhanced flare effect
def flare_effect(image, intensity=1.0):
    # this version is modified, the effect of intensity=0.6 is equivalent to the effect of intensity=1.0 in the original version
    # the conversion ratio is 1.0 / 0.6 = 5/3
    # add enhanced flare effect, intensity controls the flare intensity
    if intensity == 0:
        return image.copy()
    
    flared = image.astype(np.float32)
    height, width, _ = image.shape
    
    # define the center of the flare
    center_x = np.random.randint(width // 4, 3 * width // 4)
    center_y = np.random.randint(height // 4, 3 * height // 4)
    
    # create radial gradient flare, increase the range and intensity
    Y, X = np.ogrid[:height, :width]
    distance = np.sqrt((X - center_x)**2 + (Y - center_y)**2)
    max_distance = np.sqrt(width**2 + height**2) / 2
    # use the adjusted intensity
    flare_intensity = np.exp(-distance / (max_distance / 2)) * 400 * intensity
    flare_intensity = np.clip(flare_intensity, 0, 400)
    
    # use the screen blending mode to apply the flare
    for channel in range(3):
        # use the adjusted intensity
        intensity_scale = (0.9 if channel == 0 else 0.8 if channel == 1 else 0.7) * intensity
        flare_channel = flare_intensity * intensity_scale
        # screen blending: result = 255 - (255 - image) * (255 - flare) / 255
        flared[:, :, channel] = 255 - (255 - flared[:, :, channel]) * (255 - flare_channel) / 255
    
    # add stronger flare effect
    glow_size = max(width, height) // 10  # increase the range of the glow
    glow = np.zeros((height, width), dtype=np.float32)
    glow[center_y, center_x] = 255
    glow = gaussian_filter(glow, sigma=glow_size)
    # use the adjusted intensity
    glow = np.clip(glow / glow.max() * 200 * intensity, 0, 255)  # increase the intensity of the glow
    
    # use the screen blending mode to apply the glow
    for channel in range(3):
        flared[:, :, channel] = 255 - (255 - flared[:, :, channel]) * (255 - glow) / 255
    
    return np.clip(flared, 0, 255).astype(np.uint8)

# Simulate a low lighting gradient effect with 50 distinct sections
def low_lighting_gradient(image, intensity=1.0):
    # the image is divided into 50 equal sections from top to bottom,
    # with intensity starting at 0.5 and gradually increasing to maximum.
    
    # Section 1 (0-2%): 50% of maximum darkening
    # Section 2 (2-4%): 51% of maximum darkening
    # Section 3 (4-6%): 52% of maximum darkening
    # ...and so on...
    # Section 50 (98-100%): 99% of maximum darkening
    
    # the input image as a numpy array
    # intensity: Controls the overall strength of the darkening effect (0.0-1.0)
    # Image with 50-step progressively darkening effect starting at 50% darkening
    
    if intensity == 0:
        return image.copy()
    
    height, width, _ = image.shape
    darkened = image.astype(np.float32)
    
    # Create a step array with appropriate scaling for each section
    scaling = np.ones((height, 1, 1), dtype=np.float32)
    
    # Define the darkest possible value (at max intensity)
    min_scale = 1 - intensity * (1 - 0.01)  # 0.01 = 1% of original brightness at max darkness
    
    # Create 50 sections with progressive darkening
    for i in range(50):
        section_start = int(height * (i / 50))
        section_end = int(height * ((i + 1) / 50))
        
        # Calculate darkness for this section (0.5 to 0.99)
        # The top section (i=0) starts at 50% darkening, bottom section (i=49) has 99% darkening
        darkness_percent = 0.5 + (i / 100)
        
        # Calculate the scaling factor for this section
        section_scale = 1.0 - (1.0 - min_scale) * darkness_percent
        
        # Apply the scaling to this section
        scaling[section_start:section_end] = section_scale
    
    # Apply the scaling to the image
    darkened = darkened * scaling
    
    # Ensure proper uint8 output
    return np.clip(darkened, 0, 255).astype(np.uint8)

# Add noise and gradient low lighting effect to the input image
def add_noise(image, intensity=0.6, noise_type='T1pro', low_light_intensity=1.0):
    # add noise and gradient low lighting effect to the input image, intensity controls the noise strength, low_light_intensity controls the gradient low lighting strength
    # the input image is a numpy array, shape (height, width, 3), type uint8, range [0, 255]
    # intensity: noise strength, range [0.0, 1.0], 0 means no noise, 1 means maximum strength
    # noise_type: noise model, supports 'T1pro' or 'Sony'
    # low_light_intensity: gradient low lighting strength, range [0.0, 1.0], 0 means no low lighting effect, 1 means maximum darkening
    # if None, use the value of intensity
    # the image with noise and gradient low lighting effect, numpy array, shape (height, width, 3), type uint8, range [0, 255]
    
    if intensity == 0 and (low_light_intensity is None or low_light_intensity == 0):
        return image.copy()
    if not 0.0 <= intensity <= 1.0:
        raise ValueError("Noise intensity must be between 0.0 and 1.0")
    if low_light_intensity is not None and not 0.0 <= low_light_intensity <= 1.0:
        raise ValueError("Low light intensity must be between 0.0 and 1.0")
    
    # use intensity as the default value of low_light_intensity
    low_light_intensity = intensity if low_light_intensity is None else low_light_intensity
    
    # convert the input image to float32, range [0, 1]
    im = image.astype(np.float32) / 255.0
    
    # apply the gradient low lighting effect
    if low_light_intensity > 0:
        height, width, _ = im.shape
        # create a scaling array, simulate the gradient darkening from top to bottom
        scaling = np.ones((height, 1, 1), dtype=np.float32)
        min_scale = 1 - low_light_intensity * (1 - 0.01)  # the darkest part is 1% of the original brightness
        for i in range(50):
            section_start = int(height * (i / 50))
            section_end = int(height * ((i + 1) / 50))
            darkness_percent = 0.5 + (i / 100)  # from 50% to 99% darkening
            section_scale = 1.0 - (1.0 - min_scale) * darkness_percent
            scaling[section_start:section_end] = section_scale
        im = im * scaling
    
    # load the noise model and add noise
    if intensity > 0:
        noise_model = np.load('./jointDistribution.npy', allow_pickle=True).item()
        nsim, _ = fc.getNoisePair(noise_model, im, noise_type)
        # use intensity to interpolate the noise
        im = (1 - intensity) * im + intensity * nsim
    
    # convert back to uint8, range [0, 255]
    return np.clip(im * 255.0, 0, 255).astype(np.uint8)

# Randomly return a black image or the original image
def black_out(image, intensity=0.6):
    # randomly return a black image or the original image
    # with probability `blackout_prob` the function returns a black image; otherwise
    # it passes the input image through unchanged.  The black image will have the
    # same shape and dtype as the input.
    # image: input image array
    # blackout_prob: probability of returning a black image, range [0, 1], default 0.5
    # the function returns either the original image (`image.copy()`) or a black image consisting of zeros
    
    if not 0.0 <= intensity <= 1.0:
        raise ValueError(f"intensity must be between 0 and 1, got {intensity}")
    # Decide whether to blackout or keep the image as is.
    if np.random.rand() < intensity:
        return np.zeros_like(image)
    else:
        # Return a copy to prevent accidental modification of the original.
        return image.copy()
