library(tidyverse)
library(lubridate)

dir <- commandArgs(trailingOnly=TRUE)[[1]]
suffix <- "-ckpt"
# suffix <- ""
print(sprintf("Reading files in %s...", dir))

dispatches <-
  sprintf("dispatch%s.csv", suffix) %>%
  file.path(dir, .) %>%
  read_csv(col_types=cols()) %>%
  mutate(cost=ifelse(is_idle, idle_cost, cost)) %>%
  mutate(is_treated=is_treated)

requests <-
  sprintf("summary%s.csv", suffix) %>%
  file.path(dir, .) %>%
  read_csv(col_types=cols()) %>%
  mutate(is_treated=is_treated)

print(sprintf("Finished reading files in %s.", dir))

changed <- (dispatches
  %>% select(rider_id, driver_id, is_treated)
  %>% spread(is_treated, driver_id)
  %>% mutate(is_changed=`TRUE` != `FALSE`)
  %>% select(rider_id, is_changed))

results <- (requests
  %>% select(rider_id, is_treated)
  %>% left_join(dispatches %>% select(rider_id, is_treated),
              by=c("rider_id", "is_treated")))

df <- (requests
  %>% select(ts, rider_id, accepted, price,
           drivers_total_0, drivers_total_1,
           drivers_total_2, drivers_total_3,
           slack, price, is_treated)
  %>% replace_na(list(drivers_total_3=0.,
                    drivers_total_2=0.,
                    drivers_total_1=0.,
                    drivers_total_0=0.,
                    slack=0.))
  %>% mutate(
      ts=as.POSIXct(ts, origin="1970-01-01"),
      drivers_total=
        drivers_total_0 + drivers_total_1
      + drivers_total_2 + drivers_total_3,
      mean_capacity=
        (drivers_total_3 * 3 + drivers_total_2 * 2 + drivers_total_1)
      / drivers_total
    )
  %>% filter(drivers_total > 0, !is.na(accepted))
  %>% left_join(changed, by=c("rider_id"))
  %>% mutate(is_changed=ifelse(is.na(is_changed), FALSE, TRUE),
           hour=hour(ts))
)

dispatch_df <- (dispatches
  %>% select(rider_id, is_treated, cost, is_idle, capacity, savings))


join_to_dispatches <- function(df)
  (df
    %>% left_join(dispatch_df, by=c("rider_id", "is_treated"))
    %>% replace_na(list(cost=0, is_idle=FALSE, capacity=0, savings=0))
    %>% arrange(ts)
    %>% mutate(profit=ifelse(accepted, price - cost, 0.),
             action_indicator_tr=is_changed & is_treated,
             action_indicator_co=is_changed & !is_treated,
             drivers_interacted_tr=drivers_total * action_indicator_tr,
             drivers_interacted_co=drivers_total * action_indicator_co))

treatment <- (df
  %>% mutate(is_treated=TRUE)
  %>% join_to_dispatches)
control <- (df
  %>% mutate(is_treated=FALSE)
  %>% join_to_dispatches)
actual <- df %>% join_to_dispatches

write_csv(treatment, file.path(dir, "treatment.csv"))
write_csv(control, file.path(dir, "control.csv"))
write_csv(actual, file.path(dir, "actual.csv"))
