import copy
from components.episode_buffer import EpisodeBatch
from modules.mixers.dmaq_general import DMAQer
from modules.mixers.dmaq_qatten import DMAQ_QattenMixer
import torch.nn.functional as F
import torch as th
from torch.optim import RMSprop, Adam
import numpy as np
from modules.w_predictor.predict_net import Predict_Network_OS, Predict_Network

class COIN_DMAQ_qattenLearner:
    def __init__(self, mac, scheme, logger, args):
        self.args = args
        self.mac = mac
        self.logger = logger

        self.rnn_hidden_dim = self.args.rnn_hidden_dim
        self.n_agents = args.n_agents
        self.n_actions = args.n_actions
        self.state_dim = int(np.prod(args.state_shape))
        self.obs_dim = args.obs_shape

        self.params = list(mac.parameters())

        self.last_target_update_episode = 0

        self.mixer = None
        if args.mixer is not None:
            if args.mixer == "dmaq":
                self.mixer = DMAQer(args)
            elif args.mixer == 'dmaq_qatten':
                self.mixer = DMAQ_QattenMixer(args)
            else:
                raise ValueError("Mixer {} not recognised.".format(args.mixer))
            self.params += list(self.mixer.parameters())
            self.target_mixer = copy.deepcopy(self.mixer)

        if self.args.optimizer == 'Adam':
            self.optimiser = Adam(params=self.params,  lr=args.lr)
        else:
            self.optimiser = RMSprop(params=self.params, lr=args.lr, alpha=args.optim_alpha, eps=args.optim_eps)

        # a little wasteful to deepcopy (e.g. duplicates action selector), but should work for any MAC
        self.target_mac = copy.deepcopy(mac)

        self.log_stats_t = -self.args.learner_log_interval - 1

        
        self.show_k = self.args.show_k

        self.eval_predict_os = Predict_Network_OS((args.rnn_hidden_dim+args.n_actions)* self.n_agents, args.predict_net_dim,
                                                              self.obs_dim,self.state_dim,self.n_agents)
        self.target_predict_os = copy.deepcopy(self.eval_predict_os)
        self.predictor_params = list(self.eval_predict_os.parameters())

        self.eval_predict_with_action = Predict_Network(
                args.rnn_hidden_dim + args.n_actions+ args.n_actions, args.predict_net_dim, args.obs_shape)
        self.target_predict_with_action = copy.deepcopy(self.eval_predict_with_action)
        self.eval_predict_without_action = Predict_Network(
                args.rnn_hidden_dim + args.n_actions , args.predict_net_dim, args.obs_shape)
        self.target_predict_without_action = copy.deepcopy(self.eval_predict_without_action)
        self.predictor_params += list(self.eval_predict_with_action.parameters())
        self.predictor_params += list(self.eval_predict_without_action.parameters())
        self.predictor_optimiser = Adam(params=self.predictor_params,  lr=args.predictor_lr)

        self.beta1 = self.args.beta1
        self.beta2 = self.args.beta2
        self.beta = self.args.beta
        self.decay_stats_cur_t = 0
        self.decay_stats_inf_t = 0
    

    def sub_train(self, batch: EpisodeBatch, t_env: int, episode_num: int, mac, mixer, optimiser, params,
                  show_demo=False, save_data=None):
        # Get the relevant quantities
        rewards = batch["reward"][:, :-1]
        actions = batch["actions"][:, :-1]
        terminated = batch["terminated"][:, :-1].float()
        mask = batch["filled"][:, :-1].float()
        mask[:, 1:] = mask[:, 1:] * (1 - terminated[:, :-1])
        avail_actions = batch["avail_actions"]
        actions_onehot = batch["actions_onehot"][:, :-1]

        obs = batch["obs"][:, :-1]
        obs_next = batch["obs"][:, 1:]
        state = batch["state"][:, :-1]
        state_next = batch["state"][:, 1:]
        mask_next = batch["filled"][:, 1:].float()
        mask_next[:, 1:] = mask_next[:, 1:] * (1 - terminated[:, 1:])
        mask_next_obs = mask_next.repeat(1,1,self.obs_dim*self.n_agents)
        mask_next_state = mask_next.repeat(1,1,self.state_dim)
        bs = obs.shape[0]
        obs_next_flatten =obs_next.view(bs,-1,self.obs_dim* self.n_agents)
        obs_next_clone = obs_next.clone()



        mac_out = []
        hidden_out = []
        mac.init_hidden(batch.batch_size)
        for t in range(batch.max_seq_length):
            agent_outs, hidden_states = self.mac.forward_hid(batch, t=t)
            mac_out.append(agent_outs)
            hidden_out.append(hidden_states)
        mac_out = th.stack(mac_out, dim=1)  # Concat over time
        hidden_out = th.stack(hidden_out, dim=1)
        hidden_out_clone = hidden_out.clone()[:,:-1]

        # Pick the Q-Values for the actions taken by each agent
        chosen_action_qvals = th.gather(mac_out[:, :-1], dim=3, index=actions).squeeze(3)  # Remove the last dim

        x_mac_out = mac_out.clone().detach()
        x_mac_out[avail_actions == 0] = -9999999
        max_action_qvals, max_action_index = x_mac_out[:, :-1].max(dim=3)

        max_action_index = max_action_index.detach().unsqueeze(3)
        is_max_action = (max_action_index == actions).int().float()

        if show_demo:
            q_i_data = chosen_action_qvals.detach().cpu().numpy()
            q_data = (max_action_qvals - chosen_action_qvals).detach().cpu().numpy()
            # self.logger.log_stat('agent_1_%d_q_1' % save_data[0], np.squeeze(q_data)[0], t_env)
            # self.logger.log_stat('agent_2_%d_q_2' % save_data[1], np.squeeze(q_data)[1], t_env)

        # Calculate the Q-Values necessary for the target
        target_mac_out = []
        target_hidden_out = []
        self.target_mac.init_hidden(batch.batch_size)
        for t in range(batch.max_seq_length):
            target_agent_outs ,target_hidden_states = self.target_mac.forward_hid(batch, t=t)
            target_mac_out.append(target_agent_outs)
            target_hidden_out.append(target_hidden_states)
        # We don't need the first timesteps Q-Value estimate for calculating targets
        target_mac_out = th.stack(target_mac_out[1:], dim=1)  # Concat across time
        target_hidden_out = th.stack(target_hidden_out[1:], dim=1)

        # Mask out unavailable actions
        target_mac_out[avail_actions[:, 1:] == 0] = -9999999

        # Max over target Q-Values
        if self.args.double_q:
            # Get actions that maximise live Q (for double q-learning)
            mac_out_detach = mac_out.clone().detach()
            mac_out_detach[avail_actions == 0] = -9999999
            cur_max_actions = mac_out_detach[:, 1:].max(dim=3, keepdim=True)[1]
            target_chosen_qvals = th.gather(target_mac_out, 3, cur_max_actions).squeeze(3)
            target_max_qvals = target_mac_out.max(dim=3)[0]
            target_next_actions = cur_max_actions.detach()

            cur_max_actions_onehot = th.zeros(cur_max_actions.squeeze(3).shape + (self.n_actions,)).cuda()
            cur_max_actions_onehot = cur_max_actions_onehot.scatter_(3, cur_max_actions, 1)
        else:
            # Calculate the Q-Values necessary for the target
            target_mac_out = []
            self.target_mac.init_hidden(batch.batch_size)
            for t in range(batch.max_seq_length):
                target_agent_outs = self.target_mac.forward(batch, t=t)
                target_mac_out.append(target_agent_outs)
            # We don't need the first timesteps Q-Value estimate for calculating targets
            target_mac_out = th.stack(target_mac_out[1:], dim=1)  # Concat across time
            target_max_qvals = target_mac_out.max(dim=3)[0]


        # intrinsic
        with th.no_grad():
            actions_one_hot = batch["actions_onehot"][:, :-1] # bs,t, n_agents, n_actions
            actions_one_hot_clone =  actions_one_hot.clone()
            # intrinsic curiosity
            os_predict_input = th.cat([hidden_out_clone,actions_one_hot_clone],-1).view(bs,-1,(self.rnn_hidden_dim+self.n_actions)* self.n_agents)       
            predict_obs_next, predict_state_next=self.target_predict_os(os_predict_input)
            curiosity_intrinsic_reward = th.sum((((predict_obs_next-obs_next_flatten)* mask_next_obs)**2),dim=-1,keepdim=True)  + \
                                   th.sum(((predict_state_next-state_next)* mask_next_state)**2,dim=-1,keepdim=True)  # bs,t,1
            curiosity_intrinsic_reward = curiosity_intrinsic_reward* self.beta2
            # intrinsic influence
            
            obs_diverge = []
            for i in range(self.n_agents):
                actions_one_hot_i_pre = th.cat((actions_one_hot_clone[:,:,:i],actions_one_hot_clone[:,:,i+1:]),dim=2)
                actions_one_hot_i = actions_one_hot_clone[:,:,i].unsqueeze(-2).repeat(1,1,self.n_agents-1, 1)
                obs_next_i = th.cat((obs_next_clone[:,:,:i],obs_next_clone[:,:,i+1:]),dim=2)
                hidden_out_without_i = th.cat((hidden_out_clone[:,:,:i],hidden_out_clone[:,:,i+1:]),dim=2)
                
                influence_inputs_i_without_action= th.cat((hidden_out_without_i,actions_one_hot_i_pre),dim=-1)
                influence_inputs_i_with_action = th.cat((hidden_out_without_i,actions_one_hot_i_pre,actions_one_hot_i),dim=-1)


                log_p_o_i = self.target_predict_without_action.get_log_pi(
                influence_inputs_i_without_action,obs_next_i)

                log_q_o_i = self.target_predict_with_action.get_log_pi(
                   influence_inputs_i_with_action, obs_next_i)
                obs_diverge_i = self.args.beta * th.sum(log_q_o_i, -1, keepdim=True) \
                    - th.sum(log_p_o_i, -1, keepdim=True)  # bs, t, 1
                obs_diverge.append(obs_diverge_i * mask_next *self.args.beta1)
            influence_intrinsic_reward= th.stack(obs_diverge,dim=2).squeeze(-1) # bs, t, n_agents,1

        # Mix
        if mixer is not None:
            if self.args.mixer == "dmaq_qatten":
                ans_chosen, q_attend_regs, head_entropies = \
                    mixer(chosen_action_qvals-influence_intrinsic_reward, batch["state"][:, :-1], is_v=True)
                ans_adv, _, _ = mixer(chosen_action_qvals-influence_intrinsic_reward, batch["state"][:, :-1], actions=actions_onehot,
                                      max_q_i=max_action_qvals, is_v=False)
                chosen_action_qvals = ans_chosen + ans_adv
            else:
                if self.show_k:
                    ans_chosen, k= mixer(chosen_action_qvals-influence_intrinsic_reward, batch["state"][:, :-1], is_v=True)
                    ans_adv, _ = mixer(chosen_action_qvals, batch["state"][:, :-1], actions=actions_onehot,
                                max_q_i=max_action_qvals, is_v=False)

                else: 
                    ans_chosen = mixer(chosen_action_qvals-influence_intrinsic_reward, batch["state"][:, :-1], is_v=True)
                    ans_adv = mixer(chosen_action_qvals, batch["state"][:, :-1], actions=actions_onehot,
                                max_q_i=max_action_qvals, is_v=False)
               
                chosen_action_qvals = ans_chosen + ans_adv

            if self.args.double_q:
                if self.args.mixer == "dmaq_qatten":
                    target_chosen, _, _ = self.target_mixer(target_chosen_qvals, batch["state"][:, 1:], is_v=True)
                    target_adv, _, _ = self.target_mixer(target_chosen_qvals, batch["state"][:, 1:],
                                                         actions=cur_max_actions_onehot,
                                                         max_q_i=target_max_qvals, is_v=False)
                    target_max_qvals = target_chosen + target_adv
                else:
                    if self.show_k:
                        target_chosen, _ = self.target_mixer(target_chosen_qvals, batch["state"][:, 1:], is_v=True)
                        target_adv, _ = self.target_mixer(target_chosen_qvals, batch["state"][:, 1:],
                                                   actions=cur_max_actions_onehot,
                                                   max_q_i=target_max_qvals, is_v=False)
                    else:
                        target_chosen = self.target_mixer(target_chosen_qvals, batch["state"][:, 1:], is_v=True)
                        target_adv = self.target_mixer(target_chosen_qvals, batch["state"][:, 1:],
                                                   actions=cur_max_actions_onehot,
                                                   max_q_i=target_max_qvals, is_v=False)
                    target_max_qvals = target_chosen + target_adv
            else:
                target_max_qvals = self.target_mixer(target_max_qvals, batch["state"][:, 1:], is_v=True)

        # Calculate 1-step Q-Learning targets
        targets = rewards +  curiosity_intrinsic_reward + self.args.gamma * (1 - terminated) * target_max_qvals
        if show_demo:
            tot_q_data = chosen_action_qvals.detach().cpu().numpy()
            tot_target = targets.detach().cpu().numpy()
            print('action_pair_%d_%d' % (save_data[0], save_data[1]), np.squeeze(q_data[:, 0]),
                  np.squeeze(q_i_data[:, 0]), np.squeeze(tot_q_data[:, 0]), np.squeeze(tot_target[:, 0]))
            self.logger.log_stat('action_pair_%d_%d' % (save_data[0], save_data[1]),
                                 np.squeeze(tot_q_data[:, 0]), t_env)
            return

        # Td-error
        td_error = (chosen_action_qvals - targets.detach())

        mask = mask.expand_as(td_error)

        # 0-out the targets that came from padded data
        masked_td_error = td_error * mask

        # Normal L2 loss, take mean over actual data
        if self.args.mixer == "dmaq_qatten":
            loss = (masked_td_error ** 2).sum() / mask.sum() + q_attend_regs
        else:
            loss = (masked_td_error ** 2).sum() / mask.sum()

        masked_hit_prob = th.mean(is_max_action, dim=2) * mask
        hit_prob = masked_hit_prob.sum() / mask.sum()


        # kl for vae
        q_normal = th.normal(0, 1, size=hidden_out.size()).to(hidden_out.device)
        Kl_hidden_out = 0.0001 * F.kl_div(th.log_softmax(hidden_out,-1), th.softmax(q_normal,-1),reduction='mean').sum()
        loss += Kl_hidden_out

        # Optimise
        optimiser.zero_grad()
        loss.backward()
        grad_norm = th.nn.utils.clip_grad_norm_(params, self.args.grad_norm_clip)
        optimiser.step()


        # predictor error
        actions_one_hot = batch["actions_onehot"][:, :-1] # bs,t, n_agents, n_actions
        actions_one_hot_clone =  actions_one_hot.clone()
        # os predictor error
        os_predict_input = th.cat([hidden_out_clone,actions_one_hot_clone],-1).view(bs,-1,(self.rnn_hidden_dim+self.n_actions)* self.n_agents).detach()        
        eval_predict_obs_next, eval_predict_state_next=self.eval_predict_os(os_predict_input)
        predictor_os_diff = th.sum((((eval_predict_obs_next-obs_next_flatten) * mask_next_obs)**2),dim=-1,keepdim=True) + \
                                   th.sum(((eval_predict_state_next-state_next)* mask_next_state)**2,dim=-1,keepdim=True) # bs,t,1
        predictor_os_loss =     predictor_os_diff.sum() / mask_next.sum()
        
        #inf predictor error    
        eval_obs_diverge = []
        p_o_loss= th.tensor(0.0).to(loss.device)
        q_o_loss=th.tensor(0.0).to(loss.device)
        for i in range(self.n_agents):   
            actions_one_hot_i_pre = th.cat((actions_one_hot_clone[:,:,:i],actions_one_hot_clone[:,:,i+1:]),dim=2)
            actions_one_hot_i = actions_one_hot_clone[:,:,i].unsqueeze(-2).repeat(1,1,self.n_agents-1, 1)
            obs_next_i = th.cat((obs_next_clone[:,:,:i],obs_next_clone[:,:,i+1:]),dim=2)
            hidden_out_without_i = th.cat((hidden_out_clone[:,:,:i],hidden_out_clone[:,:,i+1:]),dim=2).detach()
            influence_inputs_i_without_action= th.cat((hidden_out_without_i,actions_one_hot_i_pre),dim=-1).detach()

            influence_inputs_i_with_action = th.cat((hidden_out_without_i,actions_one_hot_i_pre,actions_one_hot_i),dim=-1).detach()
            p_o_i_loss = self.eval_predict_without_action.update(influence_inputs_i_without_action, obs_next_i, mask_next)
            q_o_i_loss = self.eval_predict_with_action.update( influence_inputs_i_with_action, obs_next_i, mask_next)
            p_o_loss += p_o_i_loss
            q_o_loss += q_o_i_loss

        predictor_loss = predictor_os_loss + p_o_loss +q_o_loss
        # Optimise
        self.predictor_optimiser.zero_grad()
        predictor_loss.backward()
        predictor_loss_grad_norm = th.nn.utils.clip_grad_norm_(self.predictor_params, self.args.grad_norm_clip)
        self.predictor_optimiser.step()


        if self.args.influence_exp_decay:
            if t_env - self.decay_stats_inf_t >= self.args.influence_exp_decay_cycle:
                if self.args.influence_exp_decay_rate <= 1.0:
                    if self.beta1 > self.args.influence_exp_decay_stop:
                        self.beta1 = self.beta1 * self.args.influence_exp_decay_rate
                        
                    else:
                        self.beta1 = self.args.influence_exp_decay_stop
                else:
                    if self.beta1 < self.args.influence_exp_decay_stop:
                        self.beta1 = self.beta1 * self.args.influence_exp_decay_rate
                    else:
                        self.beta1 = self.args.influence_exp_decay_stop
                self.decay_stats_inf_t=t_env
        if self.args.curiosity_exp_decay:
            if t_env - self.decay_stats_cur_t >= self.args.curiosity_exp_decay_cycle:
                if self.args.curiosity_exp_decay_rate <= 1.0:
                    if self.beta2 > self.args.curiosity_exp_decay_stop:
                        self.beta2 = self.beta2 * self.args.curiosity_exp_decay_rate
                    else:
                        self.beta2 = self.args.curiosity_exp_decay_stop
                else:
                    if self.beta2 < self.args.curiosity_exp_decay_stop:
                        self.beta2 = self.beta2 * self.args.curiosity_exp_decay_rate
                    else:
                        self.beta2 = self.args.curiosity_exp_decay_stop
            
                self.decay_stats_cur_t=t_env

        if t_env - self.log_stats_t >= self.args.learner_log_interval:
            self.logger.log_stat("loss", loss.item(), t_env)
            self.logger.log_stat("predictor_loss", predictor_loss.item(), t_env)
            self.logger.log_stat("predictor_os_loss", predictor_os_loss.item(), t_env)
            self.logger.log_stat("p_o_loss", p_o_loss.item(), t_env)
            self.logger.log_stat("q_o_loss", q_o_loss.item(), t_env)
            self.logger.log_stat("grad_norm", grad_norm, t_env)
            self.logger.log_stat("beta1", self.beta1, t_env)
            self.logger.log_stat("beta2", self.beta2, t_env)
            self.logger.log_stat("hit_prob", hit_prob.item(), t_env)
            self.logger.log_stat("grad_norm", grad_norm, t_env)
            mask_elems = mask.sum().item()
            self.logger.log_stat("td_error_abs", (masked_td_error.abs().sum().item() / mask_elems), t_env)
            self.logger.log_stat("q_taken_mean",
                                 (chosen_action_qvals * mask).sum().item() / (mask_elems * self.args.n_agents), t_env)
            self.logger.log_stat("target_mean", (targets * mask).sum().item() / (mask_elems * self.args.n_agents),
                                 t_env)
            self.logger.log_stat("curiosity_intrinsic_reward", (curiosity_intrinsic_reward.sum().item()/mask_elems), t_env)
            self.logger.log_stat("influence_intrinsic_reward", (influence_intrinsic_reward.sum().item()/(mask_elems*self.n_agents)), t_env)
            if self.show_k:
                for i in range(self.n_agents):
                    self.logger.log_stat("k_taken_mean_" + str(i),
                                    (k[:,i] * mask.flatten()).sum().item() / (mask_elems ), t_env)
            self.log_stats_t = t_env

    def train(self, batch: EpisodeBatch, t_env: int, episode_num: int, show_demo=False, save_data=None):
        self.sub_train(batch, t_env, episode_num, self.mac, self.mixer, self.optimiser, self.params,
                       show_demo=show_demo, save_data=save_data)
        if (episode_num - self.last_target_update_episode) / self.args.target_update_interval >= 1.0:
            self._update_targets()
            self.last_target_update_episode = episode_num

    def _update_targets(self):
        self.target_mac.load_state(self.mac)
        if self.mixer is not None:
            self.target_mixer.load_state_dict(self.mixer.state_dict())

        self.target_predict_os.load_state_dict(self.eval_predict_os.state_dict())
        self.target_predict_with_action.load_state_dict(self.eval_predict_with_action.state_dict())
        self.target_predict_without_action.load_state_dict(self.eval_predict_without_action.state_dict())

        self.logger.console_logger.info("Updated target network")

    def cuda(self):
        self.mac.cuda()
        self.target_mac.cuda()
        if self.mixer is not None:
            self.mixer.cuda()
            self.target_mixer.cuda()
        
        self.eval_predict_os.cuda()
        self.target_predict_os.cuda()
        self.eval_predict_with_action.cuda()
        self.target_predict_with_action.cuda()
        self.eval_predict_without_action.cuda()
        self.target_predict_without_action.cuda()

    def save_models(self, path):
        self.mac.save_models(path)
        if self.mixer is not None:
            th.save(self.mixer.state_dict(), "{}/mixer.th".format(path))
        th.save(self.optimiser.state_dict(), "{}/opt.th".format(path))
        th.save(self.predictor_optimiser.state_dict(), "{}/predictor_opt.th".format(path))

    def load_models(self, path):
        self.mac.load_models(path)
        # Not quite right but I don't want to save target networks
        self.target_mac.load_models(path)
        if self.mixer is not None:
            self.mixer.load_state_dict(th.load("{}/mixer.th".format(path), map_location=lambda storage, loc: storage))
            self.target_mixer.load_state_dict(th.load("{}/mixer.th".format(path),
                                                      map_location=lambda storage, loc: storage))
        self.optimiser.load_state_dict(th.load("{}/opt.th".format(path), map_location=lambda storage, loc: storage))
        self.predictor_optimiser.load_state_dict(th.load("{}/predictor_opt.th".format(path), map_location=lambda storage, loc: storage))