//
 //  code_jam
 //
 //  Created by Paul O'Neil on 5/20/11.
 //  Copyright 2011 Paul O'Neil. All rights reserved.
 //
 
 #include <iostream>
 #include <fstream>
 #include <vector>
 #include <list>
 #include <algorithm>
 #include <cmath>
 using namespace std;
 
 struct vendor_strip_t {  
   virtual int width() const throw() = 0;
   
   virtual double time_spent() const throw() = 0;
     
   virtual double left_edge() const throw() = 0;
 
   virtual double right_edge() const throw() = 0;
   
   virtual double middle() const throw() {
     return (left_edge() + right_edge()) / 2.0;
   }
 
   virtual void add_displacement(double d) = 0;
 };
 
 struct starting_point_t : public vendor_strip_t {
   static int separation_distance;
   double position;
   double displacement;
   int vendor_count;
   
   starting_point_t( double p, int c ) : position(p), displacement(0), vendor_count(c) { }
   
   bool operator<(const starting_point_t& other) const {
     if( position == other.position ) {
       return vendor_count < other.vendor_count;
     }
     return position < other.position;
   }
   
   virtual double left_edge() const throw() {
     return position - width() / 2.0 + displacement;
   }
   
   virtual double right_edge() const throw() {
     return position + width() / 2.0 + displacement;
   }
   
   int width() const throw() {
     return vendor_count * separation_distance;
   }
   
   double time_spent() const throw() {
     return ((double)(vendor_count - 1) * separation_distance) / 2.0 + abs(displacement);
   }
   
   void add_displacement(double d) {
     displacement += d;
   }
 };
 
 double sign(double val) {
   if( sign > 0 ) {
     return 1;
   }
   else if( sign < 0 ) {
     return -1;
   }
   else {
     return 0;
   }
 }
 
 struct composite_strip_t : public vendor_strip_t {
   vendor_strip_t * sub_parts[2];
   
   composite_strip_t(vendor_strip_t * f, vendor_strip_t * s) {
     if( f->middle() > s->middle() ) {
       vendor_strip_t * tmp = f;
       f = s;
       s = tmp;
     }
     sub_parts[0] = f;
     sub_parts[1] = s;
     
     vendor_strip_t * bigger = ( f->time_spent() > s->time_spent() ? f : s );
     vendor_strip_t * smaller = ( f->time_spent() > s->time_spent() ? s : f );
     
     
     double movement = (f->right_edge() - s->left_edge());
     
     // move while the other one is setting up
     if( smaller == s ) {
       double disp = min(abs(movement), abs(bigger->time_spent() - smaller->time_spent())) * sign(movement);
       smaller->add_displacement(disp);
       movement -= disp;
     }
     else if( smaller == f ) {
       double disp = min(abs(movement), abs(bigger->time_spent() - smaller->time_spent())) * sign(movement);
       smaller->add_displacement(-disp);
       movement -= disp;      
     }
     movement /= 2;
     
     f->add_displacement(-movement);
     s->add_displacement(movement);
   }
   
   ~composite_strip_t() {
     delete sub_parts[0];
     delete sub_parts[1];
   }
   
   int width() const throw() {
     return sub_parts[0]->width() + sub_parts[1]->width();
   }
   
   double time_spent() const throw() {
     return max(sub_parts[0]->time_spent(), sub_parts[1]->time_spent());
   }
   
   double left_edge() const throw() {
     return sub_parts[0]->left_edge();
   }
   
   double right_edge() const throw() {
     return sub_parts[1]->right_edge();
   }
   
   void add_displacement(double d) throw() {
     sub_parts[0]->add_displacement(d);
     sub_parts[1]->add_displacement(d);
   }
 };
 
 int starting_point_t::separation_distance;
 
 
 bool less_time(const vendor_strip_t* a, const vendor_strip_t* b) {
   return a->time_spent() < b->time_spent();
 }
 
 bool middle_left(const vendor_strip_t* a, const vendor_strip_t * b) {
   return a->middle() < b->middle();
 }
 
 typedef vector<vendor_strip_t*> vendor_list_t;
 typedef vendor_list_t::iterator vendor_iter_t;
 
 template<typename T>
 void remove_idx( vector<T> & vec, int idx ) {
   for( int i = idx; i < vec.size() - 1; i++ ) {
     vec[i] = vec[i+1];
   }
   vec.pop_back();
 }
 
 double do_trial(istream & input) throw() {
   int start_point_count;
   input >> start_point_count >> starting_point_t::separation_distance;
   
   vendor_list_t starting_points;
   
   for( int i = 0; i < start_point_count; i++ ) {
     int pos, count;
     input >> pos >> count;
     starting_points.push_back(new starting_point_t(pos, count));
   }
   
   //starting_points.sort(middle_left);
   sort(starting_points.begin(), starting_points.end(), middle_left);
   
   for( int i = 0; i < starting_points.size() - 1; ) {
     // this node has been successfully merged to the left so far,
     // merge it with the one on the right if necessary, then check left again.
     vendor_strip_t * cur_strip = starting_points[i];
     vendor_strip_t * next_strip = starting_points[i+1];
     
     if( cur_strip->right_edge() > next_strip->left_edge() ) {
       composite_strip_t * composite = new composite_strip_t(cur_strip, next_strip);
       cur_strip = starting_points[i] = composite;
       remove_idx(starting_points, i+1);
       next_strip = starting_points[i+1];
       
       while( i > 0 ) {
         vendor_strip_t * prev_strip = starting_points[i-1];
         // merge to the left if we need to
         if( cur_strip->left_edge() > prev_strip->right_edge() ) {
           break;
         }
         composite = new composite_strip_t(prev_strip, cur_strip);
         cur_strip = starting_points[i-1] = composite;
         remove_idx(starting_points, i);
         i--;
       }
     }
     i++;
   }
   
   double time_spent = (*max_element(starting_points.begin(), starting_points.end(), less_time))->time_spent();
   
   for( int i = 0; i < starting_points.size(); i++ ) {
     delete starting_points[i];
   }
   
   return time_spent;
 }
 
 int main (int argc, const char * argv[])
 {
   int trialCount;
   istream * in;
   if( argc == 1 ) {
     in = &cin;
   }
   else {
     in = new ifstream(argv[1]);
   }
   istream & input = *in;
   input >> trialCount;
   
   for( int i = 1; i <= trialCount; i ++ ) {
     double result = do_trial(input);
     cout << "Case #" << i << ": " << result << endl;
   }
   
   if( argc > 1 ) {
     delete in;
   }
   return 0;
 }
