from copy import deepcopy
import os
import random
import warnings
import numpy as np
import torch.nn.functional as F
import torch
from backbone.ResNet18 import resnet34
from typing import Any
import torchvision.transforms as transforms
from torch.utils.data import Dataset, ConcatDataset, Subset, DataLoader
from torch.utils.data.dataloader import default_collate
from datasets.transforms.denormalization import DeNormalize
from PIL import Image, ImageFile
from datasets.utils.continual_dataset import (ContinualDataset,
                                              store_masked_loaders)
from utils.autoaugment import CIFAR10Policy
from utils.conf import base_path_dataset as base_path

class WEBVISION(Dataset):
    num_classes=10

    def __init__(self, root, transform, train=True):
        ImageFile.LOAD_TRUNCATED_IMAGES = True

        self.subsets = dict()
        self.train = train
        self.root = root
        self.transform = transform
        self.not_aug_transform = transforms.Compose([transforms.ToTensor()])
        self.load_transform = transforms.Resize((84,84))
        
        self.idx_to_realname = list()
        with open(os.path.join(self.root, 'info', 'synsets.txt'), 'r') as f:
            for i, cls in enumerate(f.readlines()):
                self.idx_to_realname.append(cls)

        self.data = list()
        self.targets = list()

        if train:
            infos_pth = os.path.join(self.root, 'info', 'train_filelist_google.txt')
        else:
            infos_pth = os.path.join(self.root, 'info', 'val_filelist.txt') 

        LABEL_LIST = [412, 480, 506, 395, 421, 121, 498, 762, 48, 89] # 10 classes, as in PuriDivER

        self.data : Any = []
        self.data_names = list()
        self.targets = list()

        with open(infos_pth, 'r') as f:
            for info in f.readlines():
                name, label = info.split(' ')
                if not train:
                    name = os.path.join('val_images_256', name)

                label = int(label)
                if label not in LABEL_LIST:
                    continue
                self.data_names.append(name)
                self.targets.append(LABEL_LIST.index(label))

        for file_name in self.data_names:
            self.data.append(np.array(Image.open(os.path.join(self.root, file_name))))
        
        self.data = np.vstack(self.data).reshape(-1, 3, 84, 84)
        self.data = self.data.transpose((0, 2, 3, 1))
        
        self.targets = torch.LongTensor(self.targets)
        self.org_targets = self.targets

        # Create subset for each class
        for y in range(self.num_classes):
            self.subsets[y] = Subset(self, torch.nonzero(self.targets == y).squeeze(1).tolist())
        self._num_samples = len(self.targets)
        self.initial_indexes = np.arange(len(self.data))
    
    def __len__(self) -> int:
        return len(self.data)
    
    def __getitem__(self, index):
        img, target, gt_target = self.data[index], self.targets[index], self.gt_targets[index]
        # to return a PIL Image
        img = Image.fromarray(img, mode='RGB')
        original_img = img.copy()

        not_aug_img = self.not_aug_transform(original_img)

        if self.transform is not None:
            img = self.transform(img)

        return img, target, not_aug_img, gt_target, self.initial_indexes[index]

class TWEBVISION(WEBVISION):
    def __init__(self, root, transform, train):
        self.root = root
        super(TWEBVISION,self).__init__(root, transform, train)
    
    def __len__(self) -> int:
        return len(self.data)
    
    def __getitem__(self, index):
        img, target = self.data[index], self.targets[index]

        # doing this so that it is consistent with all other datasets
        # to return a PIL Image
        img = Image.fromarray(img, mode='RGB')

        if self.transform is not None:
            img = self.transform(img)
            
        return img, target
    
class SequentialWEBVISION(ContinualDataset):
    
    NAME = 'seq-webvision'
    SETTING = 'class-il'
    N_CLASSES_PER_TASK = 2
    N_TASKS = 5
    MEAN, STD  = (0.5194, 0.4913, 0.4581), (0.3006, 0.2971, 0.3091)
    TRANSFORM = transforms.Compose(
            [transforms.RandomCrop(84, padding=4),
            transforms.RandomHorizontalFlip(),
            transforms.ToTensor(),
            transforms.Normalize(MEAN, STD),
            ])
    
    STRONG_TRANSFORM = transforms.Compose(
            [transforms.RandomCrop(84, padding=4),
            transforms.RandomHorizontalFlip(),
            CIFAR10Policy(),
            transforms.ToTensor(),
            transforms.Normalize(MEAN, STD),
            ])
    
    def get_data_loaders(self):
        transform = self.TRANSFORM

        test_transform = transforms.Compose(
            [transforms.ToTensor(), self.get_normalization_transform()])

        train_dataset = WEBVISION(base_path() + 'miniWEBVISION', transform=transform,
                                train=True)

        test_dataset = TWEBVISION(base_path() + 'miniWEBVISION', transform=test_transform, 
                                train=False)

        train, test = store_masked_loaders(train_dataset, test_dataset, self)

        return train, test

    @staticmethod
    def get_transform():
        transform = transforms.Compose(
            [transforms.ToPILImage(), SequentialWEBVISION.TRANSFORM])
        return transform

    @staticmethod
    def get_backbone():
        return resnet34(SequentialWEBVISION.N_CLASSES_PER_TASK
                        * SequentialWEBVISION.N_TASKS)

    @staticmethod
    def get_loss():
        return F.cross_entropy

    @staticmethod
    def get_normalization_transform():
        transform = transforms.Normalize(SequentialWEBVISION.MEAN, SequentialWEBVISION.STD)
        return transform

    @staticmethod
    def get_denormalization_transform():
        transform = DeNormalize(SequentialWEBVISION.MEAN, SequentialWEBVISION.STD)
        return transform

    @staticmethod
    def get_scheduler(model, args):
        return None

    @staticmethod
    def get_epochs():
        return 30

    @staticmethod
    def get_warmup_epochs():
        return 0
        
    @staticmethod
    def get_batch_size():
        return 32

    @staticmethod
    def get_minibatch_size():
        return SequentialWEBVISION.get_batch_size()
            
            