import sys
import torch
from torch.optim import SGD, lr_scheduler

from models.utils.continual_model import ContinualModel
from utils.args import add_management_args, add_experiment_args, add_rehearsal_args, add_noise_args, ArgumentParser
from utils.augmentations import cutmix_data
from utils.buffer import Buffer
from utils.status import progress_bar


def get_parser() -> ArgumentParser:
    parser = ArgumentParser()
    add_management_args(parser)
    add_rehearsal_args(parser)
    add_noise_args(parser)
    parser.add_argument('--maxlr', type=float, default=5e-2,
                        help='Penalty weight.')
    parser.add_argument('--minlr', type=float, default=5e-4,
                        help='Penalty weight.')
    parser.add_argument('--cutmix_alpha', type=float, default=None,
                        help='Penalty weight.')
    add_experiment_args(parser)
    return parser


def fit_buffer(self, epochs):
    optimizer = SGD(self.net.parameters(), lr=self.args.maxlr, momentum=self.args.optim_mom, weight_decay=self.args.optim_wd, nesterov=self.args.optim_nesterov)
    scheduler = lr_scheduler.CosineAnnealingWarmRestarts(optimizer, T_0=1, T_mult=2, eta_min=self.args.minlr)

    for epoch in range(epochs):
        print('Buffer fitting epoch {}/{}'.format(epoch + 1, epochs), file=sys.stderr)
        if epoch <= 0:  # Warm start of 1 epoch
            for param_group in optimizer.param_groups:
                param_group['lr'] = self.args.maxlr * 0.1
        elif epoch == 1:  # Then set to maxlr
            for param_group in optimizer.param_groups:
                param_group['lr'] = self.args.maxlr
        else:
            scheduler.step()

        all_inputs, all_labels, _, _ = self.buffer.get_data(
            len(self.buffer.examples), transform=self.transform)

        while len(all_inputs):
            optimizer.zero_grad()
            buf_inputs, buf_labels = all_inputs[:self.args.batch_size], all_labels[:self.args.batch_size]
            all_inputs, all_labels = all_inputs[self.args.batch_size:], all_labels[self.args.batch_size:]

            if self.args.cutmix_alpha is not None:
                inputs, labels_a, labels_b, lam = cutmix_data(x=buf_inputs.cpu(), y=buf_labels.cpu(), alpha=self.args.cutmix_alpha)
                buf_inputs = inputs.to(self.device)
                buf_labels_a = labels_a.to(self.device)
                buf_labels_b = labels_b.to(self.device)
                buf_outputs = self.net(buf_inputs)
                loss = lam * self.loss(buf_outputs, buf_labels_a) + (1 - lam) * self.loss(buf_outputs, buf_labels_b)
            else:
                buf_outputs = self.net(buf_inputs)
                loss = self.loss(buf_outputs, buf_labels)

            loss.backward()
            optimizer.step()
        progress_bar(epoch, epochs, 1, 'G', loss.item())


class GDumb(ContinualModel):
    NAME = 'gdumb'
    COMPATIBILITY = ['class-il', 'task-il']

    def __init__(self, backbone, loss, args, transform):
        super(GDumb, self).__init__(backbone, loss, args, transform)
        self.buffer = Buffer(self.args.buffer_size, self.device)
        self.task = 0

    def observe(self, inputs, labels, not_aug_inputs, true_labels, indexes, epoch):
        self.buffer.add_data(examples=not_aug_inputs,
                             labels=labels,
                             true_labels=true_labels,
                             sample_indexes=indexes)
        return 0

    def begin_task(self, dataset):
        if self.args.lnl_mode in ["dividemix","coteaching"]:
            self.comodel.train()

        self.reset_opt()

    def end_task(self, dataset):
        # new model
        self.task += 1
        if not (self.task == dataset.N_TASKS):
            return
        self.net = dataset.get_backbone().to(self.device)

        self.args.buffer_fitting_lr = self.args.lr

        if self.args.lnl_mode is not None:
            self.comodel = dataset.get_backbone().to(self.device)
            self.comodel.train()
            self.reset_opt()

            # clear cache torch
            torch.cuda.empty_cache()
            
            self.fit_buffer()
        else:
            fit_buffer(self, self.args.buffer_fitting_epochs)
