import numpy as np


def non_dominated_mask(points, decimals=None, return_dominators=False):
    """Compute a mask (0s and 1s) indicating the non-dominated set from a list of points.
    If return_dominators is True, also return a list of sets of dominator indices for each point.
    """
    if points.size == 0:
        return (points, []) if return_dominators else points

    points = np.asarray(points)

    if decimals is not None:
        points = np.round(points, decimals=decimals)

    A = points[:, None, :]
    B = points[None, :, :]

    eq = np.isclose(A, B)
    le_safe = (A < B) | eq
    lt_safe = (A < B) & (~eq)

    # Create pairwise dominance matrix: domination_matrix[i, j] = True if i is dominated by j
    domination_matrix = np.all(le_safe, axis=2) & np.any(lt_safe, axis=2)

    dominated = np.any(domination_matrix, axis=1)
    non_dominated = ~dominated

    if return_dominators:
        dominators = np.array([set(np.flatnonzero(domination_matrix[i])) for i in range(len(points))])
        return non_dominated, dominators
    else:
        return non_dominated

def duplicated_mask(points, decimals=None, return_dominators=False, group_duplicates=False, axis=0):
    """Compute a mask (0s and 1s) indicating the duplicated items from a list of points.
    If return_dominators is True, returns for each point the list of other indices that match it.
    """

    def fuzzy_unique(points, return_index=False, return_inverse=False, return_counts=False, axis=None):

        if axis != 0:
            raise NotImplementedError

        output = []

        # Step 1: sort points lexicographically for stability
        order = np.lexsort(points.T[::-1])
        sorted_points = points[order]

        # Step 2: compare rows pairwise with allclose
        diff = np.any(~np.isclose(sorted_points[1:], sorted_points[:-1]), axis=1)
        keep = np.ones(len(points), dtype=bool)
        keep[1:] = diff  # start of new group

        # Step 3: get unique points
        unique_points = sorted_points[keep]
        output.append(unique_points)

        if return_index:
            # Step 4: get return_index
            first_indices = order[keep]
            output.append(first_indices)

        if return_inverse:
            # Step 5: get return_inverse
            group_ids = np.cumsum(keep) - 1
            inverse = np.empty_like(order)
            inverse[order] = group_ids
            output.append(inverse)

        if return_counts:
            # Step 6: get return_counts
            counts = np.diff(np.append(np.nonzero(keep)[0], len(points)))
            output.append(counts)

        return output

    if points.size == 0:
        if return_dominators:
            return np.array([], dtype=bool), []
        return np.array([], dtype=bool)

    if decimals is None:
        #max tolerance
        decimals = 9

    points = np.round(points, decimals=decimals)

    if group_duplicates:
        _, group_ids, counts = fuzzy_unique(points, return_inverse=True, return_counts=True, axis=axis)
        mask = counts[group_ids] > 1

        if return_dominators:
            raise NotImplementedError
        else:
            return mask, group_ids
    else:
        _, idx, group_ids = fuzzy_unique(points, return_index=True, return_inverse=True, axis=axis)
        mask = np.ones(len(points), dtype=bool)
        mask[idx] = False
        if return_dominators:
            duplicates_of_each = np.full(len(points), set())
            duplicates_of_each[mask] = [set(np.where(group_ids == group_ids[i])[0]) - {i} for i in np.where(mask)[0]]

            return mask, duplicates_of_each
        else:
            return mask
