import argparse
import sys


def extend_argparser(base, additional):
    """Merges arguments from `base` into `additional`."""
    for action in base._actions:
        if action.dest not in {a.dest for a in additional._actions}:
            additional._add_action(action)
    return additional


def override_argument(parser, name, **new_kwargs):
    # Remove existing argument
    for action in list(parser._actions):
        if name in action.option_strings or name == action.dest:
            parser._handle_conflict_resolve(None, [(name, action)])

    # Add new argument with updated properties
    parser.add_argument(name, **new_kwargs)

def merge_args(args, loaded_config):
    passed_keys = {arg.lstrip('-').replace('-', '_') for arg in sys.argv[1:] if arg.startswith('--')}
    explicit_args = {k: v for k, v in vars(args).items() if k in passed_keys}

    merged_dict = {**loaded_config, **explicit_args}
    args = argparse.Namespace(**merged_dict)

    return args

def get_atari_argparser():
    parser = argparse.ArgumentParser()
    parser.add_argument("--task", type=str, default="PongNoFrameskip-v4")
    parser.add_argument("--scale-obs", type=int, default=0)
    parser.add_argument("--frames-stack", type=int, default=4)
    return parser

def get_mujoco_argparser():
    parser = argparse.ArgumentParser()
    parser.add_argument("--task", type=str, default="Ant-v4")
    parser.add_argument("--obs_norm", action="store_true", default=False)
    return parser

def get_mo_gymnasium_argparser():
    parser = argparse.ArgumentParser()
    parser.add_argument("--task", type=str, default="deep-sea-treasure-v0")
    return parser

def get_classic_control_argparser():
    parser = argparse.ArgumentParser()
    parser.add_argument("--task", type=str, default="CartPole-v1")
    return parser

def parse_task_params(s):
    def infer_type(val):
        if val.lower() == 'true': return True
        if val.lower() == 'false': return False
        try:
            return int(val)
        except ValueError:
            try:
                return float(val)
            except ValueError:
                return val

    args_str = s
    kwargs = {}
    if args_str:
        for kv in args_str.split(','):
            key, val = kv.split('=')
            kwargs[key] = infer_type(val)

    return kwargs

def parse_task_wrapper(s):
    def infer_type(val):
        if val.lower() == 'true': return True
        if val.lower() == 'false': return False
        try:
            return int(val)
        except ValueError:
            try:
                return float(val)
            except ValueError:
                return val

    name, _, args_str = s.partition(':')
    kwargs = {}
    if args_str:
        for kv in args_str.split(','):
            key, val = kv.split('=')
            kwargs[key] = infer_type(val)

    return {'class': name, 'kwargs': kwargs}
