import threading


class Node:

    def __init__(self, data=None):
        if data is None:
            data = {}
        self.data = data
        self.previous = None
        self.next = None

    def __iter__(self):
        current = self
        while current:
            yield current
            current = current.next

    def __len__(self):
        return len(self.data)

    def __repr__(self):
        return self.data

    def __copy__(self):
        new_node = type(self)(self.data.copy())
        new_node.previous = self.previous
        new_node.next = self.next
        return new_node

class _CounterNode(Node):

    _id_counter = 0
    _id_lock = threading.Lock()

    def __init__(self, data):
        with _CounterNode._id_lock:
            self.id = _CounterNode._id_counter
            _CounterNode._id_counter += 1
        super().__init__(data)

class DoubleLinkedList:

    def __init__(self):
        self.head = None
        self.tail = None
        self.id_map = {}

    def __getstate__(self):
        return self.serialize()

    def __setstate__(self, state):
        self.deserialize(state)

    def __iter__(self):
        current = self.head
        while current:
            yield current.id, current.data
            current = current.next

    def append(self, data):
        new_node = _CounterNode(data)
        if not self.head:
            self.head = new_node
            self.tail = new_node
        else:
            new_node.previous = self.tail
            self.tail.next = new_node
            self.tail = new_node

        self.id_map[new_node.id] = new_node
        return new_node

    def pop(self, node=None, id_=None):

        if node is None and id_ is None:
            raise ValueError("Node or id cannot be None")
        if node is not None and id_ is not None:
            raise ValueError("Only node or id can be used, not both")

        if id_ is not None:
            node = self.id_map[id_]

        if not isinstance(node, _CounterNode):
            raise TypeError(f"Node is of not of type Node. Actual type: {type(node)}")

        if node == self.head:
            self.head = node.next
        else:
            node.previous.next = node.next

        if node == self.tail:
            self.tail = node.previous
        else:
            node.next.previous = node.previous

    def get_by_id(self, id_):
        return self.id_map[id_]

    def display(self):
        current = self.head
        while current:
            print(current.data, end=" -> ")
            current = current.next
        print("None")

    def serialize(self):
        result = []
        current = self.head
        visited = set()

        while current and id(current) not in visited:
            visited.add(id(current))
            result.append((current.id, current.data))
            current = current.next

        if current:
            raise RuntimeError('<cycle detected>')

        return result

    def deserialize(self, state):
        self.__init__()
        for node_id, data in state:
            node = self.append(data)
            node.id = node_id
            _CounterNode._id_counter = max(_CounterNode._id_counter, node_id + 1)
