import logging
import coloredlogs
import traceback
import time
import os
import aicrowd_helper
from train.model_registry import ModelRegistry
from train.iface import ExternalInterface
from train.enums import ExecMode
from train.iface_debug import ExternalInterfaceDebug
from train.package_manager import PackageManager
from train.statement.parser import parse_training_statements, parse_eval_statements
from train.monitor.timeouts import *
from train.stats import Stats


LEVELS = {'debug': logging.DEBUG,
          'info': logging.INFO,
          'warning': logging.WARNING,
          'error': logging.ERROR,
          'critical': logging.CRITICAL}


def get_ext_iface(config) -> ExternalInterface:
    if config.debug.enabled:
        return ExternalInterfaceDebug(config)
    return ExternalInterface(config)


def elapsed_time(start_time):
    hours, rem = divmod(time.time() - start_time, 3600)
    minutes, seconds = divmod(rem, 60)
    return "{:0>2}:{:0>2}:{:05.2f}".format(int(hours), int(minutes), seconds)


class MetaController:
    def __init__(self, config):
        self.config = config
        level = LEVELS[self.config.log_level]
        coloredlogs.install(level)
        logging.basicConfig(filename=config.debug.log_file, level=level)
        self.task_mapping = {v: k for k, v in self.config.subtask.consensus_code.items()}

    def run(self):
        """
        Main entrance point for the Minecraft challenge.
        :return:
        """
        logging.info('Controller: Initializing training...')
        # initialize the package manager
        PackageManager.get_instance().initialize(self.config.packages)

        # create interface wrapper for all modules
        ext_iface = get_ext_iface(self.config)

        # create all required paths
        logging.info('Controller: Creating paths if not existing')
        ext_iface.create_paths()

        # update paths
        timestamp = time.strftime("%Y%m%d%H%M%S")
        self.config.statistics_path = os.path.join(self.config.statistics_path, timestamp)

        run_cnt = 0
        max_runs = self.config.debug.repeat_n_runs
        print(self.config.debug)
        while run_cnt < max_runs:
            # load consensus checkpoint or create new consensus
            consensus = ext_iface.get_consensus() if run_cnt == 0 else ext_iface.load_consensus()

            # initialize global statistics
            if run_cnt == 0:
                Stats.get_instance(config=self.config).reset("".join([self.task_mapping[t.target] for t in consensus]))

            if self.config.mode == ExecMode.Eval:
                logging.info('Controller: Executing in evaluation mode!')
                # parses task lists into executable evaluation statements for the inference
                statements = parse_eval_statements(self.config, ext_iface, consensus)
            else:
                logging.info('Controller: Executing in training mode!')
                # parses task lists into executable training statements
                statements = parse_training_statements(self.config, ext_iface, consensus)

            # reset model registry
            logging.info('Controller: Resetting model registry!')
            registry = ModelRegistry.get_instance(self.config)
            registry.reset()

            logging.info('Controller: Starting run {}/{}'.format(run_cnt + 1, self.config.debug.repeat_n_runs))
            # execute the statements
            logging.info('Controller: Executing statements...')
            idx = 0
            repeat = 0
            start_time = time.time()
            failure_occurred = False

            while idx < len(statements):
                if not statements[idx].task.transition_data_ready:
                    logging.info('Controller: Waiting {} for task {} ready to execute statement.'
                                 .format(elapsed_time(start_time), statements[idx].task.id))
                    time.sleep(TIMEOUT_1_SECOND)
                    continue

                stmt = statements[idx]
                try:
                    stmt.exec()
                    # progress only if statement was successful
                    if stmt.success():
                        start_time = time.time()
                        idx += 1
                        repeat = 0
                    if self.config.submit_to_aicrowd: aicrowd_helper.register_progress(idx / len(statements))
                except Exception as e:
                    tr = traceback.format_exc()
                    logging.error(tr)
                    logging.error(e)
                    err_msg = "Controller: Exception occurred for statement {}: ".format(stmt.task.id)
                    if repeat > self.config.monitor.max_statement_executions:
                        failure_occurred = True
                        err_msg += ' Skipping Statement!'
                        logging.error(err_msg)
                        if self.config.submit_to_aicrowd: aicrowd_helper.execution_error(err_msg)
                        idx += 1
                        repeat = 0
                    else:
                        err_msg += ' Repeating Statement!'
                        logging.warning(err_msg)
                        if self.config.submit_to_aicrowd: aicrowd_helper.execution_error(err_msg)
                        repeat += 1
            if self.config.monitor.interrupt_on_failure and failure_occurred:
                logging.error(
                    'Controller: Statement failure occurred! Interrupt on failure is set. Exiting meta controller.')
                break

            logging.info('Controller: Statement execution complete!')
            Stats.get_instance().save()
            run_cnt += 1

        logging.info('Controller has finished all runs!')
