import os
import numpy as np
import traceback
from collections import deque
from pathlib import Path
import logging
import gym
import json
import train
from train.behavioral_cloning.datasets.experience import Experience
from train.experience_replay.experience_success_mapping import ExperienceSuccessMapping


class OperationCompletedInterrupt(Exception):
    pass


class ConsensusStage(object):
    def __init__(self, stage, inventory_minimum):
        self.stage = stage
        self.inventory_minimum = inventory_minimum


class PartialExperienceSamplingWrapper(gym.Wrapper):
    """
    The PartialExperienceSamplingWrapper is used to reset the environment to a previously recorded state. It adds
    the capability of only replaying a trajectory partially up to a manually specified point.
    """

    def __init__(self, env, experience_folder, replay_until,
                 success_mapping_file_path='tmp/experiences/success_mapping.p',
                 verbose=False, generate_experience_video=False, debug_video=False):
        super().__init__(env)
        assert experience_folder is not None
        self.env = env
        self.log_interval = 500
        self.generate_experience_video = generate_experience_video
        self.debug_video = debug_video

        # if this is not called from the content root dir, adapt the success_mapping_file_path
        if not os.path.exists(os.path.split(success_mapping_file_path)[0]):
            path_to_train_module = Path(train.__path__[0])
            path_to_content_root = path_to_train_module / '..' / 'tmp'
            success_mapping_file_path = str(path_to_content_root / 'experiences' / 'success_mapping.p')

        # if the path still doesn't exist, something likely went wrong
        if not os.path.exists(os.path.split(success_mapping_file_path)[0]):
            raise FileNotFoundError('Are you in the correct directory? ')

        # to the same for the provided experience_folder
        # if this is not called from the content root dir, adapt the success_mapping_file_path
        if not os.path.exists(experience_folder):
            path_to_train_module = Path(train.__path__[0])
            experience_folder = str(path_to_train_module / '..' / experience_folder)

        # if the path still doesn't exist, something likely went wrong
        if not os.path.exists(experience_folder):
            raise FileNotFoundError('Are you in the correct directory? ')

        self.experience_folder = os.path.normpath(experience_folder)

        self.success_mapping_file_path = success_mapping_file_path
        self.replay_until = replay_until
        self.verbose = verbose

        # The following dict is a mapping from experience replay filename to the success probability of it running
        # up to the consensus provided. At each reached consensus stage, the probability is updated that one
        # particular experience file was able to be replayed up to this consensus stage.
        # See if there is a pickled version of this in the experience experiences_folder. If there is not, create a new dict.
        self.experience_filenames = os.listdir(experience_folder)

        self.success_mapping = ExperienceSuccessMapping(experiences_folder=self.experience_folder,
                                                        success_mapping_file_path=self.success_mapping_file_path)

    @staticmethod
    def is_inventory_sufficient_for_current_consensus_stage(live_inventory, current_goal_stage):
        for key in current_goal_stage.inventory_minimum:
            if live_inventory[key] >= current_goal_stage.inventory_minimum[key]:
                continue
            else:
                return False
        return True

    @staticmethod
    def reset_consensus_stages():
        consensus_stages = deque()
        with open('train/experience_replay/minimum_stage_resources.json') as f:
            minimum_stage_resources = json.load(f)
        for stage, resource in minimum_stage_resources.items():
            consensus_stages.appendleft(ConsensusStage(stage=stage, inventory_minimum=resource))
        return consensus_stages

    def reset_record_side_by_side(self, env_seed, consensus, experience_file, checkpoint):
        # if all seems well with the recording, set seed and reset environment
        if env_seed:
            self.env.seed(env_seed)
        super().reset()

        consensus_worked_up_to = None
        # we try to playback the env on the environment. if it doesn't work, we update its statistics and move on
        # we cannot be certain, that the sampled experience is a good experience
        # we have to catch exceptions in the reset method because we throw an InventoryMatch Exception
        # if the inventories of live and recorded trajectories don't match
        logging.info('Replay: Trying experience replay on env.')
        try:
            replay_rewards = []
            reached_replay_goal = False
            # init live_inventory variable with recorded inventory (we just use recorded inventory because
            # it has the correct layout for live_inventory
            consensus_stages = PartialExperienceSamplingWrapper.reset_consensus_stages()
            # setup counter and retry helper
            g_counter = -1
            counter = -1
            # get fist stage
            current_goal_stage = consensus_stages.pop()
            # get max timings for the different stages
            with open('train/experience_replay/timings.json') as f:
                max_counter = json.load(f)
            # used for video generation
            video_frames = []
            video_rewards = []
            video_stages = []
            video_povs = []
            video_trans_rewards = []
            cum_reward = 0
            cum_trans_reward = 0
            for i, action in enumerate(self.experience_recording.replay_action_generator(checkpoint=checkpoint)):
                counter += 1
                g_counter += 1
                if g_counter % self.log_interval == 0: logging.info(f"Replay: Reached counter step: {g_counter} current stage: {current_goal_stage.stage}")

                obs, reward, done, info = self.env.step(action)
                replay_rewards.append(reward)
                live_inventory = obs['inventory']

                # for video generation
                video_frames.append(obs['pov'])
                cum_reward += reward
                trans_info = self.experience_recording.transitions_info[i]
                cum_trans_reward += trans_info['reward']
                video_trans_rewards.append(cum_trans_reward)
                trans_pov = trans_info['info']['meta_controller']['state']['pov'] \
                    if 'meta_controller' in trans_info['info'] else None
                video_povs.append(trans_pov)
                video_rewards.append(cum_reward)
                video_stages.append(current_goal_stage.stage)

                if current_goal_stage.stage == 'SSSSSSSPPPPPLVNLAAAAYLQQQFKKKED':
                    reached_consensus_stage = reward >= 1000
                else:
                    reached_consensus_stage = PartialExperienceSamplingWrapper \
                        .is_inventory_sufficient_for_current_consensus_stage(live_inventory, current_goal_stage)

                if reached_consensus_stage:
                    logging.debug("Reached consensus stage {}".format(current_goal_stage.stage))
                    consensus_worked_up_to = current_goal_stage.stage
                    # check if the current_goal_stage was the overall goal stage
                    if consensus == current_goal_stage.stage:
                        reached_replay_goal = True
                    else:
                        # pop new consensus stage
                        current_goal_stage = consensus_stages.pop()
                        counter = -1

                if reached_replay_goal:
                    logging.info('Replay: Successfully replayed the trajectory up to {} in {} steps!'.format(consensus, g_counter))
                    logging.debug('Final inventory: {}'.format(live_inventory))
                    # update success_mapping
                    self.success_mapping.update(experience_replay_filename=experience_file, result=1)
                    # write successful video
                    Experience.video('tmp/rec/video/success', experience_file, video_frames, video_rewards,
                                     video_stages, video_povs, video_trans_rewards, video_only=False)
                    return obs

                if counter >= max_counter[current_goal_stage.stage]:
                    logging.warning('Replay: Max_counter limit exceeded in replay while trying stage {}!'.format(current_goal_stage.stage))
                    self.success_mapping.update(experience_replay_filename=experience_file, result=0)
                    # break to get into new iteration of while loop
                    break
                if done:
                    logging.warning('Replay: Environment has finished before completing stage {}!'.format(current_goal_stage.stage))
                    self.success_mapping.update(experience_replay_filename=experience_file, result=0)
                    # break to get into new iteration of while loop
                    break

            # write failed video
            Experience.video('tmp/rec/video/failed', experience_file, video_frames, video_rewards,
                             video_stages, video_povs, video_trans_rewards, video_only=False)
            logging.warning("Replay has not worked. Inventory: {}. Current stage: {} Trying new experience replay.".format(
                live_inventory, current_goal_stage.stage
            ))

        except Exception as e:
            # If the replay did not work, update the replay statistics and reset env
            logging.error('Replay: {} Something went wrong when replaying this trajectory.'.format(e))
            logging.error(
                'Replaying the trajectory for experience file {} has not worked up to {}, only up to {}. '.format(
                    experience_file, consensus, consensus_worked_up_to))
            self.success_mapping.update(experience_replay_filename=experience_file, result=0)
            raise Exception(f"Exception {e}. This code should not be reached. Check the implementation!")

    def reset_record_replay_video(self, env_seed, consensus, experience_file, video_only=False):
        # if all seems well with the recording, set seed and reset environment
        if env_seed:
            self.env.seed(env_seed)
        #obs = super().reset()

        consensus_worked_up_to = None
        # we try to playback the env on the environment. if it doesn't work, we update its statistics and move on
        # we cannot be certain, that the sampled experience is a good experience
        # we have to catch exceptions in the reset method because we throw an InventoryMatch Exception
        # if the inventories of live and recorded trajectories don't match
        logging.info('Replay: Trying experience recording from replay.')
        try:
            # init live_inventory variable with recorded inventory (we just use recorded inventory because
            # it has the correct layout for live_inventory
            consensus_stages = PartialExperienceSamplingWrapper.reset_consensus_stages()
            # get fist stage
            current_goal_stage = consensus_stages.pop()
            # used for video generation
            video_frames = []
            video_rewards = []
            video_stages = []
            cum_trans_reward = 0
            consensus_idx = 0
            reached_replay_goal = False
            consensus_checkpoints = {k.split('.')[-1]: v for k, v in self.experience_recording.checkpoints.items()}
            while consensus_idx < len(self.experience_recording.transitions_info):
                trans_info = self.experience_recording.transitions_info[consensus_idx]
                cum_trans_reward += trans_info['reward']

                frame = trans_info['info']['meta_controller']['state']['pov'] if 'meta_controller' in trans_info['info'] else None
                video_frames.append(frame)
                video_rewards.append(cum_trans_reward)
                video_stages.append(current_goal_stage.stage)

                if 'meta_controller' in trans_info['info']:
                    live_inventory = trans_info['info']['meta_controller']['state']['inventory']
                    reached_consensus_stage = PartialExperienceSamplingWrapper \
                        .is_inventory_sufficient_for_current_consensus_stage(live_inventory, current_goal_stage)
                else:
                    reached_consensus_stage = False

                if current_goal_stage.stage == 'SSSSSSSPPPPPLVNLAAAAYLQQQFKKKED':
                    reached_consensus_stage = trans_info['reward'] == 1024

                if reached_consensus_stage:
                    logging.debug("Reached consensus stage {}".format(current_goal_stage.stage))
                    consensus_worked_up_to = current_goal_stage.stage
                    # check if the current_goal_stage was the overall goal stage
                    if consensus == current_goal_stage.stage:
                        reached_replay_goal = True
                    else:
                        # HAX to jump to right consensus index
                        if current_goal_stage.stage in consensus_checkpoints:
                            consensus_idx = consensus_checkpoints[current_goal_stage.stage]

                        # pop new consensus stage
                        current_goal_stage = consensus_stages.pop()

                if reached_replay_goal or trans_info['done']:
                    break

                consensus_idx += 1

            if reached_replay_goal:
                Experience.video('tmp/rec/video/success',
                                 experience_file, video_frames, video_rewards, video_stages, video_only=video_only)
            else:
                Experience.video('tmp/rec/video/failed',
                                 experience_file, video_frames, video_rewards, video_stages, video_only=video_only)

            raise OperationCompletedInterrupt('Video generation has successfully stopped!')
        except OperationCompletedInterrupt as ci:
            logging.info('Replay: Everything completed correctly and program will interrupt now!')
            raise ci
        except Exception as e:
            # If the replay did not work, update the replay statistics and reset env
            logging.error('Replay: {} Something went wrong when replaying this trajectory.'.format(e))
            logging.error(
                'Replaying the trajectory for experience file {} has not worked up to {}, only up to {}. '.format(
                    experience_file, consensus, consensus_worked_up_to))
            self.success_mapping.update(experience_replay_filename=experience_file, result=0)
            raise Exception(f"Exception {e}. This code should not be reached. Check the implementation!")

    def reset_record_env_video(self, env_seed, consensus, experience_file, checkpoint):
        # if all seems well with the recording, set seed and reset environment
        if env_seed:
            self.env.seed(env_seed)
        super().reset()

        consensus_worked_up_to = None
        # we try to playback the env on the environment. if it doesn't work, we update its statistics and move on
        # we cannot be certain, that the sampled experience is a good experience
        # we have to catch exceptions in the reset method because we throw an InventoryMatch Exception
        # if the inventories of live and recorded trajectories don't match
        logging.info('Replay: Trying experience replay on env.')
        try:
            replay_rewards = []
            reached_replay_goal = False
            # init live_inventory variable with recorded inventory (we just use recorded inventory because
            # it has the correct layout for live_inventory
            consensus_stages = PartialExperienceSamplingWrapper.reset_consensus_stages()
            # setup counter and retry helper
            g_counter = -1
            counter = -1
            # get fist stage
            current_goal_stage = consensus_stages.pop()
            # get max timings for the different stages
            with open('train/experience_replay/timings.json') as f:
                max_counter = json.load(f)
            # used for video generation
            video_frames = []
            video_rewards = []
            video_stages = []
            cum_reward = 0
            for i, action in enumerate(self.experience_recording.replay_action_generator(checkpoint=checkpoint)):
                counter += 1
                g_counter += 1
                if g_counter % self.log_interval == 0: logging.info(f"Replay: Reached counter step: {g_counter} current stage: {current_goal_stage.stage}")

                obs, reward, done, info = self.env.step(action)
                replay_rewards.append(reward)
                live_inventory = obs['inventory']

                # for video generation
                video_frames.append(obs['pov'])
                cum_reward += reward
                video_rewards.append(cum_reward)
                video_stages.append(current_goal_stage.stage)

                if current_goal_stage.stage == 'SSSSSSSPPPPPLVNLAAAAYLQQQFKKKED':
                    reached_consensus_stage = reward >= 1000
                else:
                    reached_consensus_stage = PartialExperienceSamplingWrapper \
                        .is_inventory_sufficient_for_current_consensus_stage(live_inventory, current_goal_stage)

                if reached_consensus_stage:
                    logging.debug("Reached consensus stage {}".format(current_goal_stage.stage))
                    consensus_worked_up_to = current_goal_stage.stage
                    # check if the current_goal_stage was the overall goal stage
                    if consensus == current_goal_stage.stage:
                        reached_replay_goal = True
                    else:
                        # pop new consensus stage
                        current_goal_stage = consensus_stages.pop()
                        counter = -1

                if reached_replay_goal:
                    logging.info('Replay: Successfully replayed the trajectory up to {} in {} steps!'.format(consensus, g_counter))
                    logging.debug('Final inventory: {}'.format(live_inventory))
                    # update success_mapping
                    self.success_mapping.update(experience_replay_filename=experience_file, result=1)
                    # write successful video
                    Experience.video('tmp/rec/video/success',
                                     experience_file, video_frames, video_rewards, video_stages, video_only=False)
                    return obs

                if counter >= max_counter[current_goal_stage.stage]:
                    logging.warning('Replay: Max_counter limit exceeded in replay while trying stage {}!'.format(current_goal_stage.stage))
                    self.success_mapping.update(experience_replay_filename=experience_file, result=0)
                    # break to get into new iteration of while loop
                    break
                if done:
                    logging.warning('Replay: Environment has finished before completing stage {}!'.format(current_goal_stage.stage))
                    self.success_mapping.update(experience_replay_filename=experience_file, result=0)
                    # break to get into new iteration of while loop
                    break

            # write failed video
            Experience.video('tmp/rec/video/failed',
                             experience_file, video_frames, video_rewards, video_stages, video_only=False)
            logging.warning("Replay has not worked. Inventory: {}. Current stage: {} Trying new experience replay.".format(
                live_inventory, current_goal_stage.stage
            ))

        except Exception as e:
            # If the replay did not work, update the replay statistics and reset env
            logging.error('Replay: {} Something went wrong when replaying this trajectory.'.format(e))
            logging.error(
                'Replaying the trajectory for experience file {} has not worked up to {}, only up to {}. '.format(
                    experience_file, consensus, consensus_worked_up_to))
            self.success_mapping.update(experience_replay_filename=experience_file, result=0)
            raise Exception(f"Exception {e}. This code should not be reached. Check the implementation!")

    def reset_replay(self, env_seed, consensus, experience_file, checkpoint):
        # if all seems well with the recording, set seed and reset environment
        if env_seed:
            self.env.seed(env_seed)
        super().reset()

        consensus_worked_up_to = None
        # we try to playback the env on the environment. if it doesn't work, we update its statistics and move on
        # we cannot be certain, that the sampled experience is a good experience
        # we have to catch exceptions in the reset method because we throw an InventoryMatch Exception
        # if the inventories of live and recorded trajectories don't match
        logging.info('Replay: Trying experience replay on env.')
        try:
            replay_rewards = []
            reached_replay_goal = False
            # init live_inventory variable with recorded inventory (we just use recorded inventory because
            # it has the correct layout for live_inventory
            consensus_stages = PartialExperienceSamplingWrapper.reset_consensus_stages()
            # setup counter and retry helper
            g_counter = -1
            counter = -1
            # get fist stage
            current_goal_stage = consensus_stages.pop()
            # get max timings for the different stages
            with open('train/experience_replay/timings.json') as f:
                max_counter = json.load(f)

            for i, action in enumerate(self.experience_recording.replay_action_generator(checkpoint=checkpoint)):
                counter += 1
                g_counter += 1
                if g_counter % self.log_interval == 0: logging.info(f"Replay: Reached counter step: {g_counter} current stage: {current_goal_stage.stage}")

                obs, reward, done, info = self.env.step(action)
                replay_rewards.append(reward)
                live_inventory = obs['inventory']

                if current_goal_stage.stage == 'SSSSSSSPPPPPLVNLAAAAYLQQQFKKKED':
                    reached_consensus_stage = reward >= 1000
                else:
                    reached_consensus_stage = PartialExperienceSamplingWrapper \
                        .is_inventory_sufficient_for_current_consensus_stage(live_inventory, current_goal_stage)

                if reached_consensus_stage:
                    logging.debug("Reached consensus stage {}".format(current_goal_stage.stage))
                    consensus_worked_up_to = current_goal_stage.stage
                    # check if the current_goal_stage was the overall goal stage
                    if consensus == current_goal_stage.stage:
                        reached_replay_goal = True
                    else:
                        # pop new consensus stage
                        current_goal_stage = consensus_stages.pop()
                        counter = -1

                if reached_replay_goal:
                    logging.info('Replay: Successfully replayed the trajectory up to {} in {} steps!'.format(consensus, g_counter))
                    logging.debug('Final inventory: {}'.format(live_inventory))
                    # update success_mapping
                    self.success_mapping.update(experience_replay_filename=experience_file, result=1)
                    return obs

                if counter >= max_counter[current_goal_stage.stage]:
                    logging.warning('Replay: Max_counter limit exceeded in replay while trying stage {}!'.format(current_goal_stage.stage))
                    self.success_mapping.update(experience_replay_filename=experience_file, result=0)
                    # break to get into new iteration of while loop
                    break
                if done:
                    logging.warning('Replay: Environment has finished before completing stage {}!'.format(current_goal_stage.stage))
                    self.success_mapping.update(experience_replay_filename=experience_file, result=0)
                    # break to get into new iteration of while loop
                    break

            logging.warning("Replay has not worked. Inventory: {}. Current stage: {} Trying new experience replay.".format(
                live_inventory, current_goal_stage.stage
            ))

        except Exception as e:
            # If the replay did not work, update the replay statistics and reset env
            logging.error('Replay: {} Something went wrong when replaying this trajectory.'.format(e))
            logging.error(
                'Replaying the trajectory for experience file {} has not worked up to {}, only up to {}. '.format(
                    experience_file, consensus, consensus_worked_up_to))
            self.success_mapping.update(experience_replay_filename=experience_file, result=0)
            raise Exception(f"Exception {e}. This code should not be reached. Check the implementation!")

    def run_replay(self, env_seed, consensus, experience_file, checkpoint):
        if self.generate_experience_video:
            if self.debug_video == 'env':
                return self.reset_record_env_video(env_seed, consensus, experience_file, checkpoint)
            elif self.debug_video == 'replay':
                return self.reset_record_replay_video(env_seed, consensus, experience_file, video_only=False)
            elif self.debug_video == 'video_only_replay':
                return self.reset_record_replay_video(env_seed, consensus, experience_file, video_only=True)
            elif self.debug_video == 'side_by_side':
                return self.reset_record_side_by_side(env_seed, consensus, experience_file, checkpoint)
            else:
                raise Exception('Replay: Unknown debug video mode!')
        else:
            return self.reset_replay(env_seed, consensus, experience_file, checkpoint)

    def reset(self, **kwargs):
        # sample randomly an experience from the experience experiences_folder, based on the success weights
        while True:
            # check if we have enough uncorrupted experiences in the experiences_folder
            if len(self.experience_filenames) == 0:
                raise Exception("No experience files in this directory!")

            experience_file = self.success_mapping.sample()

            logging.info('Replay: Experience file used in next run: {}'.format(experience_file))
            try:
                self.experience_recording = Experience.load(os.path.join(self.experience_folder, experience_file))
                self.experience_recording = Experience.validate_and_fix(self.experience_recording)
            except FileNotFoundError:
                # if the file is not found for whatever reason, continue the loop anew and remove the name
                # of the file that was not found from self.experience_filenames
                logging.warning('File {} not found! Trying new file.'.format(experience_file))
                if experience_file in self.experience_filenames:
                    self.experience_filenames.remove(experience_file)
                continue
            except Exception as e:
                tr = traceback.extract_stack()
                logging.debug(tr)
                logging.warning('File {} is somehow corrupted and will be skipped. {}'.format(experience_file, e))
                if experience_file in self.experience_filenames:
                    self.experience_filenames.remove(experience_file)
                continue

            env_seed = self.experience_recording.meta_info
            checkpoint = next(reversed(self.experience_recording.checkpoints))

            # get recorded inventory up to checkpoint for a later comparison with the live agent
            recorded_inventory_all = [state['inventory'] for state in self.experience_recording.states[:self.experience_recording.checkpoints[checkpoint]]]

            # make sure that the recorded inventory has at least as many items as the consensus or self.replay_until
            # specifies
            if self.replay_until is None or self.replay_until == 'all':
                consensus = os.path.split(self.experience_folder)[-1]
            else:
                consensus = self.replay_until

            consensus_stages = PartialExperienceSamplingWrapper.reset_consensus_stages()

            # Now make sure that the currently used experience recording is not corrupted. Sometimes, the experience
            # recordings have wrong inventory item numbers. To be useful, the number of items in the inventory of
            # the live agent must match the consensus string (or the self.replay_until string for partial trajectory
            # replay). To test this, make sure that the last step of the recorded inventory has at least as many items
            # as specified by the consensus string.
            # THIS IS ONLY POSSIBLE IF self.replay_until IS NOT USED SINCE WE DONT KNOW FOR WHICH STEP TO COMPARE

            # remark: this is also valid in the probabilistic experience replay

            # get inventory_minimum for current consensus
            inventory_minimum = None
            for c_stage in consensus_stages:
                if c_stage.stage == consensus:
                    inventory_minimum = c_stage.inventory_minimum

            if inventory_minimum is None:
                raise ValueError('Wrong consensus provided - inventory check not implemented for ', consensus,
                                 'The experiences_folder must be named after the consensus!')

            if self.replay_until is None or self.replay_until == 'all':
                for key in inventory_minimum:
                    # don't compare dirt
                    if key == 'dirt':
                        continue
                    if recorded_inventory_all[-1][key] < inventory_minimum[key]:
                        # set corrupted state for this experience to true
                        logging.warning('The final inventory state for {} does not match the consensus, which means'
                                        'that something went wrong when creating the '
                                        'experience replay. Setting state to corrupted'.format(experience_file)),
                        self.experience_recording.corrupted = True
                        self.experience_recording.save(os.path.join(self.experience_folder, experience_file))

                # restart the loop if the file is corrupted
                if self.experience_recording.corrupted:
                    continue

            res = self.run_replay(env_seed, consensus, experience_file, checkpoint)
            if res is not None:
                return res

    def inventories_ok(self, live_inventory, recorded_inventory, counter, look_back, current_goal_stage):
        """
        Compares live and recorded inventories. It returns true if each entry of the live
        inventory is greater equal that the corresponding entry of the recorded inventory.
        One can choose with look_back how many steps in the past the recorded inventory will be checked.

        The comparison of the inventory can depend on the current consensus stage: when smelting iron ingots, it seems
        to be random if the agent takes a log or a plank to fuel the smelting process. This has to be taken into
        account when checking for inventory consistency.
        """
        bad_keys = ['dirt']
        if current_goal_stage is None:
            print('hoir')
        if current_goal_stage.stage in ['SSSSSSSPPPPPLVNLAAAAYLQQQFK', 'SSSSSSSPPPPPLVNLAAAAYLQQQFKK',
                                        'SSSSSSSPPPPPLVNLAAAAYLQQQFKKK']:
            bad_keys.append('log')
            bad_keys.append('planks')

        for key in live_inventory:
            if key in bad_keys:
                # don't compare items in the bad_keys list
                continue
            else:
                if live_inventory[key] >= recorded_inventory[counter - look_back][key]:
                    pass
                else:
                    return False

        return True