import logging

from train.envs.env_client import RemoteGym
from train.enums import ClientEnvOption
from train.statement.base_stmt import SubStatement, Statement


class PrepareEnvironmentStatement(SubStatement):

    def __init__(self, stmt: Statement, num_envs: int, option: ClientEnvOption = ClientEnvOption.Normal,
                 seed: int = None):
        super(PrepareEnvironmentStatement, self).__init__(stmt)
        self.num_envs = num_envs
        self.envs = None
        self.env_id = None
        self.running = False
        self.seed = seed
        self.option = option

    def finalize(self):
        if self.envs is not None:
            logging.info('Statement: Closing environments.')
            self.envs.close()
        self.envs = None
        self.env_id = None
        self.running = False

    def exec(self):
        # check if server / client environments are already running
        if self.running and self.env_id == self.config.env.env:
            logging.info('Statement: Servers already running! Reset...')
            # if environment running then simply reset
            self.envs.reset()
        # if not running, startup
        else:
            # ping if env servers are already started with the same environment version
            response = None
            try:
                # create ping instance to check server version and running status
                gym = RemoteGym(self.config.env.host, self.config.env.port)
                response = gym.version()
                logging.info('Statement: Ping server. Version: {}'.format(response))
            except:
                pass

            # if existing versions do not match the required env version or no servers are active then run new instances
            if response != self.config.env.env:
                # if existing instance existed, then close the previous environments
                if self.envs is not None:
                    logging.info('Statement: Server instances already running. Stopping...')
                    self.envs.close()
                    self.envs = None

                # startup new server environments
                logging.info('Statement: Starting {} environment servers.'.format(self.config.env.num_env))
                proc = self.ext_iface.create_env_pool(num_envs=self.config.env.num_env)
                proc.wait()
                assert proc.success()

            # create environment client connectors
            logging.info('Statement: Creating {} environment clients.'.format(self.num_envs))
            proc = self.ext_iface.make_env(num_envs=self.num_envs,
                                           option=self.option,
                                           seed_list=[self.seed])
            proc.wait()
            self.envs = proc.env
            assert proc.success() and self.envs is not None
            self.env_id = self.config.env.env
            self.running = True
            logging.info('Statement: {} environments up and running!'.format(self.num_envs))

        self._complete()
