from train.envs.env_client import RemoteGym
#import gym
import numpy as np
import random
import os
import matplotlib.pyplot as plt
import minerl


class DeterministicEnv(object):
    def __init__(self, seed, experience_recorder=None, folder_to_store_records=None):
        """ Initialize MineRLObtainDiamond-v0 and set seeds"""
        gym = RemoteGym()
        self.env = gym.make('MineRLObtainDiamond-v0')
        self.seed = seed
        self.experience_recorder = experience_recorder
        self.folder_to_store_records = folder_to_store_records
        if self.folder_to_store_records:
            self.init_record_folder(folder_to_store_records)
        # temporary variable needed to record the current observation
        self.current_obs = None
        self.record_counter = 0

    def init_record_folder(self, folder_to_store_records):
        self.folder_to_store_records = folder_to_store_records
        os.makedirs(self.folder_to_store_records, exist_ok=True)

    def reset(self, render=False, record=False):
        """ Reset gym env member """
        if self.seed:
            random.seed(self.seed)
            np.random.seed(self.seed)
            self.env.action_space.seed(self.seed)
            self.env.seed(self.seed)

        obs = self.env.reset()
        if self.experience_recorder:
            self.current_obs = obs
        self.execute_deterministic_actions(render=render, record=record)
        return self.env

    def close(self):
        self.env.close()

    def execute_deterministic_actions(self, render, record):
        raise NotImplementedError

    def move_randomly_for_n_frames(self, n_frames, render=False, record=False):
        """
        Agent takes random walk (forward, turn left, turn right, jump) through the environment
        """

        for i in range(n_frames):
            if self.experience_recorder:
                # record current observation
                self.experience_recorder.append_state(self.current_obs)

            action = self.env.action_space.noop()
            action['forward'] = np.random.choice([0, 1])
            action['left'] = np.random.choice([0, 1])
            action['right'] = np.random.choice([0, 1])
            action['jump'] = np.random.choice([0, 1])
            action['camera'] = [np.random.uniform(-20, 20), np.random.uniform(-90, 90)]

            next_obs, reward, done, info = self.env.step(action)

            if self.experience_recorder:
                self.current_obs = next_obs
                self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
            if record:
                self.record(next_obs)
            if render:
                self.env.render()

    def record(self, obs):
        """
        Save image of obs to disk
        """
        if self.folder_to_store_records:
            plt.imsave(os.path.join(self.folder_to_store_records, '{:05d}.png'.format(self.record_counter)),
                       obs['pov'])
            self.record_counter += 1
        else:
            raise ValueError(
                "You need to define a folder where to store the records!")

    def go_forward(self, render=False, record=False):
        """
        Agent takes one step forward.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['forward'] = 1
        next_obs, reward, done, info = self.env.step(action)

        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def go_forward_for_n_steps(self, n, render=False, record=False):
        """
        Agent takes n steps forward.
        """
        action = self.env.action_space.noop()
        action['forward'] = 1
        for i in range(n):
            if self.experience_recorder:
                # record current observation
                self.experience_recorder.append_state(self.current_obs)
            next_obs, reward, done, info = self.env.step(action)
            if self.experience_recorder:
                self.current_obs = next_obs
                self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
            if record:
                self.record(next_obs)
            if render:
                self.env.render()

    def go_backward(self, render=False, record=False):
        """
        Agent takes one step back.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['back'] = 1

        next_obs, reward, done, info = self.env.step(action)

        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def jump(self, render=False, record=False):
        """
        Agent jumps.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['jump'] = 1

        next_obs, reward, done, info = self.env.step(action)

        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def turn_left(self, render=False, record=False):
        """
        Agent turns left by 10 degrees.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['camera'] = [0, -10]

        next_obs, reward, done, info = self.env.step(action)

        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def turn_left_for_n_steps(self, n, render=False, record=False):
        """
        Agent turns left n times by 10 degrees.
        """
        action = self.env.action_space.noop()
        action['camera'] = [0, -10]

        for i in range(n):
            if self.experience_recorder:
                # record current observation
                self.experience_recorder.append_state(self.current_obs)
            next_obs, reward, done, info = self.env.step(action)
            if self.experience_recorder:
                self.current_obs = next_obs
                self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
            if record:
                self.record(next_obs)
            if render:
                self.env.render()

    def turn_right(self, render=False, record=False):
        """
        Agent turns right by 10 degrees.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['camera'] = [0, 10]

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def turn_right_for_n_steps(self, n, render=False, record=False):
        """
        Agent turns right n times by 10 degrees.
        """
        action = self.env.action_space.noop()
        action['camera'] = [0, 10]
        for i in range(n):
            if self.experience_recorder:
                # record current observation
                self.experience_recorder.append_state(self.current_obs)

            next_obs, reward, done, info = self.env.step(action)

            if self.experience_recorder:
                self.current_obs = next_obs
                self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
            if record:
                self.record(next_obs)
            if render:
                self.env.render()

    def strafe_left(self, render=False, record=False):
        """
        Agent strafes left.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['left'] = 1

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def strafe_right(self, render=False, record=False):
        """
        Agent strafes right.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['right'] = 1

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def look_down(self, render=False, record=False):
        """
        Agent looks down (camera is adjusted) by 10 degrees.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['camera'] = [10, 0]

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def look_down_for_n_frames(self, n, render=False, record=False):
        """
        Agent looks down (camera is adjusted) by 10 degrees n times.
        """
        action = self.env.action_space.noop()
        action['camera'] = [10, 0]
        for i in range(n):
            if self.experience_recorder:
                # record current observation
                self.experience_recorder.append_state(self.current_obs)
            next_obs, reward, done, info = self.env.step(action)
            if self.experience_recorder:
                self.current_obs = next_obs
                self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
            if record:
                self.record(next_obs)
            if render:
                self.env.render()

    def look_up(self, render=False, record=False):
        """
        Agent looks up (camera is adjusted) by 10 degrees.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['camera'] = [-10, 0]

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def look_up_for_n_frames(self, n, render=False, record=False):
        """
        Agent looks up (camera is adjusted) by 10 degrees n times.
        """
        action = self.env.action_space.noop()
        action['camera'] = [-10, 0]
        for i in range(n):
            if self.experience_recorder:
                # record current observation
                self.experience_recorder.append_state(self.current_obs)
            next_obs, reward, done, info = self.env.step(action)
            if self.experience_recorder:
                self.current_obs = next_obs
                self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
            if record:
                self.record(next_obs)
            if render:
                self.env.render()

    def attack(self, render=False, record=False):
        """
        Agent attacks once.
        """

        action = self.env.action_space.noop()
        action['attack'] = 1

        next_obs, reward, done, info = self.env.step(action)

        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
            self.current_obs = next_obs
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def attack_for_n_frames(self, n, render=False, record=False):
        """
        Agent attacks n times.
        """
        action = self.env.action_space.noop()
        action['attack'] = 1
        for i in range(n):
            if self.experience_recorder:
                # record current observation
                self.experience_recorder.append_state(self.current_obs)
            next_obs, reward, done, info = self.env.step(action)
            if self.experience_recorder:
                self.current_obs = next_obs
                self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
            if record:
                self.record(next_obs)
            if render:
                self.env.render()

    def attack_forward(self, render=False, record=False):
        """
        Agent attacks and moves forward at the same time.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)
        action = self.env.action_space.noop()

        action['attack'] = 1
        action['forward'] = 1

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def mine_vertical_arc(self, render=False, record=False):
        """
        Agent attacks in an "arc"-shaped way. Don't use, was good for exploration.
        :param render:
        :param record:
        :return:
        """
        self.attack_for_n_frames(30, render, record)
        self.wait_for_n_frames(2)
        for i in range(5):
            self.look_up(render, record)
        self.attack_for_n_frames(30, render, record)
        self.wait_for_n_frames(2)
        for i in range(10):
            self.look_down(render, record)
        self.attack_for_n_frames(30, render, record)
        self.wait_for_n_frames(2)
        for i in range(5):
            self.look_up(render, record)

    def craft_planks(self, render=False, record=False):
        """
        Agent crafts 4 planks from 1 log.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['craft'] = 'planks'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def craft_stick(self, render=False, record=False):
        """
        Agent crafts 4 sticks from 2 planks.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['craft'] = 'stick'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def craft_torch(self, render=False, record=False):
        """
        Agent crafts torch from 1 stick and 1 coal.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['craft'] = 'torch'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def place_torch(self, render=False, record=False):
        """
        Agent places torch on the ground.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['place'] = 'torch'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def craft_crafting_table(self, render=False, record=False):
        """
        Agent crafts a crafting table from 4 planks.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['craft'] = 'crafting_table'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def place_crafting_table(self, render=False, record=False):
        """
        Agent places crafting table on the ground.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['place'] = 'crafting_table'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def craft_wooden_pickaxe_at_nearby_crafting_table(self, render=False, record=False):
        """
        Agent crafts a wooden pickaxe from 3 planks and 2 sticks.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['nearbyCraft'] = 'wooden_pickaxe'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def craft_stone_pickaxe_at_nearby_crafting_table(self, render=False, record=False):
        """
        Agent crafts a wooden pickaxe from 3 cobblestones and 2 sticks.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['nearbyCraft'] = 'stone_pickaxe'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def craft_iron_pickaxe_at_nearby_crafting_table(self, render=False, record=False):
        """
        Agent crafts a wooden pickaxe from 3 iron ingots and 2 sticks at a nearby crafting
        table.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['nearbyCraft'] = 'iron_pickaxe'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def craft_furnace_at_nearby_crafting_table(self, render=False, record=False):
        """
        Agent crafts a furnace from 8 cobblestones at a nearby crafting table.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['nearbyCraft'] = 'furnace'
        next_obs, reward, done, info = self.env.step(action)

        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def place_furnace(self, render=False, record=False):
        """
        Agent places furnace on ground.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['place'] = 'furnace'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def smelt_coal_at_nearby_furnace(self, render=False, record=False):
        """
        According to the authors of minerl, the agent is supposed to be able to
        create a torch from smelting coal.
        Doesn't work in minerl 0.2.4 though, so don't use it until it's fixed.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['nearbySmelt'] = 'coal'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def smelt_iron_at_nearby_furnace(self, render=False, record=False):
        """
        Agent smelts 1 iron ingots from 1 iron ore and 1 coal at a nearby furnace.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['nearbySmelt'] = 'iron_ingot'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def empty_hand(self, render=False, record=False):
        """
        Agent unequips current item.
        :param render:
        :param record:
        :return:
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['equip'] = 'air'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def equip_wooden_pickaxe(self, render=False, record=False):
        """
        Agent equips wooden pickaxe.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['equip'] = 'wooden_pickaxe'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def equip_stone_pickaxe(self, render=False, record=False):
        """
        Agent equips stone pickaxe.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['equip'] = 'stone_pickaxe'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def equip_iron_pickaxe(self, render=False, record=False):
        """
        Agent equips stone pickaxe.
        """
        if self.experience_recorder:
            # record current observation
            self.experience_recorder.append_state(self.current_obs)

        action = self.env.action_space.noop()
        action['equip'] = 'iron_pickaxe'

        next_obs, reward, done, info = self.env.step(action)
        if self.experience_recorder:
            self.current_obs = next_obs
            self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
        if record:
            self.record(next_obs)
        if render:
            self.env.render()

    def get_inventory(self, render=False, record=False):
        """
        Returns inventory state at next frame.
        """
        next_obs, reward, done, info = self.env.step(self.env.action_space.noop())
        if record:
            self.record(next_obs)
        if render:
            self.env.render()
        return next_obs['inventory']

    def get_obs(self, render=False, record=False):
        """
        Returns observation at next frame.
        """
        next_obs, reward, done, info = self.env.step(self.env.action_space.noop())
        if record:
            self.record(next_obs)
        if render:
            self.env.render()
        return next_obs

    def wait_for_n_frames(self, n, render=False, record=False):
        """
        Agent waits for n frames.
        """
        for i in range(n):
            if self.experience_recorder:
                # record current observation
                self.experience_recorder.append_state(self.current_obs)
            action = self.env.action_space.noop()
            next_obs, reward, done, info = self.env.step(action)
            if self.experience_recorder:
                self.current_obs = next_obs
                self.experience_recorder.append_action_and_info(action=action, reward=reward, done=done, info=info)
            if record:
                self.record(next_obs)
            if render:
                self.env.render()

    def get_unstuck(self, render=False, record=False):
        """
        In some environment resets, the agent respawns IN the ground. It has to jump up to
        a certain frame in order to get out. At the moment, this is set to 4 but one might
        have to increase the value if one notices that the agent gets stuck after resetting.
        """
        for i in range(4):
            self.jump(render, record)

