import logging
from threading import RLock
from .common.utils import import_module


class PackageManager:
    """
    Singleton ModelSelector instance to handle model pre-training for general agents.
    """
    __instance = None
    __mutex = RLock()

    def __init__(self):
        """ Virtually private constructor. """
        if PackageManager.__instance is not None:
            raise Exception("This class is a singleton!")
        self.packages = {}
        self.current_version = None
        self.description = None
        self.model = None
        self.train_strategy = None
        self.dataset = None
        self.eval_hook = None
        self._use_package_manager = False
        logging.info('PackageManager: Created new selector.')

    @staticmethod
    def get_instance():
        """
        Returns singleton of the PackageManager class.
        :return:
        """
        PackageManager.__mutex.acquire()
        try:
            if PackageManager.__instance is None:
                PackageManager.__instance = PackageManager()
            return PackageManager.__instance
        finally:
            PackageManager.__mutex.release()

    def initialize(self, package_dict):
        """
        Initializes the package manager based on a version dictionary.
        Format JSON:
        {
            "default": "version_name",
            "version_name": {
                "description": "Description of the current version"
                "model": "model_package_path",
                "train_strategy": "train_strategy_path",
                "dataset": "dataset_path",
                "eval_hook": "eval_hook_path"
            },
            ...
        }
        :param package_dict:
        :return:
        """
        PackageManager.__mutex.acquire()
        try:
            self.packages = package_dict
            self.default()
            self._use_package_manager = True
            logging.info('PackageManager: Initialized and enabled package manager!')
        finally:
            PackageManager.__mutex.release()

    def enabled(self):
        PackageManager.__mutex.acquire()
        try:
            return self._use_package_manager
        finally:
            PackageManager.__mutex.release()

    def default(self):
        default_version = self.packages['default']
        self.switch(default_version)

    def switch(self, version):
        if version != self.current_version:
            self.description = self.packages[version]['description']
            self.model = import_module(self.packages[version]['model'])
            self.train_strategy = import_module(self.packages[version]['train_strategy'])
            self.dataset = import_module(self.packages[version]['dataset'])
            self.eval_hook = import_module(self.packages[version]['eval_hook'])
            self.current_version = version
