import logging
import random
from train.behavioral_cloning.datasets.experience import SeedParser
from train.envs.minerl_env import make_minerl
from train.proc.proc_base import Subprocess, repeated_run
from train.common.utils import import_module
from train.enums import ClientEnvOption
from train.stats import Stats
import numpy as np


class UnsupportedSeedOptionException(Exception):
    pass


class EnvClientProc(Subprocess):
    def __init__(self, config, num_envs,
                 option: ClientEnvOption = ClientEnvOption.Normal,
                 seed_list=None,
                 env_server=True):
        super(EnvClientProc, self).__init__('Environment Client', config)
        self.num_envs = num_envs
        self.gym = None
        self.env = None
        self.seed_list = None
        self.seed_mode = self.config.seed_mode
        if self.seed_mode == "programmatic":
            self.seed_list = seed_list
        elif self.seed_mode == "manual":
            self.seed_list = self.config.manual_seed_list
        elif self.seed_mode == "file":
            self.seed_list = SeedParser.load(self.config.file_seed_list)
        elif self.seed_mode == "none":
            self.seed_list = None
        elif self.seed_mode == "random":
            self.seed_list = [np.random.randint(1, 8000000)]
        elif self.seed_mode == "fixed":
            self.seed_list = [int(self.config.fixed_seed)]
        else:
            raise UnsupportedSeedOptionException()
        self.option = option
        self.replay_until = self.config.replay.replay_until
        self.checkpoint = self.config.replay.checkpoint
        self.env_server = env_server

    @repeated_run
    def run(self):
        selected_seed = int(random.choice(self.seed_list)) if self.seed_list is not None else None
        logging.info("Proc: Selected initial seed {}".format(selected_seed))
        Stats.get_instance().statistics['current']['seed'] = selected_seed
        dataset = import_module(self.config.rl.dataset)
        logging.info('Proc: Creating MineRL clients...')
        if self.option == ClientEnvOption.Record:
            seed = selected_seed
            record = True
            experience_folder = None
        elif self.option == ClientEnvOption.Replay:
            seed = None # not used with experience replay folder
            record = False
            experience_folder = self.config.replay.experience_path
        elif self.option == ClientEnvOption.All:
            seed = selected_seed
            record = True
            experience_folder = self.config.replay.experience_path
        else:
            seed = selected_seed
            record = False
            experience_folder = None
        self.env = make_minerl(self.config.env.env, n_cpu=self.num_envs, seq_len=dataset.SEQ_LENGTH,
                               transforms=dataset.DATA_TRANSFORM, input_space=dataset.INPUT_SPACE,
                               env_server=self.env_server, host=self.config.env.host, port=self.config.env.port,
                               seed=seed, make_new_recording=record, experience_folder=experience_folder,
                               replay_until=self.replay_until, exec_mode=self.config.mode, craft_equip=True,
                               checkpoint=self.checkpoint, frame_skip=dataset.FRAME_SKIP)
        self._complete()
