import logging
from train.iface import ExternalInterface
from train.statement.eval_stmt import EvaluateAgentStatement
from train.statement.learning_stmt import LearningStatement
from train.task import TaskType
from train.statement.base_stmt import Statement
from train.statement.imitation_stmt import ImitationStatement


def parse_eval_statements(config, ext_iface: ExternalInterface, tasks):
    """
    Creates executable statements from a list of tasks built from the alignment sequence.
    :param config: Configs from the meta controller
    :param ext_iface: External interface for accessing the processes.
    :param tasks: List of tasks to be parsed into executable statements.
    :return: Sequence of commands to execute the agent training.
    """
    assert len(tasks) > 0
    # the first tasks[0] access is only to define where the statement begins
    stmt = Statement(config, ext_iface, tasks[0], tasks)
    return [EvaluateAgentStatement(stmt)]


def parse_training_statements(config, ext_iface: ExternalInterface, tasks):
    """
    Creates executable statements from a list of tasks built from the alignment sequence.
    :param config: Configs from the meta controller
    :param ext_iface: External interface for accessing the processes.
    :param tasks: List of tasks to be parsed into executable statements.
    :return: Sequence of commands to execute the agent training.
    """
    assert len(tasks) > 0

    logging.info('Parsing consensus')
    statement_list = []
    task_history = []

    # parses a list of statements to be executed
    for i, task in enumerate(tasks):
        stmt = Statement(config, ext_iface, task, task_history)
        print('Scheduler: Creating statement for {} task...'.format(task.id))
        if task.task_type == TaskType.Learning:
            # used for training the models
            statement_list.append(LearningStatement(stmt))
        elif task.task_type == TaskType.Imitation:
            statement_list.append(ImitationStatement(stmt))
        else:
            # raises not implemented error - should never occur
            statement_list.append(stmt)
        task_history.append(task)
    return statement_list
